'use strict';

var compile = require('@lingui/core/compile');

const isString = s => typeof s === "string";
const isFunction = f => typeof f === "function";

/** Memoized cache */
const numberFormats = new Map();
const dateFormats = new Map();
function date(locales) {
  let format = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  let memoize = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : true;
  return value => {
    if (isString(value)) value = new Date(value);
    if (memoize) {
      const key = cacheKey(locales, format);
      const cachedFormatter = dateFormats.get(key);
      if (cachedFormatter) {
        return cachedFormatter.format(value);
      }
      const formatter = new Intl.DateTimeFormat(locales, format);
      dateFormats.set(key, formatter);
      return formatter.format(value);
    }
    const formatter = new Intl.DateTimeFormat(locales, format);
    return formatter.format(value);
  };
}
function number(locales) {
  let format = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  let memoize = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : true;
  return value => {
    if (memoize) {
      const key = cacheKey(locales, format);
      const cachedFormatter = numberFormats.get(key);
      if (cachedFormatter) {
        return cachedFormatter.format(value);
      }
      const formatter = new Intl.NumberFormat(locales, format);
      numberFormats.set(key, formatter);
      return formatter.format(value);
    }
    const formatter = new Intl.NumberFormat(locales, format);
    return formatter.format(value);
  };
}

/** Memoize helpers */
function cacheKey(locales) {
  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  const localeKey = Array.isArray(locales) ? locales.sort().join("-") : locales;
  return `${localeKey}-${JSON.stringify(options)}`;
}

var formats = /*#__PURE__*/Object.freeze({
  __proto__: null,
  date: date,
  number: number
});

const UNICODE_REGEX = /\\u[a-fA-F0-9]{4}|\\x[a-fA-F0-9]{2}/g;
const defaultFormats = function (locale, locales) {
  let localeData = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {
    plurals: undefined
  };
  let formats = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
  locales = locales || locale;
  const {
    plurals
  } = localeData;
  const style = format => isString(format) ? formats[format] || {
    style: format
  } : format;
  const replaceOctothorpe = (value, message) => {
    return ctx => {
      const msg = isFunction(message) ? message(ctx) : message;
      const norm = Array.isArray(msg) ? msg : [msg];
      const numberFormat = Object.keys(formats).length ? style("number") : {};
      const valueStr = number(locales, numberFormat)(value);
      return norm.map(m => isString(m) ? m.replace("#", valueStr) : m);
    };
  };
  if (!plurals) {
    console.error(`Plurals for locale ${locale} aren't loaded. Use i18n.loadLocaleData method to load plurals for specific locale. Using other plural rule as a fallback.`);
  }
  return {
    plural: (value, _ref) => {
      let {
        offset = 0,
        ...rules
      } = _ref;
      const message = rules[value] || rules[plurals === null || plurals === void 0 ? void 0 : plurals(value - offset)] || rules.other;
      return replaceOctothorpe(value - offset, message);
    },
    selectordinal: (value, _ref2) => {
      let {
        offset = 0,
        ...rules
      } = _ref2;
      const message = rules[value] || rules[plurals === null || plurals === void 0 ? void 0 : plurals(value - offset, true)] || rules.other;
      return replaceOctothorpe(value - offset, message);
    },
    select: (value, rules) => rules[value] || rules.other,
    number: (value, format) => number(locales, style(format))(value),
    date: (value, format) => date(locales, style(format))(value),
    undefined: value => value
  };
};

// Params -> CTX
/**
 * Creates a context object, which formats ICU MessageFormat arguments based on
 * argument type.
 *
 * @param locale     - Locale of message
 * @param locales      - Locales to be used when formatting the numbers or dates
 * @param values       - Parameters for variable interpolation
 * @param localeData - Locale data (e.g: plurals)
 * @param formats - Custom format styles
 * @returns {function(string, string, any)}
 */
function context(locale, locales, values, formats, localeData) {
  const formatters = defaultFormats(locale, locales, localeData, formats);
  const ctx = (name, type, format) => {
    const value = values[name];
    const formatted = formatters[type](value, format);
    const message = isFunction(formatted) ? formatted(ctx) : formatted;
    return Array.isArray(message) ? message.join("") : message;
  };
  return ctx;
}
function interpolate(translation, locale, locales, localeData) {
  return function (values) {
    let formats = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const ctx = context(locale, locales, values, formats, localeData);
    const formatMessage = message => {
      if (!Array.isArray(message)) return message;
      return message.reduce((message, token) => {
        if (isString(token)) return message + token;
        const [name, type, format] = token;
        let interpolatedFormat = {};
        if (format != null && !isString(format)) {
          Object.keys(format).forEach(key => {
            interpolatedFormat[key] = formatMessage(format[key]);
          });
        } else {
          interpolatedFormat = format;
        }
        const value = ctx(name, type, interpolatedFormat);
        if (value == null) return message;
        return message + value;
      }, "");
    };
    const result = formatMessage(translation);
    if (isString(result) && UNICODE_REGEX.test(result)) return JSON.parse(`"${result.trim()}"`);
    if (isString(result)) return result.trim();
    return result;
  };
}

class EventEmitter {
  _events = {};
  on(event, listener) {
    if (!this._hasEvent(event)) this._events[event] = [];
    this._events[event].push(listener);
    return () => this.removeListener(event, listener);
  }
  removeListener(event, listener) {
    if (!this._hasEvent(event)) return;
    const index = this._events[event].indexOf(listener);
    if (~index) this._events[event].splice(index, 1);
  }
  emit(event) {
    for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      args[_key - 1] = arguments[_key];
    }
    if (!this._hasEvent(event)) return;
    this._events[event].map(listener => listener.apply(this, args));
  }
  _hasEvent(event) {
    return Array.isArray(this._events[event]);
  }
}

class I18n extends EventEmitter {
  constructor(params) {
    super();
    this._messages = {};
    this._localeData = {};
    if (params.missing != null) this._missing = params.missing;
    if (params.messages != null) this.load(params.messages);
    if (params.localeData != null) this.loadLocaleData(params.localeData);
    if (params.locale != null || params.locales != null) {
      this.activate(params.locale, params.locales);
    }
  }
  get locale() {
    return this._locale;
  }
  get locales() {
    return this._locales;
  }
  get messages() {
    return this._messages[this._locale] ?? {};
  }
  get localeData() {
    return this._localeData[this._locale] ?? {};
  }
  _loadLocaleData(locale, localeData) {
    if (this._localeData[locale] == null) {
      this._localeData[locale] = localeData;
    } else {
      Object.assign(this._localeData[locale], localeData);
    }
  }
  loadLocaleData(localeOrAllData, localeData) {
    if (localeData != null) {
      // loadLocaleData('en', enLocaleData)
      // Loading locale data for a single locale.
      this._loadLocaleData(localeOrAllData, localeData);
    } else {
      // loadLocaleData(allLocaleData)
      // Loading all locale data at once.
      Object.keys(localeOrAllData).forEach(locale => this._loadLocaleData(locale, localeOrAllData[locale]));
    }
    this.emit("change");
  }
  _load(locale, messages) {
    if (this._messages[locale] == null) {
      this._messages[locale] = messages;
    } else {
      Object.assign(this._messages[locale], messages);
    }
  }
  load(localeOrMessages, messages) {
    if (messages != null) {
      // load('en', catalog)
      // Loading a catalog for a single locale.
      this._load(localeOrMessages, messages);
    } else {
      // load(catalogs)
      // Loading several locales at once.
      Object.keys(localeOrMessages).forEach(locale => this._load(locale, localeOrMessages[locale]));
    }
    this.emit("change");
  }
  activate(locale, locales) {
    if (process.env.NODE_ENV !== "production") {
      if (!this._messages[locale]) {
        console.warn(`Messages for locale "${locale}" not loaded.`);
      }
      if (!this._localeData[locale]) {
        console.warn(`Locale data for locale "${locale}" not loaded. Plurals won't work correctly.`);
      }
    }
    this._locale = locale;
    this._locales = locales;
    this.emit("change");
  }

  // method for translation and formatting
  _(id) {
    let values = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    let {
      message,
      formats,
      context
    } = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
    if (!isString(id)) {
      values = id.values || values;
      message = id.message;
      context = id.context;
      id = id.id;
    }
    const messageMissing = !context && !this.messages[id];
    const contextualMessageMissing = context && !this.messages[context][id];
    const messageUnreachable = contextualMessageMissing || messageMissing;

    // replace missing messages with custom message for debugging
    const missing = this._missing;
    if (missing && messageUnreachable) {
      return isFunction(missing) ? missing(this._locale, id, context) : missing;
    }
    if (messageUnreachable) {
      this.emit("missing", {
        id,
        context,
        locale: this._locale
      });
    }
    let translation;
    if (context && !contextualMessageMissing) {
      // context is like a subdirectory of other keys
      translation = this.messages[context][id] || message || id;
    } else {
      translation = this.messages[id] || message || id;
    }
    if (process.env.NODE_ENV !== "production") {
      translation = isString(translation) ? compile.compileMessage(translation) : translation;
    }

    // hack for parsing unicode values inside a string to get parsed in react native environments
    if (isString(translation) && UNICODE_REGEX.test(translation)) return JSON.parse(`"${translation}"`);
    if (isString(translation)) return translation;
    return interpolate(translation, this._locale, this._locales, this.localeData)(values, formats);
  }
  date(value, format) {
    return date(this._locales || this._locale, format)(value);
  }
  number(value, format) {
    return number(this._locales || this._locale, format)(value);
  }
}
function setupI18n() {
  let params = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
  return new I18n(params);
}

const i18n = setupI18n();

exports.I18n = I18n;
exports.formats = formats;
exports.i18n = i18n;
exports.setupI18n = setupI18n;
//# sourceMappingURL=index.js.map
