'use strict';

var _extends = require('@babel/runtime/helpers/extends');
var React = require('react');

const LinguiContext = /*#__PURE__*/React.createContext(null);
function useLingui() {
  const context = React.useContext(LinguiContext);
  if (process.env.NODE_ENV !== "production") {
    if (context == null) {
      throw new Error("useLingui hook was used without I18nProvider.");
    }
  }
  return context;
}
function withI18n(o) {
  return WrappedComponent => {
    return props => {
      if (process.env.NODE_ENV !== "production") {
        if (typeof o === "function" || /*#__PURE__*/React.isValidElement(o)) {
          throw new Error("withI18n([options]) takes options as a first argument, " + "but received React component itself. Without options, the Component " + "should be wrapped as withI18n()(Component), not withI18n(Component).");
        }
      }
      const {
        i18n
      } = useLingui();
      return /*#__PURE__*/React.createElement(WrappedComponent, _extends({}, props, {
        i18n: i18n
      }));
    };
  };
}
const I18nProvider = _ref => {
  let {
    i18n,
    defaultComponent,
    forceRenderOnLocaleChange = true,
    children
  } = _ref;
  /**
   * We can't pass `i18n` object directly through context, because even when locale
   * or messages are changed, i18n object is still the same. Context provider compares
   * reference identity and suggested workaround is create a wrapper object every time
   * we need to trigger re-render. See https://reactjs.org/docs/context.html#caveats.
   *
   * Due to this effect we also pass `defaultComponent` in the same context, instead
   * of creating a separate Provider/Consumer pair.
   *
   * We can't use useMemo hook either, because we want to recalculate value manually.
   */
  const makeContext = () => ({
    i18n,
    defaultComponent
  });
  const getRenderKey = () => {
    return forceRenderOnLocaleChange ? i18n.locale || "default" : "default";
  };
  const [context, setContext] = React.useState(makeContext()),
    [renderKey, setRenderKey] = React.useState(getRenderKey());

  /**
   * Subscribe for locale/message changes
   *
   * I18n object from `@lingui/core` is the single source of truth for all i18n related
   * data (active locale, catalogs). When new messages are loaded or locale is changed
   * we need to trigger re-rendering of LinguiContext.Consumers.
   *
   * We call `setContext(makeContext())` after adding the observer in case the `change`
   * event would already have fired between the inital renderKey calculation and the
   * `useEffect` hook being called. This can happen if locales are loaded/activated
   * async.
   */
  React.useEffect(() => {
    const unsubscribe = i18n.on("change", () => {
      setContext(makeContext());
      setRenderKey(getRenderKey());
    });
    if (renderKey === "default") {
      setRenderKey(getRenderKey());
    }
    if (forceRenderOnLocaleChange && renderKey === "default") {
      console.log("I18nProvider did not render. A call to i18n.activate still needs to happen or forceRenderOnLocaleChange must be set to false.");
    }
    return () => unsubscribe();
  }, []);
  if (forceRenderOnLocaleChange && renderKey === "default") return null;
  return /*#__PURE__*/React.createElement(LinguiContext.Provider, {
    value: context,
    key: renderKey
  }, children);
};

// match <tag>paired</tag> and <tag/> unpaired tags
const tagRe = /<([a-zA-Z0-9]+)>(.*?)<\/\1>|<([a-zA-Z0-9]+)\/>/;
const nlRe = /(?:\r\n|\r|\n)/g;

// For HTML, certain tags should omit their close tag. We keep a whitelist for
// those special-case tags.
const voidElementTags = {
  area: true,
  base: true,
  br: true,
  col: true,
  embed: true,
  hr: true,
  img: true,
  input: true,
  keygen: true,
  link: true,
  meta: true,
  param: true,
  source: true,
  track: true,
  wbr: true,
  menuitem: true
};

/**
 * `formatElements` - parse string and return tree of react elements
 *
 * `value` is string to be formatted with <tag>Paired<tag/> or <tag/> (unpaired)
 * placeholders. `elements` is a array of react elements which indexes
 * correspond to element indexes in formatted string
 */
function formatElements(value) {
  let elements = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  const uniqueId = makeCounter(0, "$lingui$");
  const parts = value.replace(nlRe, "").split(tagRe);

  // no inline elements, return
  if (parts.length === 1) return value;
  const tree = [];
  const before = parts.shift();
  if (before) tree.push(before);
  for (const [index, children, after] of getElements(parts)) {
    let element = elements[index];
    if (!element || voidElementTags[element.type] && children) {
      if (!element) {
        console.error(`Can use element at index '${index}' as it is not declared in the original translation`);
      } else {
        console.error(`${element.type} is a void element tag therefore it must have no children`);
      }

      // ignore problematic element but push its children and elements after it
      element = /*#__PURE__*/React.createElement(React.Fragment);
    }
    tree.push( /*#__PURE__*/React.cloneElement(element, {
      key: uniqueId()
    },
    // format children for pair tags
    // unpaired tags might have children if it's a component passed as a variable
    children ? formatElements(children, elements) : element.props.children));
    if (after) tree.push(after);
  }
  return tree;
}

/*
 * `getElements` - return array of element indexes and element childrens
 *
 * `parts` is array of [pairedIndex, children, unpairedIndex, textAfter, ...]
 * where:
 * - `pairedIndex` is index of paired element (undef for unpaired)
 * - `children` are children of paired element (undef for unpaired)
 * - `unpairedIndex` is index of unpaired element (undef for paired)
 * - `textAfter` is string after all elements (empty string, if there's nothing)
 *
 * `parts` length is always multiply of 4
 *
 * Returns: Array<[elementIndex, children, after]>
 */
function getElements(parts) {
  if (!parts.length) return [];
  const [paired, children, unpaired, after] = parts.slice(0, 4);
  return [[paired || unpaired, children || "", after]].concat(getElements(parts.slice(4, parts.length)));
}
const makeCounter = function () {
  let count = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0;
  let prefix = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : "";
  return () => `${prefix}_${count++}`;
};

function Trans(props) {
  const {
    i18n,
    defaultComponent
  } = useLingui();
  const {
    render,
    component,
    id,
    message,
    formats
  } = props;
  const values = {
    ...props.values
  };
  const components = {
    ...props.components
  };
  if (values) {
    /*
      Related discussion: https://github.com/lingui/js-lingui/issues/183
       Values *might* contain React elements with static content.
      They're replaced with <INDEX /> placeholders and added to `components`.
       Example:
      Translation: Hello {name}
      Values: { name: <strong>Jane</strong> }
       It'll become "Hello <0 />" with components=[<strong>Jane</strong>]
      */

    Object.keys(values).forEach(key => {
      const value = values[key];
      if (! /*#__PURE__*/React.isValidElement(value)) return;
      const index = Object.keys(components).length;
      components[index] = value;
      values[key] = `<${index}/>`;
    });
  }
  const _translation = i18n && typeof i18n._ === "function" ? i18n._(id, values, {
    message,
    formats
  }) : id; // i18n provider isn't loaded at all

  const translation = _translation ? formatElements(_translation, components) : null;
  if (render === null || component === null) {
    // Although `string` is a valid react element, types only allow `Element`
    // Upstream issue: https://github.com/DefinitelyTyped/DefinitelyTyped/issues/20544
    return translation;
  }
  const FallbackComponent = defaultComponent || React.Fragment;
  const i18nProps = {
    id,
    message,
    translation,
    isTranslated: id !== translation && message !== translation
  };

  // Validation of `render` and `component` props
  if (render && component) {
    console.error("You can't use both `component` and `render` prop at the same time. `component` is ignored.");
  } else if (render && typeof render !== "function") {
    console.error(`Invalid value supplied to prop \`render\`. It must be a function, provided ${render}`);
  } else if (component && typeof component !== "function") {
    // Apparently, both function components and class components are functions
    // See https://stackoverflow.com/a/41658173/1535540
    console.error(`Invalid value supplied to prop \`component\`. It must be a React component, provided ${component}`);
    return /*#__PURE__*/React.createElement(FallbackComponent, i18nProps, translation);
  }

  // Rendering using a render prop
  if (typeof render === "function") {
    // Component: render={(props) => <a title={props.translation}>x</a>}
    return render(i18nProps);
  }

  // `component` prop has a higher precedence over `defaultComponent`
  const Component = component || FallbackComponent;
  const DefaultComponent = defaultComponent;
  return DefaultComponent && !component ? /*#__PURE__*/React.createElement(DefaultComponent, i18nProps, translation) : /*#__PURE__*/React.createElement(Component, null, translation);
}
Trans.defaultProps = {
  values: {},
  components: {}
};

exports.I18nProvider = I18nProvider;
exports.Trans = Trans;
exports.useLingui = useLingui;
exports.withI18n = withI18n;
//# sourceMappingURL=index.js.map
