"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSliderThumb = useSliderThumb;

var _utils = require("@react-aria/utils");

var _utils2 = require("./utils");

var _react = require("react");

var _focus = require("@react-aria/focus");

var _label = require("@react-aria/label");

var _useMove = require("./useMove");

var _utils3 = require("@react-native-aria/utils");

/**
 * Provides behavior and accessibility for a thumb of a slider component.
 *
 * @param opts Options for this Slider thumb.
 * @param state Slider state, created via `useSliderState`.
 */
function useSliderThumb(opts, state, isReversed) {
  var _opts$ariaLabelledby;

  let {
    index,
    isRequired,
    isDisabled,
    validationState,
    trackLayout,
    inputRef
  } = opts;
  let isVertical = opts.orientation === 'vertical';
  let direction = (0, _utils3.isRTL)() ? 'rtl' : undefined;
  let {
    addGlobalListener,
    removeGlobalListener
  } = (0, _utils.useGlobalListeners)();

  let labelId = _utils2.sliderIds.get(state);

  const {
    labelProps,
    fieldProps
  } = (0, _label.useLabel)({ ...opts,
    'id': (0, _utils2.getSliderThumbId)(state, index),
    'aria-labelledby': `${labelId} ${(_opts$ariaLabelledby = opts['aria-labelledby']) !== null && _opts$ariaLabelledby !== void 0 ? _opts$ariaLabelledby : ''}`.trim()
  });
  const value = state.values[index];
  const focusInput = (0, _react.useCallback)(() => {
    if (inputRef.current) {
      (0, _utils.focusWithoutScrolling)(inputRef.current);
    }
  }, [inputRef]);
  const isFocused = state.focusedThumb === index;
  (0, _react.useEffect)(() => {
    if (isFocused) {
      focusInput();
    }
  }, [isFocused, focusInput]);
  const stateRef = (0, _react.useRef)(null);
  stateRef.current = state;
  let reverseX = isReversed || direction === 'rtl';
  let currentPosition = (0, _react.useRef)(null);
  let {
    moveProps
  } = (0, _useMove.useMove)({
    onMoveStart() {
      currentPosition.current = null;
      state.setThumbDragging(index, true);
    },

    onMove({
      deltaX,
      deltaY,
      pointerType
    }) {
      let size = isVertical ? trackLayout.height : trackLayout.width;

      if (currentPosition.current == null) {
        currentPosition.current = stateRef.current.getThumbPercent(index) * size;
      }

      if (pointerType === 'keyboard') {
        // (invert left/right according to language direction) + (according to vertical)
        let delta = ((reverseX ? -deltaX : deltaX) + (reverseX ? deltaY : -deltaY)) * stateRef.current.step;
        currentPosition.current += delta * size;
        stateRef.current.setThumbValue(index, stateRef.current.getThumbValue(index) + delta);
      } else {
        let delta = isVertical ? deltaY : deltaX;

        if (reverseX) {
          if (!isVertical) {
            delta = -delta;
          }
        } else {
          if (isVertical) {
            delta = -delta;
          }
        }

        currentPosition.current += delta;
        stateRef.current.setThumbPercent(index, (0, _utils.clamp)(currentPosition.current / size, 0, 1));
      }
    },

    onMoveEnd() {
      state.setThumbDragging(index, false);
    }

  }); // Immediately register editability with the state

  state.setThumbEditable(index, !isDisabled);
  const {
    focusableProps
  } = (0, _focus.useFocusable)((0, _utils.mergeProps)(opts, {
    onFocus: () => state.setFocusedThumb(index),
    onBlur: () => state.setFocusedThumb(undefined)
  }), inputRef);
  let currentPointer = (0, _react.useRef)(undefined);

  let onDown = id => {
    focusInput();
    currentPointer.current = id;
    state.setThumbDragging(index, true);
    addGlobalListener(window, 'mouseup', onUp, false);
    addGlobalListener(window, 'touchend', onUp, false);
    addGlobalListener(window, 'pointerup', onUp, false);
  };

  let onUp = e => {
    var _e$pointerId, _e$changedTouches;

    let id = (_e$pointerId = e.pointerId) !== null && _e$pointerId !== void 0 ? _e$pointerId : (_e$changedTouches = e.changedTouches) === null || _e$changedTouches === void 0 ? void 0 : _e$changedTouches[0].identifier;

    if (id === currentPointer.current) {
      focusInput();
      state.setThumbDragging(index, false);
      removeGlobalListener(window, 'mouseup', onUp, false);
      removeGlobalListener(window, 'touchend', onUp, false);
      removeGlobalListener(window, 'pointerup', onUp, false);
    }
  }; // We install mouse handlers for the drag motion on the thumb div, but
  // not the key handler for moving the thumb with the slider.  Instead,
  // we focus the range input, and let the browser handle the keyboard
  // interactions; we then listen to input's onChange to update state.


  return {
    inputProps: (0, _utils.mergeProps)(focusableProps, fieldProps, {
      'type': 'range',
      'tabIndex': !isDisabled ? 0 : undefined,
      'min': state.getThumbMinValue(index),
      'max': state.getThumbMaxValue(index),
      'step': state.step,
      'value': value,
      'disabled': isDisabled,
      'aria-orientation': opts.orientation,
      'aria-valuetext': state.getThumbValueLabel(index),
      'aria-required': isRequired || undefined,
      'aria-invalid': validationState === 'invalid' || undefined,
      'aria-errormessage': opts['aria-errormessage'],
      'onChange': e => {
        state.setThumbValue(index, parseFloat(e.target.value));
      }
    }),
    thumbProps: !isDisabled ? (0, _utils.mergeProps)(moveProps, {
      onMouseDown: e => {
        if (e.button !== 0 || e.altKey || e.ctrlKey || e.metaKey) {
          return;
        }

        onDown();
      },
      onPointerDown: e => {
        if (e.button !== 0 || e.altKey || e.ctrlKey || e.metaKey) {
          return;
        }

        onDown(e.pointerId);
      },
      onTouchStart: e => {
        onDown(e.changedTouches[0].identifier);
      }
    }) : {},
    labelProps
  };
}
//# sourceMappingURL=useSliderThumb.web.js.map