"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSlider = void 0;

var _utils = require("@react-aria/utils");

var _utils2 = require("./utils");

var _react = require("react");

var _interactions = require("@react-aria/interactions");

var _label = require("@react-aria/label");

var _utils3 = require("@react-native-aria/utils");

var _useMove = require("./useMove");

function useSliderWeb(props, state, trackLayout, isReversed) {
  var _labelProps$id;

  let {
    labelProps,
    fieldProps
  } = (0, _label.useLabel)(props);
  let isVertical = props.orientation === 'vertical'; // Attach id of the label to the state so it can be accessed by useSliderThumb.

  _utils2.sliderIds.set(state, (_labelProps$id = labelProps.id) !== null && _labelProps$id !== void 0 ? _labelProps$id : fieldProps.id);

  const direction = (0, _utils3.isRTL)() ? 'rtl' : undefined;
  let {
    addGlobalListener,
    removeGlobalListener
  } = (0, _utils.useGlobalListeners)(); // When the user clicks or drags the track, we want the motion to set and drag the
  // closest thumb.  Hence we also need to install useMove() on the track element.
  // Here, we keep track of which index is the "closest" to the drag start point.
  // It is set onMouseDown/onTouchDown; see trackProps below.

  const realTimeTrackDraggingIndex = (0, _react.useRef)(null);
  const stateRef = (0, _react.useRef)(null);
  stateRef.current = state;
  const reverseX = isReversed || direction === 'rtl';
  const currentPosition = (0, _react.useRef)(null);
  const {
    moveProps
  } = (0, _useMove.useMove)({
    onMoveStart() {
      currentPosition.current = null;
    },

    onMove({
      deltaX,
      deltaY
    }) {
      let size = isVertical ? trackLayout.height : trackLayout.width;

      if (currentPosition.current == null) {
        currentPosition.current = stateRef.current.getThumbPercent(realTimeTrackDraggingIndex.current) * size;
      }

      let delta = isVertical ? deltaY : deltaX;

      if (reverseX) {
        if (!isVertical) {
          delta = -delta;
        }
      } else {
        if (isVertical) {
          delta = -delta;
        }
      }

      currentPosition.current += delta;

      if (realTimeTrackDraggingIndex.current != null) {
        const percent = (0, _utils.clamp)(currentPosition.current / size, 0, 1);
        stateRef.current.setThumbPercent(realTimeTrackDraggingIndex.current, percent);
      }
    },

    onMoveEnd() {
      if (realTimeTrackDraggingIndex.current != null) {
        stateRef.current.setThumbDragging(realTimeTrackDraggingIndex.current, false);
        realTimeTrackDraggingIndex.current = null;
      }
    }

  });
  let currentPointer = (0, _react.useRef)(undefined);

  let onDownTrack = (e, id, clientX, clientY) => {
    // We only trigger track-dragging if the user clicks on the track itself and nothing is currently being dragged.
    if (!props.isDisabled && state.values.every((_, i) => !state.isThumbDragging(i))) {
      let size = isVertical ? trackLayout.height : trackLayout.width; // Find the closest thumb

      const trackPosition = trackLayout[isVertical ? 'top' : 'left'];
      const clickPosition = isVertical ? clientY : clientX;
      const offset = clickPosition - trackPosition;
      let percent = offset / size;

      if (reverseX) {
        if (!isVertical) {
          percent = 1 - percent;
        }
      } else {
        if (isVertical) {
          percent = 1 - percent;
        }
      }

      let value = state.getPercentValue(percent); // to find the closet thumb we split the array based on the first thumb position to the "right/end" of the click.

      let closestThumb;
      let split = state.values.findIndex(v => value - v < 0);

      if (split === 0) {
        // If the index is zero then the closetThumb is the first one
        closestThumb = split;
      } else if (split === -1) {
        // If no index is found they've clicked past all the thumbs
        closestThumb = state.values.length - 1;
      } else {
        let lastLeft = state.values[split - 1];
        let firstRight = state.values[split]; // Pick the last left/start thumb, unless they are stacked on top of each other, then pick the right/end one

        if (Math.abs(lastLeft - value) < Math.abs(firstRight - value)) {
          closestThumb = split - 1;
        } else {
          closestThumb = split;
        }
      } // Confirm that the found closest thumb is editable, not disabled, and move it


      if (closestThumb >= 0 && state.isThumbEditable(closestThumb)) {
        // Don't unfocus anything
        e.preventDefault();
        realTimeTrackDraggingIndex.current = closestThumb;
        state.setFocusedThumb(closestThumb);
        currentPointer.current = id;
        state.setThumbDragging(realTimeTrackDraggingIndex.current, true);
        state.setThumbValue(closestThumb, value);
        addGlobalListener(window, 'mouseup', onUpTrack, false);
        addGlobalListener(window, 'touchend', onUpTrack, false);
        addGlobalListener(window, 'pointerup', onUpTrack, false);
      } else {
        realTimeTrackDraggingIndex.current = null;
      }
    }
  };

  let onUpTrack = e => {
    var _e$pointerId, _e$changedTouches;

    let id = (_e$pointerId = e.pointerId) !== null && _e$pointerId !== void 0 ? _e$pointerId : (_e$changedTouches = e.changedTouches) === null || _e$changedTouches === void 0 ? void 0 : _e$changedTouches[0].identifier;

    if (id === currentPointer.current) {
      if (realTimeTrackDraggingIndex.current != null) {
        state.setThumbDragging(realTimeTrackDraggingIndex.current, false);
        realTimeTrackDraggingIndex.current = null;
      }

      removeGlobalListener(window, 'mouseup', onUpTrack, false);
      removeGlobalListener(window, 'touchend', onUpTrack, false);
      removeGlobalListener(window, 'pointerup', onUpTrack, false);
    }
  };

  if (labelProps.htmlFor) {
    // Ideally the `for` attribute should point to the first thumb, but VoiceOver on iOS
    // causes this to override the `aria-labelledby` on the thumb. This causes the first
    // thumb to only be announced as the slider label rather than its individual name as well.
    // See https://bugs.webkit.org/show_bug.cgi?id=172464.
    delete labelProps.htmlFor;

    labelProps.onClick = () => {
      var _document$getElementB;

      // Safari does not focus <input type="range"> elements when clicking on an associated <label>,
      // so do it manually. In addition, make sure we show the focus ring.
      (_document$getElementB = document.getElementById((0, _utils2.getSliderThumbId)(state, 0))) === null || _document$getElementB === void 0 ? void 0 : _document$getElementB.focus();
      (0, _interactions.setInteractionModality)('keyboard');
    };
  }

  return {
    labelProps,
    // The root element of the Slider will have role="group" to group together
    // all the thumb inputs in the Slider.  The label of the Slider will
    // be used to label the group.
    groupProps: {
      role: 'group',
      ...fieldProps
    },
    trackProps: (0, _utils.mergeProps)({
      onMouseDown(e) {
        if (e.button !== 0 || e.altKey || e.ctrlKey || e.metaKey) {
          return;
        }

        onDownTrack(e, undefined, e.clientX, e.clientY);
      },

      onPointerDown(e) {
        if (e.pointerType === 'mouse' && (e.button !== 0 || e.altKey || e.ctrlKey || e.metaKey)) {
          return;
        }

        onDownTrack(e, e.pointerId, e.clientX, e.clientY);
      },

      onTouchStart(e) {
        onDownTrack(e, e.changedTouches[0].identifier, e.changedTouches[0].clientX, e.changedTouches[0].clientY);
      }

    }, moveProps),
    outputProps: {
      'htmlFor': state.values.map((_, index) => (0, _utils2.getSliderThumbId)(state, index)).join(' '),
      'aria-live': 'off'
    }
  };
}

const useSlider = (props, state, ref, isReversed) => {
  let {
    groupProps: webGroupProps,
    ...rest
  } = useSliderWeb(props, state, ref, isReversed);
  let groupProps = (0, _utils3.mapDomPropsToRN)(webGroupProps);
  let labelProps = (0, _utils3.mapDomPropsToRN)(rest.labelProps);
  return {
    groupProps,
    ...rest,
    labelProps
  };
};

exports.useSlider = useSlider;
//# sourceMappingURL=useSlider.web.js.map