import { useEffect } from 'react';
import { useFocusWithin } from '@react-aria/interactions';
const visibleOverlays = [];

/**
 * Provides the behavior for overlays such as dialogs, popovers, and menus.
 * Hides the overlay when the user interacts outside it, when the Escape key is pressed,
 * or optionally, on blur. Only the top-most overlay will close at once.
 */
export function useOverlay(props, ref) {
  let {
    onClose,
    shouldCloseOnBlur,
    isOpen,
    isKeyboardDismissDisabled = false
  } = props;

  // Add the overlay ref to the stack of visible overlays on mount, and remove on unmount.
  useEffect(() => {
    if (isOpen) {
      visibleOverlays.push(ref);
    }
    return () => {
      let index = visibleOverlays.indexOf(ref);
      if (index >= 0) {
        visibleOverlays.splice(index, 1);
      }
    };
  }, [isOpen, ref]);

  // Only hide the overlay when it is the topmost visible overlay in the stack.
  let onHide = () => {
    if (visibleOverlays[visibleOverlays.length - 1] === ref && onClose) {
      onClose();
    }
  };

  // Handle the escape key
  let onKeyDown = e => {
    if (e.key === 'Escape' && !isKeyboardDismissDisabled) {
      e.preventDefault();
      onHide();
    }
  };

  // Handle clicking outside the overlay to close it

  let {
    focusWithinProps
  } = useFocusWithin({
    isDisabled: !shouldCloseOnBlur,
    onBlurWithin: () => {
      onClose && onClose();
    }
  });
  return {
    overlayProps: {
      onKeyDown,
      ...focusWithinProps
    }
  };
}
//# sourceMappingURL=useOverlay.web.js.map