"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.catalogMigration = catalogMigration;
exports.defaultConfig = exports.configValidation = void 0;
exports.fallbackLanguageMigration = fallbackLanguageMigration;
exports.getConfig = getConfig;
exports.makeConfig = makeConfig;
exports.replaceRootDir = replaceRootDir;
var _path = _interopRequireDefault(require("path"));
var _fs = _interopRequireDefault(require("fs"));
var _chalk = _interopRequireDefault(require("chalk"));
var _cosmiconfig = require("cosmiconfig");
var _jestValidate = require("jest-validate");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
// Enforce posix path delimiters internally
const pathJoinPosix = (...values) => _path.default
// normalize double slashes
.join(...values)
// convert platform specific path.sep to posix
.split(_path.default.sep).join("/");
const defaultConfig = {
  catalogs: [{
    path: pathJoinPosix("<rootDir>", "locale", "{locale}", "messages"),
    include: ["<rootDir>"],
    exclude: ["*/node_modules/*"]
  }],
  catalogsMergePath: "",
  compileNamespace: "cjs",
  compilerBabelOptions: {
    minified: true,
    jsescOption: {
      minimal: true
    }
  },
  extractBabelOptions: {
    plugins: [],
    presets: []
  },
  fallbackLocales: {},
  format: "po",
  formatOptions: {
    origins: true,
    lineNumbers: true
  },
  locales: [],
  orderBy: "messageId",
  pseudoLocale: "",
  rootDir: ".",
  runtimeConfigModule: ["@lingui/core", "i18n"],
  sourceLocale: "",
  service: {
    name: "",
    apiKey: ""
  }
};
exports.defaultConfig = defaultConfig;
function configExists(path) {
  return path && _fs.default.existsSync(path);
}
function TypeScriptLoader() {
  let loaderInstance;
  return (filepath, content) => {
    if (!loaderInstance) {
      const {
        TypeScriptLoader
      } = require("cosmiconfig-typescript-loader");
      loaderInstance = TypeScriptLoader();
    }
    return loaderInstance(filepath, content);
  };
}
function getConfig({
  cwd,
  configPath,
  skipValidation = false
} = {}) {
  const defaultRootDir = cwd || process.cwd();
  const moduleName = "lingui";
  const configExplorer = (0, _cosmiconfig.cosmiconfigSync)(moduleName, {
    searchPlaces: ["package.json", `.${moduleName}rc`, `.${moduleName}rc.json`, `.${moduleName}rc.yaml`, `.${moduleName}rc.yml`, `.${moduleName}rc.ts`, `.${moduleName}rc.js`, `${moduleName}.config.ts`, `${moduleName}.config.js`],
    loaders: {
      ".ts": TypeScriptLoader()
    }
  });
  const result = configExists(configPath) ? configExplorer.load(configPath) : configExplorer.search(defaultRootDir);
  const userConfig = result ? result.config : {};
  return makeConfig({
    rootDir: result ? _path.default.dirname(result.filepath) : defaultRootDir,
    ...userConfig
  }, {
    skipValidation
  });
}
function makeConfig(userConfig, opts = {}) {
  const config = {
    ...defaultConfig,
    ...userConfig
  };
  if (!opts.skipValidation) {
    (0, _jestValidate.validate)(config, configValidation);
    return pipe(
    // List config migrations from oldest to newest
    fallbackLanguageMigration, catalogMigration,
    // Custom validation
    validateLocales,
    // `replaceRootDir` should always be the last
    config => replaceRootDir(config, config.rootDir))(config);
  } else {
    return replaceRootDir(config, config.rootDir);
  }
}
const exampleConfig = {
  ...defaultConfig,
  extractors: (0, _jestValidate.multipleValidOptions)([], ["babel"], [{
    match: fileName => false,
    extract: (filename, targetDir, options) => {}
  }]),
  runtimeConfigModule: (0, _jestValidate.multipleValidOptions)({
    i18n: ["@lingui/core", "i18n"],
    Trans: ["@lingui/react", "Trans"]
  }, ["@lingui/core", "i18n"]),
  fallbackLocales: (0, _jestValidate.multipleValidOptions)({}, {
    "en-US": "en"
  }, {
    "en-US": ["en"]
  }, {
    default: "en"
  }, false),
  extractBabelOptions: {
    extends: "babelconfig.js",
    rootMode: "rootmode",
    plugins: ["plugin"],
    presets: ["preset"],
    targets: (0, _jestValidate.multipleValidOptions)({}, "> 0.5%", ["> 0.5%", "not dead"], undefined),
    assumptions: (0, _jestValidate.multipleValidOptions)({}, undefined),
    browserslistConfigFile: (0, _jestValidate.multipleValidOptions)(true, undefined),
    browserslistEnv: (0, _jestValidate.multipleValidOptions)(".browserslistrc", undefined)
  }
};
const deprecatedConfig = {
  fallbackLocale: config => ` Option ${_chalk.default.bold("fallbackLocale")} was replaced by ${_chalk.default.bold("fallbackLocales")}

    You can find more information here: https://github.com/lingui/js-lingui/issues/791

    @lingui/cli now treats your current configuration as:
    {
      ${_chalk.default.bold('"fallbackLocales"')}: {
        default: ${_chalk.default.bold(`"${config.fallbackLocale}"`)}
      }
    }

    Please update your configuration.
    `,
  localeDir: config => ` Option ${_chalk.default.bold("localeDir")} is deprecated. Configure source paths using ${_chalk.default.bold("catalogs")} instead.

    @lingui/cli now treats your current configuration as:

    {
      ${_chalk.default.bold('"catalogs"')}: ${JSON.stringify(catalogMigration(config).catalogs, null, 2)}
    }

    Please update your configuration.
    `,
  srcPathDirs: config => ` Option ${_chalk.default.bold("srcPathDirs")} is deprecated. Configure source paths using ${_chalk.default.bold("catalogs")} instead.

    @lingui/cli now treats your current configuration as:

    {
      ${_chalk.default.bold('"catalogs"')}: ${JSON.stringify(catalogMigration(config).catalogs, null, 2)}
    }

    Please update your configuration.
    `,
  srcPathIgnorePatterns: config => ` Option ${_chalk.default.bold("srcPathIgnorePatterns")} is deprecated. Configure excluded source paths using ${_chalk.default.bold("catalogs")} instead.

    @lingui/cli now treats your current configuration as:

    {
      ${_chalk.default.bold('"catalogs"')}: ${JSON.stringify(catalogMigration(config).catalogs, null, 2)}
    }

    Please update your configuration.
    `
};
const configValidation = {
  exampleConfig,
  deprecatedConfig,
  comment: "Documentation: https://lingui.dev/ref/conf"
};
exports.configValidation = configValidation;
function validateLocales(config) {
  if (!Array.isArray(config.locales) || !config.locales.length) {
    console.error("No locales defined!\n");
    console.error(`Add ${_chalk.default.yellow("'locales'")} to your configuration. See ${_chalk.default.underline("https://lingui.dev/ref/conf#locales")}`);
  }
  return config;
}
function replaceRootDir(config, rootDir) {
  return function replaceDeep(value, rootDir) {
    const replace = s => s.replace("<rootDir>", rootDir);
    if (value == null) {
      return value;
    } else if (typeof value === "string") {
      return replace(value);
    } else if (Array.isArray(value)) {
      return value.map(item => replaceDeep(item, rootDir));
    } else if (typeof value === "object") {
      Object.keys(value).forEach(key => {
        const newKey = replaceDeep(key, rootDir);
        value[newKey] = replaceDeep(value[key], rootDir);
        if (key !== newKey) delete value[key];
      });
    }
    return value;
  }(config, rootDir);
}

/**
 * Replace fallbackLocale, by the new standard fallbackLocales
 * - https://github.com/lingui/js-lingui/issues/791
 * - Remove anytime after 4.x
 */

function fallbackLanguageMigration(config) {
  const {
    fallbackLocale,
    fallbackLocales
  } = config;
  if (fallbackLocales === false) return {
    ...config,
    fallbackLocales: null
  };
  const DEFAULT_FALLBACK = (fallbackLocales === null || fallbackLocales === void 0 ? void 0 : fallbackLocales.default) || fallbackLocale;
  if (DEFAULT_FALLBACK) {
    if (!config.fallbackLocales) config.fallbackLocales = {};
    config.fallbackLocales.default = DEFAULT_FALLBACK;
  }
  if (config.fallbackLocales !== false && !config.fallbackLocales.default) {
    config.locales.forEach(locale => {
      const fl = getCldrParentLocale(locale.toLowerCase());
      if (fl && !config.fallbackLocales[locale]) {
        config.fallbackLocales = {
          ...config.fallbackLocales,
          [locale]: fl
        };
      }
    });
  }
  return config;
}
function getCldrParentLocale(sourceLocale) {
  return {
    "en-ag": "en",
    "en-ai": "en",
    "en-au": "en",
    "en-bb": "en",
    "en-bm": "en",
    "en-bs": "en",
    "en-bw": "en",
    "en-bz": "en",
    "en-ca": "en",
    "en-cc": "en",
    "en-ck": "en",
    "en-cm": "en",
    "en-cx": "en",
    "en-cy": "en",
    "en-dg": "en",
    "en-dm": "en",
    "en-er": "en",
    "en-fj": "en",
    "en-fk": "en",
    "en-fm": "en",
    "en-gb": "en",
    "en-gd": "en",
    "en-gg": "en",
    "en-gh": "en",
    "en-gi": "en",
    "en-gm": "en",
    "en-gy": "en",
    "en-hk": "en",
    "en-ie": "en",
    "en-il": "en",
    "en-im": "en",
    "en-in": "en",
    "en-io": "en",
    "en-je": "en",
    "en-jm": "en",
    "en-ke": "en",
    "en-ki": "en",
    "en-kn": "en",
    "en-ky": "en",
    "en-lc": "en",
    "en-lr": "en",
    "en-ls": "en",
    "en-mg": "en",
    "en-mo": "en",
    "en-ms": "en",
    "en-mt": "en",
    "en-mu": "en",
    "en-mw": "en",
    "en-my": "en",
    "en-na": "en",
    "en-nf": "en",
    "en-ng": "en",
    "en-nr": "en",
    "en-nu": "en",
    "en-nz": "en",
    "en-pg": "en",
    "en-ph": "en",
    "en-pk": "en",
    "en-pn": "en",
    "en-pw": "en",
    "en-rw": "en",
    "en-sb": "en",
    "en-sc": "en",
    "en-sd": "en",
    "en-sg": "en",
    "en-sh": "en",
    "en-sl": "en",
    "en-ss": "en",
    "en-sx": "en",
    "en-sz": "en",
    "en-tc": "en",
    "en-tk": "en",
    "en-to": "en",
    "en-tt": "en",
    "en-tv": "en",
    "en-tz": "en",
    "en-ug": "en",
    "en-us": "en",
    "en-vc": "en",
    "en-vg": "en",
    "en-vu": "en",
    "en-ws": "en",
    "en-za": "en",
    "en-zm": "en",
    "en-zw": "en",
    "en-at": "en",
    "en-be": "en",
    "en-ch": "en",
    "en-de": "en",
    "en-dk": "en",
    "en-fi": "en",
    "en-nl": "en",
    "en-se": "en",
    "en-si": "en",
    "es-ar": "es",
    "es-bo": "es",
    "es-br": "es",
    "es-bz": "es",
    "es-cl": "es",
    "es-co": "es",
    "es-cr": "es",
    "es-cu": "es",
    "es-do": "es",
    "es-ec": "es",
    "es-es": "es",
    "es-gt": "es",
    "es-hn": "es",
    "es-mx": "es",
    "es-ni": "es",
    "es-pa": "es",
    "es-pe": "es",
    "es-pr": "es",
    "es-py": "es",
    "es-sv": "es",
    "es-us": "es",
    "es-uy": "es",
    "es-ve": "es",
    "pt-ao": "pt",
    "pt-ch": "pt",
    "pt-cv": "pt",
    "pt-fr": "pt",
    "pt-gq": "pt",
    "pt-gw": "pt",
    "pt-lu": "pt",
    "pt-mo": "pt",
    "pt-mz": "pt",
    "pt-pt": "pt",
    "pt-st": "pt",
    "pt-tl": "pt",
    "az-arab": "az",
    "az-cyrl": "az",
    "blt-latn": "blt",
    "bm-nkoo": "bm",
    "bs-cyrl": "bs",
    "byn-latn": "byn",
    "cu-glag": "cu",
    "dje-arab": "dje",
    "dyo-arab": "dyo",
    "en-dsrt": "en",
    "en-shaw": "en",
    "ff-adlm": "ff",
    "ff-arab": "ff",
    "ha-arab": "ha",
    "hi-latn": "hi",
    "iu-latn": "iu",
    "kk-arab": "kk",
    "ks-deva": "ks",
    "ku-arab": "ku",
    "ky-arab": "ky",
    "ky-latn": "ky",
    "ml-arab": "ml",
    "mn-mong": "mn",
    "mni-mtei": "mni",
    "ms-arab": "ms",
    "pa-arab": "pa",
    "sat-deva": "sat",
    "sd-deva": "sd",
    "sd-khoj": "sd",
    "sd-sind": "sd",
    "shi-latn": "shi",
    "so-arab": "so",
    "sr-latn": "sr",
    "sw-arab": "sw",
    "tg-arab": "tg",
    "ug-cyrl": "ug",
    "uz-arab": "uz",
    "uz-cyrl": "uz",
    "vai-latn": "vai",
    "wo-arab": "wo",
    "yo-arab": "yo",
    "yue-hans": "yue",
    "zh-hant": "zh",
    "zh-hant-hk": "zh",
    "zh-hant-mo": "zh-hant-hk"
  }[sourceLocale];
}

/**
 * Replace localeDir, srcPathDirs and srcPathIgnorePatterns with catalogs
 *
 * Released in @lingui/conf 3.0
 * Remove anytime after 4.x
 */

function catalogMigration(config) {
  let {
    localeDir,
    srcPathDirs,
    srcPathIgnorePatterns,
    ...newConfig
  } = config;
  if (localeDir || srcPathDirs || srcPathIgnorePatterns) {
    // Replace missing values with default ones
    if (localeDir === undefined) localeDir = pathJoinPosix("<rootDir>", "locale", "{locale}", "messages");
    if (srcPathDirs === undefined) srcPathDirs = ["<rootDir>"];
    if (srcPathIgnorePatterns === undefined) srcPathIgnorePatterns = ["**/node_modules/**"];
    let newLocaleDir = localeDir.split(_path.default.sep).join("/");
    if (newLocaleDir.slice(-1) !== _path.default.sep) {
      newLocaleDir += "/";
    }
    if (!Array.isArray(newConfig.catalogs)) {
      newConfig.catalogs = [];
    }
    newConfig.catalogs.push({
      path: pathJoinPosix(newLocaleDir, "{locale}", "messages"),
      include: srcPathDirs,
      exclude: srcPathIgnorePatterns
    });
  }
  return newConfig;
}
const pipe = (...functions) => args => functions.reduce((arg, fn) => fn(arg), args);