"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.command = command;
var _chalk = _interopRequireDefault(require("chalk"));
var _chokidar = _interopRequireDefault(require("chokidar"));
var _fs = _interopRequireDefault(require("fs"));
var R = _interopRequireWildcard(require("ramda"));
var _commander = _interopRequireDefault(require("commander"));
var plurals = _interopRequireWildcard(require("make-plural"));
var _conf = require("@lingui/conf");
var _catalog = require("./api/catalog");
var _compile = require("./api/compile");
var _help = require("./api/help");
var _api = require("./api");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const noMessages = R.pipe(R.map(R.isEmpty), R.values, R.all(R.equals(true)));
function command(config, options) {
  const catalogs = (0, _catalog.getCatalogs)(config);

  // fixme: this is definitely doesn't work
  if (noMessages(catalogs)) {
    console.error("Nothing to compile, message catalogs are empty!\n");
    console.error(`(use "${_chalk.default.yellow((0, _help.helpRun)("extract"))}" to extract messages from source files)`);
    return false;
  }

  // Check config.compile.merge if catalogs for current locale are to be merged into a single compiled file
  const doMerge = !!config.catalogsMergePath;
  let mergedCatalogs = {};
  console.log("Compiling message catalogs…");
  for (const locale of config.locales) {
    const [language] = locale.split(/[_-]/);
    // todo: this validation should be in @lingui/conf
    if (locale !== config.pseudoLocale && !plurals[language]) {
      console.error(_chalk.default.red(`Error: Invalid locale ${_chalk.default.bold(locale)} (missing plural rules)!`));
      console.error();
    }
    for (const catalog of catalogs) {
      const messages = catalog.getTranslations(locale, {
        fallbackLocales: config.fallbackLocales,
        sourceLocale: config.sourceLocale
      });
      if (!options.allowEmpty) {
        const missingMsgIds = R.pipe(R.pickBy(R.isNil), R.keys)(messages);
        if (missingMsgIds.length > 0) {
          console.error(_chalk.default.red(`Error: Failed to compile catalog for locale ${_chalk.default.bold(locale)}!`));
          if (options.verbose) {
            console.error(_chalk.default.red("Missing translations:"));
            missingMsgIds.forEach(msgId => console.error(msgId));
          } else {
            console.error(_chalk.default.red(`Missing ${missingMsgIds.length} translation(s)`));
          }
          console.error();
          return false;
        }
      }
      if (doMerge) {
        mergedCatalogs = {
          ...mergedCatalogs,
          ...messages
        };
      } else {
        const namespace = options.typescript ? "ts" : options.namespace || config.compileNamespace;
        const compiledCatalog = (0, _compile.createCompiledCatalog)(locale, messages, {
          strict: false,
          namespace,
          pseudoLocale: config.pseudoLocale,
          compilerBabelOptions: config.compilerBabelOptions
        });
        const compiledPath = catalog.writeCompiled(locale, compiledCatalog, namespace);
        if (options.typescript) {
          const typescriptPath = compiledPath.replace(/\.ts?$/, "") + ".d.ts";
          _fs.default.writeFileSync(typescriptPath, `import { Messages } from '@lingui/core';
          declare const messages: Messages;
          export { messages };
          `);
        }
        options.verbose && console.error(_chalk.default.green(`${locale} ⇒ ${compiledPath}`));
      }
    }
    if (doMerge) {
      const compileCatalog = (0, _catalog.getCatalogForMerge)(config);
      const namespace = options.namespace || config.compileNamespace;
      const compiledCatalog = (0, _compile.createCompiledCatalog)(locale, mergedCatalogs, {
        strict: false,
        namespace: namespace,
        pseudoLocale: config.pseudoLocale,
        compilerBabelOptions: config.compilerBabelOptions
      });
      const compiledPath = compileCatalog.writeCompiled(locale, compiledCatalog, namespace);
      options.verbose && console.log(_chalk.default.green(`${locale} ⇒ ${compiledPath}`));
    }
  }
  return true;
}
if (require.main === module) {
  _commander.default.description("Add compile message catalogs and add language data (plurals) to compiled bundle.").option("--config <path>", "Path to the config file").option("--strict", "Disable defaults for missing translations").option("--verbose", "Verbose output").option("--format <format>", "Format of message catalog").option("--typescript", "Create Typescript definition for compiled bundle").option("--namespace <namespace>", "Specify namespace for compiled bundle. Ex: cjs(default) -> module.exports, es -> export, window.test -> window.test").option("--watch", "Enables Watch Mode").option("--debounce <delay>", "Debounces compilation for given amount of milliseconds").on("--help", function () {
    console.log("\n  Examples:\n");
    console.log("    # Compile translations and use defaults or message IDs for missing translations");
    console.log(`    $ ${(0, _help.helpRun)("compile")}`);
    console.log("");
    console.log("    # Compile translations but fail when there are missing");
    console.log("    # translations (don't replace missing translations with");
    console.log("    # default messages or message IDs)");
    console.log(`    $ ${(0, _help.helpRun)("compile --strict")}`);
  }).parse(process.argv);
  const config = (0, _conf.getConfig)({
    configPath: _commander.default.config
  });
  if (_commander.default.format) {
    const msg = "--format option is deprecated and will be removed in @lingui/cli@3.0.0." + " Please set format in configuration https://lingui.dev/ref/conf#format";
    console.warn(msg);
    config.format = _commander.default.format;
  }
  const compile = () => command(config, {
    verbose: _commander.default.watch || _commander.default.verbose || false,
    allowEmpty: !_commander.default.strict,
    typescript: _commander.default.typescript || config.compileNamespace === "ts" || false,
    namespace: _commander.default.namespace // we want this to be undefined if user does not specify so default can be used
  });

  let debounceTimer;
  const dispatchCompile = () => {
    // Skip debouncing if not enabled
    if (!_commander.default.debounce) return compile();

    // CLear the previous timer if there is any, and schedule the next
    debounceTimer && clearTimeout(debounceTimer);
    debounceTimer = setTimeout(() => compile(), _commander.default.debounce);
  };

  // Check if Watch Mode is enabled
  if (_commander.default.watch) {
    console.info(_chalk.default.bold("Initializing Watch Mode..."));
    const catalogs = (0, _catalog.getCatalogs)(config);
    let paths = [];
    const catalogExtension = (0, _api.getFormat)(config.format).catalogExtension;
    config.locales.forEach(locale => {
      catalogs.forEach(catalog => {
        paths.push(`${catalog.path.replace(/{locale}/g, locale).replace(/{name}/g, "*")}${catalogExtension}`);
      });
    });
    const watcher = _chokidar.default.watch(paths, {
      persistent: true
    });
    const onReady = () => {
      console.info(_chalk.default.green.bold("Watcher is ready!"));
      watcher.on("add", () => dispatchCompile()).on("change", () => dispatchCompile());
    };
    watcher.on("ready", () => onReady());
  } else {
    const results = compile();
    if (!results) {
      process.exit(1);
    }
    console.log("Done!");
  }
}