"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cleanObsolete = exports.Catalog = void 0;
exports.getCatalogForFile = getCatalogForFile;
exports.getCatalogForMerge = getCatalogForMerge;
exports.getCatalogs = getCatalogs;
exports.normalizeRelativePath = normalizeRelativePath;
exports.order = order;
exports.orderByMessageId = orderByMessageId;
exports.orderByOrigin = orderByOrigin;
var _os = _interopRequireDefault(require("os"));
var _fsExtra = _interopRequireDefault(require("fs-extra"));
var _path = _interopRequireDefault(require("path"));
var R = _interopRequireWildcard(require("ramda"));
var _chalk = _interopRequireDefault(require("chalk"));
var _glob = _interopRequireDefault(require("glob"));
var _micromatch = _interopRequireDefault(require("micromatch"));
var _normalizePath = _interopRequireDefault(require("normalize-path"));
var _formats = _interopRequireDefault(require("./formats"));
var _extractors = _interopRequireDefault(require("./extractors"));
var _utils = require("./utils");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const NAME = "{name}";
const NAME_REPLACE_RE = /{name}/g;
const LOCALE = "{locale}";
const LOCALE_REPLACE_RE = /{locale}/g;
const LOCALE_SUFFIX_RE = /\{locale\}.*$/;
const PATHSEP = "/"; // force posix everywhere

class Catalog {
  constructor({
    name,
    path,
    include,
    exclude = []
  }, config) {
    this.name = name;
    this.path = normalizeRelativePath(path);
    this.include = include.map(normalizeRelativePath);
    this.exclude = [this.localeDir, ...exclude.map(normalizeRelativePath)];
    this.config = config;
    this.format = (0, _formats.default)(config.format);
  }
  async make(options) {
    const nextCatalog = await this.collect(options);
    if (!nextCatalog) return false;
    const prevCatalogs = this.readAll();
    const catalogs = this.merge(prevCatalogs, nextCatalog, {
      overwrite: options.overwrite,
      files: options.files
    });

    // Map over all locales and post-process each catalog
    const cleanAndSort = R.map(R.pipe(
    // Clean obsolete messages
    options.clean ? cleanObsolete : R.identity,
    // Sort messages
    order(options.orderBy)));
    const sortedCatalogs = cleanAndSort(catalogs);
    if (options.locale) {
      this.write(options.locale, sortedCatalogs[options.locale]);
    } else {
      this.writeAll(sortedCatalogs);
    }
    return true;
  }
  async makeTemplate(options) {
    const catalog = await this.collect(options);
    if (!catalog) return false;
    const sort = order(options.orderBy);
    this.writeTemplate(sort(catalog));
    return true;
  }

  /**
   * Collect messages from source paths. Return a raw message catalog as JSON.
   */
  async collect(options) {
    const tmpDir = _path.default.join(_os.default.tmpdir(), `lingui-${process.pid}`);
    if (_fsExtra.default.existsSync(tmpDir)) {
      (0, _utils.removeDirectory)(tmpDir, true);
    } else {
      _fsExtra.default.mkdirSync(tmpDir);
    }
    try {
      let paths = this.sourcePaths;
      if (options.files) {
        options.files = options.files.map(p => (0, _normalizePath.default)(p, false));
        const regex = new RegExp(options.files.join("|"), "i");
        paths = paths.filter(path => regex.test(path));
      }
      let catalogSuccess = true;
      for (let filename of paths) {
        const fileSuccess = await (0, _extractors.default)(filename, tmpDir, {
          verbose: options.verbose,
          configPath: options.configPath,
          babelOptions: this.config.extractBabelOptions,
          extractors: options.extractors,
          projectType: options.projectType
        });
        catalogSuccess && (catalogSuccess = fileSuccess);
      }
      if (!catalogSuccess) return undefined;
      return function traverse(directory) {
        return _fsExtra.default.readdirSync(directory).map(filename => {
          const filepath = _path.default.join(directory, filename);
          if (_fsExtra.default.lstatSync(filepath).isDirectory()) {
            return traverse(filepath);
          }
          if (!filename.endsWith(".json")) return;
          try {
            return JSON.parse(_fsExtra.default.readFileSync(filepath).toString());
          } catch (e) {}
        }).filter(Boolean).reduce((catalog, messages) => R.mergeWithKey(mergeOriginsAndExtractedComments, catalog, messages), {});
      }(tmpDir);
    } catch (e) {
      throw e;
    } finally {
      (0, _utils.removeDirectory)(tmpDir);
    }
  }
  merge(prevCatalogs, nextCatalog, options) {
    const nextKeys = R.keys(nextCatalog).map(String);
    return R.mapObjIndexed((prevCatalog, locale) => {
      const prevKeys = R.keys(prevCatalog).map(String);
      const newKeys = R.difference(nextKeys, prevKeys);
      const mergeKeys = R.intersection(nextKeys, prevKeys);
      const obsoleteKeys = R.difference(prevKeys, nextKeys);

      // Initialize new catalog with new keys
      const newMessages = R.mapObjIndexed((message, key) => ({
        translation: this.config.sourceLocale === locale ? message.message || key : "",
        ...message
      }), R.pick(newKeys, nextCatalog));

      // Merge translations from previous catalog
      const mergedMessages = mergeKeys.map(key => {
        const updateFromDefaults = this.config.sourceLocale === locale && (prevCatalog[key].translation === prevCatalog[key].message || options.overwrite);
        const translation = updateFromDefaults ? nextCatalog[key].message || key : prevCatalog[key].translation;
        return {
          [key]: {
            translation,
            ...R.omit(["obsolete, translation"], nextCatalog[key])
          }
        };
      });

      // Mark all remaining translations as obsolete
      // Only if *options.files* is not provided
      const obsoleteMessages = obsoleteKeys.map(key => ({
        [key]: {
          ...prevCatalog[key],
          obsolete: options.files ? false : true
        }
      }));
      return R.mergeAll([newMessages, ...mergedMessages, ...obsoleteMessages]);
    }, prevCatalogs);
  }
  getTranslations(locale, options) {
    const catalogs = this.readAll();
    const template = this.readTemplate() || {};
    return R.mapObjIndexed((_value, key) => this.getTranslation(catalogs, locale, key, options), {
      ...template,
      ...catalogs[locale]
    });
  }
  getTranslation(catalogs, locale, key, {
    fallbackLocales,
    sourceLocale
  }) {
    var _catalog$key;
    const catalog = catalogs[locale] || {};
    const getTranslation = _locale => {
      const configLocales = this.config.locales.join('", "');
      const localeCatalog = catalogs[_locale] || {};
      if (!localeCatalog) {
        console.warn(`
        Catalog "${_locale}" isn't present in locales config parameter
        Add "${_locale}" to your lingui.config.js:
        {
          locales: ["${configLocales}", "${_locale}"]
        }
      `);
        return null;
      }
      if (!localeCatalog.hasOwnProperty(key)) {
        return null;
      }
      if (catalogs[_locale]) {
        return catalogs[_locale][key].translation;
      }
      return null;
    };
    const getMultipleFallbacks = _locale => {
      const fL = fallbackLocales && fallbackLocales[_locale];

      // some probably the fallback will be undefined, so just search by locale
      if (!fL) return null;
      if (Array.isArray(fL)) {
        for (const fallbackLocale of fL) {
          if (catalogs[fallbackLocale]) {
            return getTranslation(fallbackLocale);
          }
        }
      } else {
        return getTranslation(fL);
      }
    };
    return (
      // Get translation in target locale
      getTranslation(locale) ||
      // We search in fallbackLocales as dependent of each locale
      getMultipleFallbacks(locale) ||
      // Get translation in fallbackLocales.default (if any)
      (fallbackLocales === null || fallbackLocales === void 0 ? void 0 : fallbackLocales.default) && getTranslation(fallbackLocales.default) || ( // Get message default
      (_catalog$key = catalog[key]) === null || _catalog$key === void 0 ? void 0 : _catalog$key.defaults) ||
      // If sourceLocale is either target locale of fallback one, use key
      sourceLocale && sourceLocale === locale && key || sourceLocale && (fallbackLocales === null || fallbackLocales === void 0 ? void 0 : fallbackLocales.default) && sourceLocale === fallbackLocales.default && key ||
      // Otherwise no translation is available
      undefined
    );
  }
  write(locale, messages) {
    const filename = this.path.replace(LOCALE_REPLACE_RE, locale) + this.format.catalogExtension;
    const created = !_fsExtra.default.existsSync(filename);
    const basedir = _path.default.dirname(filename);
    if (!_fsExtra.default.existsSync(basedir)) {
      _fsExtra.default.mkdirpSync(basedir);
    }
    const options = {
      ...this.config.formatOptions,
      locale
    };
    this.format.write(filename, messages, options);
    return [created, filename];
  }
  writeAll(catalogs) {
    this.locales.forEach(locale => this.write(locale, catalogs[locale]));
  }
  writeTemplate(messages) {
    const filename = this.templateFile;
    const basedir = _path.default.dirname(filename);
    if (!_fsExtra.default.existsSync(basedir)) {
      _fsExtra.default.mkdirpSync(basedir);
    }
    const options = {
      ...this.config.formatOptions,
      locale: undefined
    };
    this.format.write(filename, messages, options);
  }
  writeCompiled(locale, compiledCatalog, namespace) {
    let ext;
    if (namespace === "es") {
      ext = "mjs";
    } else if (namespace === "ts") {
      ext = "ts";
    } else {
      ext = "js";
    }
    const filename = `${this.path.replace(LOCALE_REPLACE_RE, locale)}.${ext}`;
    const basedir = _path.default.dirname(filename);
    if (!_fsExtra.default.existsSync(basedir)) {
      _fsExtra.default.mkdirpSync(basedir);
    }
    _fsExtra.default.writeFileSync(filename, compiledCatalog);
    return filename;
  }
  read(locale) {
    const filename = this.path.replace(LOCALE_REPLACE_RE, locale) + this.format.catalogExtension;
    if (!_fsExtra.default.existsSync(filename)) return null;
    return this.format.read(filename);
  }
  readAll() {
    return R.mergeAll(this.locales.map(locale => ({
      [locale]: this.read(locale)
    })));
  }
  readTemplate() {
    const filename = this.templateFile;
    if (!_fsExtra.default.existsSync(filename)) return null;
    return this.format.read(filename);
  }
  get sourcePaths() {
    const includeGlobs = this.include.map(includePath => {
      const isDir = _fsExtra.default.existsSync(includePath) && _fsExtra.default.lstatSync(includePath).isDirectory();
      /**
       * glob library results from absolute patterns such as /foo/* are mounted onto the root setting using path.join.
       * On windows, this will by default result in /foo/* matching C:\foo\bar.txt.
       */
      return isDir ? (0, _normalizePath.default)(_path.default.resolve(process.cwd(), includePath === "/" ? "" : includePath, "**/*.*")) : includePath;
    });
    const patterns = includeGlobs.length > 1 ? `{${includeGlobs.join(",")}}` : includeGlobs[0];
    return _glob.default.sync(patterns, {
      ignore: this.exclude,
      mark: true
    });
  }
  get templateFile() {
    return this.path.replace(LOCALE_SUFFIX_RE, "messages.pot");
  }
  get localeDir() {
    const localePatternIndex = this.path.indexOf(LOCALE);
    if (localePatternIndex === -1) {
      throw Error(`Invalid catalog path: ${LOCALE} variable is missing`);
    }
    return this.path.substr(0, localePatternIndex);
  }
  get locales() {
    return this.config.locales;
  }
}

/**
 * Parse `config.catalogs` and return a list of configured Catalog instances.
 */
exports.Catalog = Catalog;
function getCatalogs(config) {
  const catalogsConfig = config.catalogs;
  const catalogs = [];
  catalogsConfig.forEach(catalog => {
    // Validate that `catalogPath` doesn't end with trailing slash
    if (catalog.path.endsWith(PATHSEP)) {
      const extension = (0, _formats.default)(config.format).catalogExtension;
      const correctPath = catalog.path.slice(0, -1);
      const examplePath = correctPath.replace(LOCALE_REPLACE_RE,
      // Show example using one of configured locales (if any)
      (config.locales || [])[0] || "en") + extension;
      throw new Error(
      // prettier-ignore
      `Remove trailing slash from "${catalog.path}". Catalog path isn't a directory,` + ` but translation file without extension. For example, catalog path "${correctPath}"` + ` results in translation file "${examplePath}".`);
    }
    const include = ensureArray(catalog.include).map(normalizeRelativePath);
    const exclude = ensureArray(catalog.exclude).map(normalizeRelativePath);

    // catalog.path without {name} pattern -> always refers to a single catalog
    if (!catalog.path.includes(NAME)) {
      // Validate that sourcePaths doesn't use {name} pattern either
      const invalidSource = include.find(path => path.includes(NAME));
      if (invalidSource !== undefined) {
        throw new Error(`Catalog with path "${catalog.path}" doesn't have a {name} pattern` + ` in it, but one of source directories uses it: "${invalidSource}".` + ` Either add {name} pattern to "${catalog.path}" or remove it` + ` from all source directories.`);
      }

      // catalog name is the last directory of catalog.path.
      // If the last part is {locale}, then catalog doesn't have an explicit name
      const name = function () {
        const _name = catalog.path.split(PATHSEP).slice(-1)[0];
        return _name !== LOCALE ? _name : null;
      }();
      catalogs.push(new Catalog({
        name,
        path: normalizeRelativePath(catalog.path),
        include,
        exclude
      }, config));
      return;
    }
    const patterns = include.map(path => path.replace(NAME_REPLACE_RE, "*"));
    const candidates = _glob.default.sync(patterns.length > 1 ? `{${patterns.join(",")}}` : patterns[0], {
      ignore: exclude,
      mark: true
    });
    candidates.forEach(catalogDir => {
      const name = _path.default.basename(catalogDir);
      catalogs.push(new Catalog({
        name,
        path: normalizeRelativePath(catalog.path.replace(NAME_REPLACE_RE, name)),
        include: include.map(path => path.replace(NAME_REPLACE_RE, name)),
        exclude: exclude.map(path => path.replace(NAME_REPLACE_RE, name))
      }, config));
    });
  });
  return catalogs;
}
function getCatalogForFile(file, catalogs) {
  for (const catalog of catalogs) {
    const catalogFile = `${catalog.path}${catalog.format.catalogExtension}`;
    const catalogGlob = catalogFile.replace(LOCALE_REPLACE_RE, "*");
    const match = _micromatch.default.capture(normalizeRelativePath(_path.default.relative(catalog.config.rootDir, catalogGlob)), normalizeRelativePath(file));
    if (match) {
      return {
        locale: match[0],
        catalog
      };
    }
  }
  return null;
}

/**
 * Create catalog for merged messages.
 */
function getCatalogForMerge(config) {
  const catalogConfig = config;
  if (catalogConfig.catalogsMergePath.endsWith(PATHSEP)) {
    const extension = (0, _formats.default)(config.format).catalogExtension;
    const correctPath = catalogConfig.catalogsMergePath.slice(0, -1);
    const examplePath = correctPath.replace(LOCALE_REPLACE_RE,
    // Show example using one of configured locales (if any)
    (config.locales || [])[0] || "en") + extension;
    throw new Error(
    // prettier-ignore
    `Remove trailing slash from "${catalogConfig.catalogsMergePath}". Catalog path isn't a directory,` + ` but translation file without extension. For example, catalog path "${correctPath}"` + ` results in translation file "${examplePath}".`);
  }

  // catalog name is the last directory of catalogPath.
  // If the last part is {locale}, then catalog doesn't have an explicit name
  const name = function () {
    const _name = _path.default.basename(normalizeRelativePath(catalogConfig.catalogsMergePath));
    return _name !== LOCALE ? _name : null;
  }();
  const catalog = new Catalog({
    name,
    path: normalizeRelativePath(catalogConfig.catalogsMergePath),
    include: [],
    exclude: []
  }, config);
  return catalog;
}

/**
 * Merge origins and extractedComments for messages found in different places. All other attributes
 * should be the same (raise an error if defaults are different).
 */
function mergeOriginsAndExtractedComments(msgId, prev, next) {
  if (prev.defaults !== next.defaults) {
    throw new Error(`Encountered different defaults for message ${_chalk.default.yellow(msgId)}` + `\n${_chalk.default.yellow((0, _utils.prettyOrigin)(prev.origin))} ${prev.defaults}` + `\n${_chalk.default.yellow((0, _utils.prettyOrigin)(next.origin))} ${next.defaults}`);
  }
  return {
    ...next,
    extractedComments: R.concat(prev.extractedComments, next.extractedComments),
    origin: R.concat(prev.origin, next.origin)
  };
}

/**
 * Ensure that value is always array. If not, turn it into an array of one element.
 */
const ensureArray = value => {
  if (value == null) return [];
  return Array.isArray(value) ? value : [value];
};

/**
 * Remove ./ at the beginning: ./relative  => relative
 *                             relative    => relative
 * Preserve directories:       ./relative/ => relative/
 * Preserve absolute paths:    /absolute/path => /absolute/path
 */
function normalizeRelativePath(sourcePath) {
  if (_path.default.isAbsolute(sourcePath)) {
    // absolute path
    return (0, _normalizePath.default)(sourcePath, false);
  }
  const isDir = _fsExtra.default.existsSync(sourcePath) && _fsExtra.default.lstatSync(sourcePath).isDirectory();
  return (0, _normalizePath.default)(_path.default.relative(process.cwd(), sourcePath), false) + (isDir ? "/" : "");
}
const cleanObsolete = R.filter(message => !message.obsolete);
exports.cleanObsolete = cleanObsolete;
function order(by) {
  return {
    messageId: orderByMessageId,
    origin: orderByOrigin
  }[by];
}

/**
 * Object keys are in the same order as they were created
 * https://stackoverflow.com/a/31102605/1535540
 */
function orderByMessageId(messages) {
  const orderedMessages = {};
  Object.keys(messages).sort().forEach(function (key) {
    orderedMessages[key] = messages[key];
  });
  return orderedMessages;
}
function orderByOrigin(messages) {
  function getFirstOrigin(messageKey) {
    const sortedOrigins = messages[messageKey].origin.sort((a, b) => {
      if (a[0] < b[0]) return -1;
      if (a[0] > b[0]) return 1;
      return 0;
    });
    return sortedOrigins[0];
  }
  return Object.keys(messages).sort(function (a, b) {
    const [aFile, aLineNumber] = getFirstOrigin(a);
    const [bFile, bLineNumber] = getFirstOrigin(b);
    if (aFile < bFile) return -1;
    if (aFile > bFile) return 1;
    if (aLineNumber < bLineNumber) return -1;
    if (aLineNumber > bLineNumber) return 1;
    return 0;
  }).reduce((acc, key) => {
    acc[key] = messages[key];
    return acc;
  }, {});
}