import 'package:flutter/material.dart';

class BurstPetalShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const BurstPetalShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _BurstPetalShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _BurstPetalPainter(
            color: color,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
            gradient: gradient,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _BurstPetalClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _BurstPetalPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _BurstPetalPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildBurstPetalPath(size);

    final paint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      paint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, paint);

    if (isStroked) {
      final strokePaint = Paint()
        ..style = PaintingStyle.stroke
        ..color = strokeColor
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => true;
}

class _BurstPetalShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _BurstPetalShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildBurstPetalPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _BurstPetalClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildBurstPetalPath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildBurstPetalPath(Size size) {
  final path = Path();
  path.moveTo(size.width * 0.74, size.height * 0.4);
  path.cubicTo(
    size.width * 0.76,
    size.height * 0.3,
    size.width * 0.9,
    size.height * 0.3,
    size.width * 0.92,
    size.height * 0.4,
  );
  path.cubicTo(
    size.width * 0.93,
    size.height * 0.47,
    size.width,
    size.height / 2,
    size.width * 1.06,
    size.height * 0.47,
  );
  path.cubicTo(
    size.width * 1.15,
    size.height * 0.41,
    size.width * 1.25,
    size.height * 0.51,
    size.width * 1.19,
    size.height * 0.59,
  );
  path.cubicTo(
    size.width * 1.16,
    size.height * 0.65,
    size.width * 1.19,
    size.height * 0.72,
    size.width * 1.25,
    size.height * 0.74,
  );
  path.cubicTo(
    size.width * 1.35,
    size.height * 0.76,
    size.width * 1.35,
    size.height * 0.9,
    size.width * 1.26,
    size.height * 0.92,
  );
  path.cubicTo(
    size.width * 1.19,
    size.height * 0.93,
    size.width * 1.16,
    size.height,
    size.width * 1.19,
    size.height * 1.06,
  );
  path.cubicTo(
    size.width * 1.25,
    size.height * 1.15,
    size.width * 1.15,
    size.height * 1.25,
    size.width * 1.07,
    size.height * 1.19,
  );
  path.cubicTo(
    size.width,
    size.height * 1.16,
    size.width * 0.93,
    size.height * 1.19,
    size.width * 0.92,
    size.height * 1.25,
  );
  path.cubicTo(
    size.width * 0.9,
    size.height * 1.35,
    size.width * 0.76,
    size.height * 1.35,
    size.width * 0.74,
    size.height * 1.26,
  );
  path.cubicTo(
    size.width * 0.72,
    size.height * 1.19,
    size.width * 0.65,
    size.height * 1.16,
    size.width * 0.6,
    size.height * 1.19,
  );
  path.cubicTo(
    size.width * 0.51,
    size.height * 1.25,
    size.width * 0.41,
    size.height * 1.15,
    size.width * 0.46,
    size.height * 1.07,
  );
  path.cubicTo(
    size.width / 2,
    size.height,
    size.width * 0.47,
    size.height * 0.93,
    size.width * 0.41,
    size.height * 0.92,
  );
  path.cubicTo(
    size.width * 0.3,
    size.height * 0.9,
    size.width * 0.3,
    size.height * 0.76,
    size.width * 0.4,
    size.height * 0.74,
  );
  path.cubicTo(
    size.width * 0.47,
    size.height * 0.72,
    size.width / 2,
    size.height * 0.65,
    size.width * 0.47,
    size.height * 0.6,
  );
  path.cubicTo(
    size.width * 0.41,
    size.height * 0.51,
    size.width * 0.51,
    size.height * 0.41,
    size.width * 0.59,
    size.height * 0.46,
  );
  path.cubicTo(
    size.width * 0.65,
    size.height / 2,
    size.width * 0.72,
    size.height * 0.47,
    size.width * 0.74,
    size.height * 0.41,
  );
  path.close();
  return path;
}
