import 'package:flutter/material.dart';

class PetalShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const PetalShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final Widget petal = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _PetalShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _PetalPainter(
            color: color,
            gradient: gradient,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _PetalClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: petal),
    );
  }
}

class _PetalPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _PetalPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildPetalPath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => true;
}

class _PetalShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _PetalShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildPetalPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => false;
}

class _PetalClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildPetalPath(size);

  @override
  bool shouldReclip(CustomClipper<Path> oldClipper) => false;
}

Path _buildPetalPath(Size size) {
  Path path = Path();

  path.moveTo(size.width * 0.57, size.height * 0.23);
  path.cubicTo(
    size.width * 0.63,
    size.height * 0.13,
    size.width * 0.76,
    size.height * 0.13,
    size.width * 0.81,
    size.height * 0.23,
  );
  path.cubicTo(
    size.width * 0.81,
    size.height * 0.23,
    size.width * 1.18,
    size.height * 0.92,
    size.width * 1.18,
    size.height * 0.92,
  );
  path.cubicTo(
    size.width * 1.23,
    size.height * 1.03,
    size.width * 1.16,
    size.height * 1.15,
    size.width * 1.06,
    size.height * 1.15,
  );
  path.cubicTo(
    size.width * 1.06,
    size.height * 1.15,
    size.width / 3,
    size.height * 1.15,
    size.width / 3,
    size.height * 1.15,
  );
  path.cubicTo(
    size.width * 0.23,
    size.height * 1.15,
    size.width * 0.16,
    size.height * 1.03,
    size.width / 5,
    size.height * 0.92,
  );
  path.cubicTo(
    size.width / 5,
    size.height * 0.92,
    size.width * 0.57,
    size.height * 0.23,
    size.width * 0.57,
    size.height * 0.23,
  );
  path.close();

  return path;
}
