import 'package:flutter/material.dart';

class VerticalBulgedShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const VerticalBulgedShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _VerticalBulgedShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _VerticalBulgedPainter(
            color: color,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
            gradient: gradient,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _VerticalBulgedClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _VerticalBulgedPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _VerticalBulgedPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildVerticalBulgedPath(size);

    final paint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      paint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, paint);

    if (isStroked) {
      final strokePaint = Paint()
        ..style = PaintingStyle.stroke
        ..color = strokeColor
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => true;
}

class _VerticalBulgedShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _VerticalBulgedShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildVerticalBulgedPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _VerticalBulgedClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildVerticalBulgedPath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildVerticalBulgedPath(Size size) {
  final path = Path();
  path.moveTo(size.width * 0.45, size.height * 0.02);
  path.cubicTo(
    size.width * 0.48,
    -0.01,
    size.width * 0.52,
    -0.01,
    size.width * 0.55,
    size.height * 0.02,
  );
  path.cubicTo(
    size.width * 0.64,
    size.height * 0.11,
    size.width * 0.67,
    size.height * 0.12,
    size.width * 0.8,
    size.height * 0.12,
  );
  path.cubicTo(
    size.width * 0.84,
    size.height * 0.12,
    size.width * 0.88,
    size.height * 0.16,
    size.width * 0.88,
    size.height / 5,
  );
  path.cubicTo(
    size.width * 0.88,
    size.height / 3,
    size.width * 0.88,
    size.height * 0.35,
    size.width * 0.89,
    size.height * 0.36,
  );
  path.cubicTo(
    size.width * 0.98,
    size.height * 0.45,
    size.width,
    size.height * 0.52,
    size.width * 0.98,
    size.height * 0.55,
  );
  path.cubicTo(
    size.width * 0.89,
    size.height * 0.64,
    size.width * 0.88,
    size.height * 0.67,
    size.width * 0.88,
    size.height * 0.8,
  );
  path.cubicTo(
    size.width * 0.88,
    size.height * 0.84,
    size.width * 0.84,
    size.height * 0.88,
    size.width * 0.8,
    size.height * 0.88,
  );
  path.cubicTo(
    size.width * 0.67,
    size.height * 0.88,
    size.width * 0.64,
    size.height * 0.89,
    size.width * 0.55,
    size.height * 0.98,
  );
  path.cubicTo(
    size.width * 0.52,
    size.height,
    size.width * 0.48,
    size.height,
    size.width * 0.45,
    size.height * 0.98,
  );
  path.cubicTo(
    size.width * 0.36,
    size.height * 0.89,
    size.width / 3,
    size.height * 0.88,
    size.width / 5,
    size.height * 0.88,
  );
  path.cubicTo(
    size.width * 0.16,
    size.height * 0.88,
    size.width * 0.12,
    size.height * 0.84,
    size.width * 0.12,
    size.height * 0.8,
  );
  path.cubicTo(
    size.width * 0.12,
    size.height * 0.67,
    size.width * 0.11,
    size.height * 0.64,
    size.width * 0.02,
    size.height * 0.55,
  );
  path.cubicTo(
    -0.01,
    size.height * 0.48,
    -0.01,
    size.height * 0.52,
    size.width * 0.02,
    size.height * 0.45,
  );
  path.cubicTo(
    size.width * 0.11,
    size.height * 0.36,
    size.width * 0.12,
    size.height * 0.34,
    size.width * 0.12,
    size.height / 3,
  );
  path.cubicTo(
    size.width * 0.12,
    size.height / 5,
    size.width * 0.16,
    size.height * 0.12,
    size.width / 5,
    size.height * 0.12,
  );
  path.cubicTo(
    size.width / 3,
    size.height * 0.12,
    size.width * 0.34,
    size.height * 0.12,
    size.width * 0.36,
    size.height * 0.11,
  );
  path.cubicTo(
    size.width * 0.45,
    size.height * 0.02,
    size.width * 0.45,
    size.height * 0.02,
    size.width * 0.45,
    size.height * 0.02,
  );
  path.close();
  return path;
}
