import 'package:flutter/material.dart';

class RoundedSquareShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const RoundedSquareShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final Widget shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _RoundedSquareShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _RoundedSquarePainter(
            color: color,
            gradient: gradient,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _RoundedSquareClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _RoundedSquarePainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _RoundedSquarePainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final rRect = RRect.fromRectAndRadius(
      Rect.fromLTWH(0, 0, size.width, size.height),
      Radius.circular(size.width * 0.246), // equivalent to 123 on 500
    );

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawRRect(rRect, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawRRect(rRect, strokePaint);
    }
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => true;
}

class _RoundedSquareShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _RoundedSquareShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final rRect = RRect.fromRectAndRadius(
      Rect.fromLTWH(0, 0, size.width, size.height),
      Radius.circular(size.width * 0.246),
    );
    final paint = shadow.toPaint();
    canvas.drawRRect(rRect.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _RoundedSquareClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) {
    final rRect = RRect.fromRectAndRadius(
      Rect.fromLTWH(0, 0, size.width, size.height),
      Radius.circular(size.width * 0.246),
    );
    return Path()..addRRect(rRect);
  }

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}
