import 'package:flutter/material.dart';

class RippleHeartShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const RippleHeartShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _RippleHeartShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _RippleHeartPainter(
            color: color,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
            gradient: gradient,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _RippleHeartClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _RippleHeartPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _RippleHeartPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildRippleHeartPath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => true;
}

class _RippleHeartShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _RippleHeartShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildRippleHeartPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _RippleHeartClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildRippleHeartPath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildRippleHeartPath(Size size) {
  Path path = Path();
  path.moveTo(size.width / 2, 0);
  path.cubicTo(
    size.width * 0.62,
    0,
    size.width * 0.54,
    size.height * 0.31,
    size.width * 0.54,
    size.height * 0.31,
  );
  path.cubicTo(
    size.width * 0.54,
    size.height * 0.31,
    size.width * 0.58,
    -0.01,
    size.width * 0.69,
    size.height * 0.04,
  );
  path.cubicTo(
    size.width * 0.8,
    size.height * 0.09,
    size.width * 0.61,
    size.height * 0.34,
    size.width * 0.61,
    size.height * 0.34,
  );
  path.cubicTo(
    size.width * 0.61,
    size.height * 0.34,
    size.width * 0.77,
    size.height * 0.06,
    size.width * 0.85,
    size.height * 0.15,
  );
  path.cubicTo(
    size.width * 0.94,
    size.height * 0.23,
    size.width * 0.66,
    size.height * 0.39,
    size.width * 0.66,
    size.height * 0.39,
  );
  path.cubicTo(
    size.width * 0.66,
    size.height * 0.39,
    size.width * 0.91,
    size.height / 5,
    size.width * 0.96,
    size.height * 0.31,
  );
  path.cubicTo(
    size.width,
    size.height * 0.42,
    size.width * 0.69,
    size.height * 0.46,
    size.width * 0.69,
    size.height * 0.46,
  );
  path.cubicTo(
    size.width * 0.69,
    size.height * 0.46,
    size.width,
    size.height * 0.38,
    size.width,
    size.height / 2,
  );
  path.cubicTo(
    size.width,
    size.height * 0.62,
    size.width * 0.69,
    size.height * 0.54,
    size.width * 0.69,
    size.height * 0.54,
  );
  path.cubicTo(
    size.width * 0.69,
    size.height * 0.54,
    size.width,
    size.height * 0.58,
    size.width * 0.96,
    size.height * 0.69,
  );
  path.cubicTo(
    size.width * 0.91,
    size.height * 0.8,
    size.width * 0.66,
    size.height * 0.61,
    size.width * 0.66,
    size.height * 0.61,
  );
  path.cubicTo(
    size.width * 0.66,
    size.height * 0.61,
    size.width * 0.94,
    size.height * 0.77,
    size.width * 0.85,
    size.height * 0.85,
  );
  path.cubicTo(
    size.width * 0.77,
    size.height * 0.94,
    size.width * 0.61,
    size.height * 0.66,
    size.width * 0.61,
    size.height * 0.66,
  );
  path.cubicTo(
    size.width * 0.61,
    size.height * 0.66,
    size.width * 0.8,
    size.height * 0.91,
    size.width * 0.69,
    size.height * 0.96,
  );
  path.cubicTo(
    size.width * 0.58,
    size.height,
    size.width * 0.54,
    size.height * 0.69,
    size.width * 0.54,
    size.height * 0.69,
  );
  path.cubicTo(
    size.width * 0.54,
    size.height * 0.69,
    size.width * 0.62,
    size.height,
    size.width / 2,
    size.height,
  );
  path.cubicTo(
    size.width * 0.38,
    size.height,
    size.width * 0.46,
    size.height * 0.69,
    size.width * 0.46,
    size.height * 0.69,
  );
  path.cubicTo(
    size.width * 0.46,
    size.height * 0.69,
    size.width * 0.42,
    size.height,
    size.width * 0.31,
    size.height * 0.96,
  );
  path.cubicTo(
    size.width / 5,
    size.height * 0.91,
    size.width * 0.39,
    size.height * 0.66,
    size.width * 0.39,
    size.height * 0.66,
  );
  path.cubicTo(
    size.width * 0.39,
    size.height * 0.66,
    size.width * 0.23,
    size.height * 0.94,
    size.width * 0.15,
    size.height * 0.85,
  );
  path.cubicTo(
    size.width * 0.06,
    size.height * 0.77,
    size.width * 0.34,
    size.height * 0.61,
    size.width * 0.34,
    size.height * 0.61,
  );
  path.cubicTo(
    size.width * 0.34,
    size.height * 0.61,
    size.width * 0.09,
    size.height * 0.8,
    size.width * 0.04,
    size.height * 0.69,
  );
  path.cubicTo(
    -0.01,
    size.height * 0.58,
    size.width * 0.31,
    size.height * 0.54,
    size.width * 0.31,
    size.height * 0.54,
  );
  path.cubicTo(
    size.width * 0.31,
    size.height * 0.54,
    0,
    size.height * 0.62,
    0,
    size.height / 2,
  );
  path.cubicTo(
    0,
    size.height * 0.38,
    size.width * 0.31,
    size.height * 0.46,
    size.width * 0.31,
    size.height * 0.46,
  );
  path.cubicTo(
    size.width * 0.31,
    size.height * 0.46,
    -0.01,
    size.height * 0.42,
    size.width * 0.04,
    size.height * 0.31,
  );
  path.cubicTo(
    size.width * 0.09,
    size.height / 5,
    size.width * 0.34,
    size.height * 0.39,
    size.width * 0.34,
    size.height * 0.39,
  );
  path.cubicTo(
    size.width * 0.34,
    size.height * 0.39,
    size.width * 0.06,
    size.height * 0.23,
    size.width * 0.15,
    size.height * 0.15,
  );
  path.cubicTo(
    size.width * 0.23,
    size.height * 0.06,
    size.width * 0.39,
    size.height * 0.34,
    size.width * 0.39,
    size.height * 0.34,
  );
  path.cubicTo(
    size.width * 0.39,
    size.height * 0.34,
    size.width / 5,
    size.height * 0.09,
    size.width * 0.31,
    size.height * 0.04,
  );
  path.cubicTo(
    size.width * 0.42,
    -0.01,
    size.width * 0.46,
    size.height * 0.31,
    size.width * 0.46,
    size.height * 0.31,
  );
  path.cubicTo(
    size.width * 0.46,
    size.height * 0.31,
    size.width * 0.38,
    0,
    size.width / 2,
    0,
  );
  path.close();

  return path;
}
