import 'package:flutter/material.dart';

class RoundedCardBlobShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const RoundedCardBlobShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final Widget shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _RoundedCardBlobShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _RoundedCardBlobPainter(
            color: color,
            gradient: gradient,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _RoundedCardBlobClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _RoundedCardBlobPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _RoundedCardBlobPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildRoundedCardBlobPath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => true;
}

class _RoundedCardBlobShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _RoundedCardBlobShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildRoundedCardBlobPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _RoundedCardBlobClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildRoundedCardBlobPath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildRoundedCardBlobPath(Size size) {
  final path = Path();
  path.moveTo(0, 0);
  path.moveTo(size.width * 0.07, size.height * 0.22);
  path.cubicTo(
    size.width * 0.09,
    size.height * 0.1,
    size.width * 0.19,
    0,
    size.width * 0.31,
    0,
  );
  path.cubicTo(
    size.width * 0.31,
    0,
    size.width * 0.79,
    0,
    size.width * 0.79,
    0,
  );
  path.cubicTo(
    size.width * 0.93,
    0,
    size.width * 1.04,
    size.height * 0.13,
    size.width * 1.02,
    size.height * 0.27,
  );
  path.cubicTo(
    size.width * 1.02,
    size.height * 0.27,
    size.width * 0.97,
    size.height * 0.78,
    size.width * 0.97,
    size.height * 0.78,
  );
  path.cubicTo(
    size.width * 0.96,
    size.height * 0.9,
    size.width * 0.86,
    size.height,
    size.width * 0.74,
    size.height,
  );
  path.cubicTo(
    size.width * 0.74,
    size.height,
    size.width * 0.26,
    size.height,
    size.width * 0.26,
    size.height,
  );
  path.cubicTo(
    size.width * 0.12,
    size.height,
    size.width * 0.01,
    size.height * 0.87,
    size.width * 0.03,
    size.height * 0.73,
  );
  path.cubicTo(
    size.width * 0.03,
    size.height * 0.73,
    size.width * 0.07,
    size.height * 0.22,
    size.width * 0.07,
    size.height * 0.22,
  );
  return path;
}
