import 'package:flutter/material.dart';

class BlobCircleShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const BlobCircleShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _BlobCircleShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _BlobCirclePainter(
            color: color,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
            gradient: gradient,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _BlobCircleClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _BlobCirclePainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _BlobCirclePainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildBlobCirclePath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => true;
}

class _BlobCircleShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _BlobCircleShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildBlobCirclePath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _BlobCircleClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildBlobCirclePath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildBlobCirclePath(Size size) {
  Path path = Path();

  path.moveTo(size.width / 2, 0);
  path.cubicTo(
    size.width * 0.59,
    0,
    size.width * 0.66,
    size.height * 0.07,
    size.width * 0.66,
    size.height * 0.16,
  );
  path.cubicTo(
    size.width * 0.66,
    size.height * 0.16,
    size.width * 0.66,
    size.height * 0.19,
    size.width * 0.66,
    size.height * 0.19,
  );
  path.cubicTo(
    size.width * 0.66,
    size.height * 0.19,
    size.width * 0.69,
    size.height * 0.19,
    size.width * 0.69,
    size.height * 0.19,
  );
  path.cubicTo(
    size.width * 0.76,
    size.height * 0.19,
    size.width * 0.81,
    size.height * 0.24,
    size.width * 0.81,
    size.height * 0.31,
  );
  path.cubicTo(
    size.width * 0.81,
    size.height * 0.31,
    size.width * 0.81,
    size.height * 0.34,
    size.width * 0.81,
    size.height * 0.34,
  );
  path.cubicTo(
    size.width * 0.81,
    size.height * 0.34,
    size.width * 0.84,
    size.height * 0.34,
    size.width * 0.84,
    size.height * 0.34,
  );
  path.cubicTo(
    size.width * 0.93,
    size.height * 0.34,
    size.width,
    size.height * 0.41,
    size.width,
    size.height / 2,
  );
  path.cubicTo(
    size.width,
    size.height * 0.59,
    size.width * 0.93,
    size.height * 0.66,
    size.width * 0.84,
    size.height * 0.66,
  );
  path.cubicTo(
    size.width * 0.84,
    size.height * 0.66,
    size.width * 0.81,
    size.height * 0.66,
    size.width * 0.81,
    size.height * 0.66,
  );
  path.cubicTo(
    size.width * 0.81,
    size.height * 0.66,
    size.width * 0.81,
    size.height * 0.69,
    size.width * 0.81,
    size.height * 0.69,
  );
  path.cubicTo(
    size.width * 0.81,
    size.height * 0.76,
    size.width * 0.76,
    size.height * 0.81,
    size.width * 0.69,
    size.height * 0.81,
  );
  path.cubicTo(
    size.width * 0.69,
    size.height * 0.81,
    size.width * 0.66,
    size.height * 0.81,
    size.width * 0.66,
    size.height * 0.81,
  );
  path.cubicTo(
    size.width * 0.66,
    size.height * 0.81,
    size.width * 0.66,
    size.height * 0.84,
    size.width * 0.66,
    size.height * 0.84,
  );
  path.cubicTo(
    size.width * 0.66,
    size.height * 0.93,
    size.width * 0.59,
    size.height,
    size.width / 2,
    size.height,
  );
  path.cubicTo(
    size.width * 0.41,
    size.height,
    size.width * 0.34,
    size.height * 0.93,
    size.width * 0.34,
    size.height * 0.84,
  );
  path.cubicTo(
    size.width * 0.34,
    size.height * 0.84,
    size.width * 0.34,
    size.height * 0.81,
    size.width * 0.34,
    size.height * 0.81,
  );
  path.cubicTo(
    size.width * 0.34,
    size.height * 0.81,
    size.width * 0.31,
    size.height * 0.81,
    size.width * 0.31,
    size.height * 0.81,
  );
  path.cubicTo(
    size.width * 0.24,
    size.height * 0.81,
    size.width * 0.19,
    size.height * 0.76,
    size.width * 0.19,
    size.height * 0.69,
  );
  path.cubicTo(
    size.width * 0.19,
    size.height * 0.69,
    size.width * 0.19,
    size.height * 0.66,
    size.width * 0.19,
    size.height * 0.66,
  );
  path.cubicTo(
    size.width * 0.19,
    size.height * 0.66,
    size.width * 0.16,
    size.height * 0.66,
    size.width * 0.16,
    size.height * 0.66,
  );
  path.cubicTo(
    size.width * 0.07,
    size.height * 0.66,
    0,
    size.height * 0.59,
    0,
    size.height / 2,
  );
  path.cubicTo(
    0,
    size.height * 0.41,
    size.width * 0.07,
    size.height * 0.34,
    size.width * 0.16,
    size.height * 0.34,
  );
  path.cubicTo(
    size.width * 0.16,
    size.height * 0.34,
    size.width * 0.19,
    size.height * 0.34,
    size.width * 0.19,
    size.height * 0.34,
  );
  path.cubicTo(
    size.width * 0.19,
    size.height * 0.34,
    size.width * 0.19,
    size.height * 0.31,
    size.width * 0.19,
    size.height * 0.31,
  );
  path.cubicTo(
    size.width * 0.19,
    size.height * 0.24,
    size.width * 0.24,
    size.height * 0.19,
    size.width * 0.31,
    size.height * 0.19,
  );
  path.cubicTo(
    size.width * 0.31,
    size.height * 0.19,
    size.width * 0.34,
    size.height * 0.19,
    size.width * 0.34,
    size.height * 0.19,
  );
  path.cubicTo(
    size.width * 0.34,
    size.height * 0.19,
    size.width * 0.34,
    size.height * 0.16,
    size.width * 0.34,
    size.height * 0.16,
  );
  path.cubicTo(
    size.width * 0.34,
    size.height * 0.07,
    size.width * 0.41,
    0,
    size.width / 2,
    0,
  );
  path.close();

  return path;
}
