import 'package:flutter/material.dart';

class DoubleBlobShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const DoubleBlobShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _DoubleBlobShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _DoubleBlobPainter(
            color: color,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
            gradient: gradient,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _DoubleBlobClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _DoubleBlobPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _DoubleBlobPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildDoubleBlobPath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => true;
}

class _DoubleBlobShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _DoubleBlobShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildDoubleBlobPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _DoubleBlobClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildDoubleBlobPath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildDoubleBlobPath(Size size) {
  Path path = Path();

  path.moveTo(size.width * 0.4, 0);
  path.cubicTo(
    size.width * 0.44,
    0,
    size.width * 0.48,
    size.height * 0.03,
    size.width / 2,
    size.height * 0.08,
  );
  path.cubicTo(
    size.width * 0.52,
    size.height * 0.04,
    size.width * 0.56,
    0,
    size.width * 0.6,
    0,
  );
  path.cubicTo(
    size.width * 0.66,
    0,
    size.width * 0.71,
    size.height * 0.07,
    size.width * 0.71,
    size.height * 0.14,
  );
  path.cubicTo(
    size.width * 0.71,
    size.height * 0.17,
    size.width * 0.73,
    size.height * 0.15,
    size.width * 0.73,
    size.height * 0.15,
  );
  path.cubicTo(
    size.width * 0.77,
    size.height * 0.1,
    size.width * 0.84,
    size.height * 0.1,
    size.width * 0.88,
    size.height * 0.15,
  );
  path.cubicTo(
    size.width * 0.93,
    size.height / 5,
    size.width * 0.93,
    size.height * 0.3,
    size.width * 0.88,
    size.height * 0.35,
  );
  path.cubicTo(
    size.width * 0.87,
    size.height * 0.36,
    size.width * 0.89,
    size.height * 0.36,
    size.width * 0.89,
    size.height * 0.36,
  );
  path.cubicTo(
    size.width * 0.95,
    size.height * 0.36,
    size.width,
    size.height * 0.42,
    size.width,
    size.height / 2,
  );
  path.cubicTo(
    size.width,
    size.height * 0.58,
    size.width * 0.95,
    size.height * 0.64,
    size.width * 0.89,
    size.height * 0.64,
  );
  path.cubicTo(
    size.width * 0.87,
    size.height * 0.64,
    size.width * 0.88,
    size.height * 0.66,
    size.width * 0.88,
    size.height * 0.66,
  );
  path.cubicTo(
    size.width * 0.93,
    size.height * 0.71,
    size.width * 0.93,
    size.height * 0.8,
    size.width * 0.88,
    size.height * 0.85,
  );
  path.cubicTo(
    size.width * 0.84,
    size.height * 0.91,
    size.width * 0.77,
    size.height * 0.91,
    size.width * 0.73,
    size.height * 0.85,
  );
  path.cubicTo(
    size.width * 0.71,
    size.height * 0.84,
    size.width * 0.71,
    size.height * 0.86,
    size.width * 0.71,
    size.height * 0.86,
  );
  path.cubicTo(
    size.width * 0.71,
    size.height * 0.94,
    size.width * 0.66,
    size.height,
    size.width * 0.6,
    size.height,
  );
  path.cubicTo(
    size.width * 0.56,
    size.height,
    size.width * 0.52,
    size.height * 0.97,
    size.width / 2,
    size.height * 0.92,
  );
  path.cubicTo(
    size.width * 0.48,
    size.height * 0.96,
    size.width * 0.44,
    size.height,
    size.width * 0.4,
    size.height,
  );
  path.cubicTo(
    size.width * 0.34,
    size.height,
    size.width * 0.29,
    size.height * 0.93,
    size.width * 0.29,
    size.height * 0.86,
  );
  path.cubicTo(
    size.width * 0.29,
    size.height * 0.84,
    size.width * 0.28,
    size.height * 0.85,
    size.width * 0.28,
    size.height * 0.85,
  );
  path.cubicTo(
    size.width * 0.23,
    size.height * 0.91,
    size.width * 0.16,
    size.height * 0.91,
    size.width * 0.12,
    size.height * 0.85,
  );
  path.cubicTo(
    size.width * 0.07,
    size.height * 0.8,
    size.width * 0.07,
    size.height * 0.71,
    size.width * 0.12,
    size.height * 0.66,
  );
  path.cubicTo(
    size.width * 0.13,
    size.height * 0.64,
    size.width * 0.11,
    size.height * 0.64,
    size.width * 0.11,
    size.height * 0.64,
  );
  path.cubicTo(
    size.width * 0.05,
    size.height * 0.64,
    0,
    size.height * 0.58,
    0,
    size.height / 2,
  );
  path.cubicTo(
    0,
    size.height * 0.42,
    size.width * 0.05,
    size.height * 0.36,
    size.width * 0.11,
    size.height * 0.36,
  );
  path.cubicTo(
    size.width * 0.13,
    size.height * 0.36,
    size.width * 0.12,
    size.height * 0.34,
    size.width * 0.12,
    size.height * 0.34,
  );
  path.cubicTo(
    size.width * 0.07,
    size.height * 0.29,
    size.width * 0.07,
    size.height / 5,
    size.width * 0.12,
    size.height * 0.15,
  );
  path.cubicTo(
    size.width * 0.16,
    size.height * 0.09,
    size.width * 0.23,
    size.height * 0.09,
    size.width * 0.28,
    size.height * 0.15,
  );
  path.cubicTo(
    size.width * 0.29,
    size.height * 0.16,
    size.width * 0.29,
    size.height * 0.14,
    size.width * 0.29,
    size.height * 0.14,
  );
  path.cubicTo(
    size.width * 0.29,
    size.height * 0.06,
    size.width * 0.34,
    0,
    size.width * 0.4,
    0,
  );
  path.close();

  return path;
}
