import 'package:flutter/material.dart';

class PixelTriangleShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const PixelTriangleShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _PixelTriangleShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _PixelTrianglePainter(
            color: color,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
            gradient: gradient,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _PixelTriangleClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _PixelTrianglePainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _PixelTrianglePainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildPixelTrianglePath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => true;
}

class _PixelTriangleShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _PixelTriangleShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildPixelTrianglePath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _PixelTriangleClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildPixelTrianglePath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildPixelTrianglePath(Size size) {
  Path path = Path();

  path.moveTo(size.width * 0.26, size.height * 0.08);
  path.cubicTo(
    size.width * 0.26,
    size.height * 0.08,
    size.width * 0.48,
    size.height * 0.08,
    size.width * 0.48,
    size.height * 0.08,
  );
  path.cubicTo(
    size.width * 0.48,
    size.height * 0.08,
    size.width * 0.48,
    size.height * 0.17,
    size.width * 0.48,
    size.height * 0.17,
  );
  path.cubicTo(
    size.width * 0.48,
    size.height * 0.17,
    size.width * 0.61,
    size.height * 0.17,
    size.width * 0.61,
    size.height * 0.17,
  );
  path.cubicTo(
    size.width * 0.61,
    size.height * 0.17,
    size.width * 0.61,
    size.height / 4,
    size.width * 0.61,
    size.height / 4,
  );
  path.cubicTo(
    size.width * 0.61,
    size.height / 4,
    size.width * 0.74,
    size.height / 4,
    size.width * 0.74,
    size.height / 4,
  );
  path.cubicTo(
    size.width * 0.74,
    size.height / 4,
    size.width * 0.74,
    size.height / 3,
    size.width * 0.74,
    size.height / 3,
  );
  path.cubicTo(
    size.width * 0.74,
    size.height / 3,
    size.width * 0.87,
    size.height / 3,
    size.width * 0.87,
    size.height / 3,
  );
  path.cubicTo(
    size.width * 0.87,
    size.height / 3,
    size.width * 0.87,
    size.height * 0.42,
    size.width * 0.87,
    size.height * 0.42,
  );
  path.cubicTo(
    size.width * 0.87,
    size.height * 0.42,
    size.width,
    size.height * 0.42,
    size.width,
    size.height * 0.42,
  );
  path.cubicTo(
    size.width,
    size.height * 0.42,
    size.width,
    size.height * 0.58,
    size.width,
    size.height * 0.58,
  );
  path.cubicTo(
    size.width,
    size.height * 0.58,
    size.width * 0.87,
    size.height * 0.58,
    size.width * 0.87,
    size.height * 0.58,
  );
  path.cubicTo(
    size.width * 0.87,
    size.height * 0.58,
    size.width * 0.87,
    size.height * 0.67,
    size.width * 0.87,
    size.height * 0.67,
  );
  path.cubicTo(
    size.width * 0.87,
    size.height * 0.67,
    size.width * 0.74,
    size.height * 0.67,
    size.width * 0.74,
    size.height * 0.67,
  );
  path.cubicTo(
    size.width * 0.74,
    size.height * 0.67,
    size.width * 0.74,
    size.height * 0.75,
    size.width * 0.74,
    size.height * 0.75,
  );
  path.cubicTo(
    size.width * 0.74,
    size.height * 0.75,
    size.width * 0.61,
    size.height * 0.75,
    size.width * 0.61,
    size.height * 0.75,
  );
  path.cubicTo(
    size.width * 0.61,
    size.height * 0.75,
    size.width * 0.61,
    size.height * 0.83,
    size.width * 0.61,
    size.height * 0.83,
  );
  path.cubicTo(
    size.width * 0.61,
    size.height * 0.83,
    size.width * 0.48,
    size.height * 0.83,
    size.width * 0.48,
    size.height * 0.83,
  );
  path.cubicTo(
    size.width * 0.48,
    size.height * 0.83,
    size.width * 0.48,
    size.height * 0.92,
    size.width * 0.48,
    size.height * 0.92,
  );
  path.cubicTo(
    size.width * 0.48,
    size.height * 0.92,
    size.width * 0.26,
    size.height * 0.92,
    size.width * 0.26,
    size.height * 0.92,
  );
  path.cubicTo(
    size.width * 0.26,
    size.height * 0.92,
    size.width * 0.26,
    size.height,
    size.width * 0.26,
    size.height,
  );
  path.cubicTo(size.width * 0.26, size.height, 0, size.height, 0, size.height);
  path.cubicTo(0, size.height, 0, 0, 0, 0);
  path.cubicTo(0, 0, size.width * 0.26, 0, size.width * 0.26, 0);
  path.cubicTo(
    size.width * 0.26,
    0,
    size.width * 0.26,
    size.height * 0.08,
    size.width * 0.26,
    size.height * 0.08,
  );
  path.close();

  return path;
}
