import 'package:flutter/material.dart';

class WindowFrameShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const WindowFrameShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _WindowFrameShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _WindowFramePainter(
            color: color,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
            gradient: gradient,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _WindowFrameClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _WindowFramePainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _WindowFramePainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildWindowFramePath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => true;
}

class _WindowFrameShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _WindowFrameShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildWindowFramePath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _WindowFrameClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildWindowFramePath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildWindowFramePath(Size size) {
  Path path = Path();

  path.moveTo(size.width * 0.75, size.height * 0.05);
  path.cubicTo(
    size.width * 0.75,
    size.height * 0.05,
    size.width * 0.85,
    size.height * 0.05,
    size.width * 0.85,
    size.height * 0.05,
  );
  path.cubicTo(
    size.width * 0.85,
    size.height * 0.05,
    size.width * 0.85,
    size.height * 0.15,
    size.width * 0.85,
    size.height * 0.15,
  );
  path.cubicTo(
    size.width * 0.85,
    size.height * 0.15,
    size.width * 0.95,
    size.height * 0.15,
    size.width * 0.95,
    size.height * 0.15,
  );
  path.cubicTo(
    size.width * 0.95,
    size.height * 0.15,
    size.width * 0.95,
    size.height / 4,
    size.width * 0.95,
    size.height / 4,
  );
  path.cubicTo(
    size.width * 0.95,
    size.height / 4,
    size.width,
    size.height / 4,
    size.width,
    size.height / 4,
  );
  path.cubicTo(
    size.width,
    size.height / 4,
    size.width,
    size.height * 0.75,
    size.width,
    size.height * 0.75,
  );
  path.cubicTo(
    size.width,
    size.height * 0.75,
    size.width * 0.95,
    size.height * 0.75,
    size.width * 0.95,
    size.height * 0.75,
  );
  path.cubicTo(
    size.width * 0.95,
    size.height * 0.75,
    size.width * 0.95,
    size.height * 0.85,
    size.width * 0.95,
    size.height * 0.85,
  );
  path.cubicTo(
    size.width * 0.95,
    size.height * 0.85,
    size.width * 0.85,
    size.height * 0.85,
    size.width * 0.85,
    size.height * 0.85,
  );
  path.cubicTo(
    size.width * 0.85,
    size.height * 0.85,
    size.width * 0.85,
    size.height * 0.95,
    size.width * 0.85,
    size.height * 0.95,
  );
  path.cubicTo(
    size.width * 0.85,
    size.height * 0.95,
    size.width * 0.75,
    size.height * 0.95,
    size.width * 0.75,
    size.height * 0.95,
  );
  path.cubicTo(
    size.width * 0.75,
    size.height * 0.95,
    size.width * 0.75,
    size.height,
    size.width * 0.75,
    size.height,
  );
  path.cubicTo(
    size.width * 0.75,
    size.height,
    size.width / 4,
    size.height,
    size.width / 4,
    size.height,
  );
  path.cubicTo(
    size.width / 4,
    size.height,
    size.width / 4,
    size.height * 0.95,
    size.width / 4,
    size.height * 0.95,
  );
  path.cubicTo(
    size.width / 4,
    size.height * 0.95,
    size.width * 0.15,
    size.height * 0.95,
    size.width * 0.15,
    size.height * 0.95,
  );
  path.cubicTo(
    size.width * 0.15,
    size.height * 0.95,
    size.width * 0.15,
    size.height * 0.85,
    size.width * 0.15,
    size.height * 0.85,
  );
  path.cubicTo(
    size.width * 0.15,
    size.height * 0.85,
    size.width * 0.05,
    size.height * 0.85,
    size.width * 0.05,
    size.height * 0.85,
  );
  path.cubicTo(
    size.width * 0.05,
    size.height * 0.85,
    size.width * 0.05,
    size.height * 0.75,
    size.width * 0.05,
    size.height * 0.75,
  );
  path.cubicTo(
    size.width * 0.05,
    size.height * 0.75,
    0,
    size.height * 0.75,
    0,
    size.height * 0.75,
  );
  path.cubicTo(0, size.height * 0.75, 0, size.height / 4, 0, size.height / 4);
  path.cubicTo(
    0,
    size.height / 4,
    size.width * 0.05,
    size.height / 4,
    size.width * 0.05,
    size.height / 4,
  );
  path.cubicTo(
    size.width * 0.05,
    size.height / 4,
    size.width * 0.05,
    size.height * 0.15,
    size.width * 0.05,
    size.height * 0.15,
  );
  path.cubicTo(
    size.width * 0.05,
    size.height * 0.15,
    size.width * 0.15,
    size.height * 0.15,
    size.width * 0.15,
    size.height * 0.15,
  );
  path.cubicTo(
    size.width * 0.15,
    size.height * 0.15,
    size.width * 0.15,
    size.height * 0.05,
    size.width * 0.15,
    size.height * 0.05,
  );
  path.cubicTo(
    size.width * 0.15,
    size.height * 0.05,
    size.width / 4,
    size.height * 0.05,
    size.width / 4,
    size.height * 0.05,
  );
  path.cubicTo(
    size.width / 4,
    size.height * 0.05,
    size.width / 4,
    0,
    size.width / 4,
    0,
  );
  path.cubicTo(size.width / 4, 0, size.width * 0.75, 0, size.width * 0.75, 0);
  path.cubicTo(
    size.width * 0.75,
    0,
    size.width * 0.75,
    size.height * 0.05,
    size.width * 0.75,
    size.height * 0.05,
  );
  path.close();

  return path;
}
