import 'package:flutter/material.dart';

class HeartShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const HeartShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final Widget heartShape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _HeartShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _HeartPainter(
            color: color,
            gradient: gradient,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _HeartClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: heartShape),
    );
  }
}

class _HeartPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _HeartPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildHeartPath(size);
    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => true;
}

class _HeartShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _HeartShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildHeartPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _HeartClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildHeartPath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildHeartPath(Size size) {
  final path = Path();
  final double scaleX = size.width / 184;
  final double scaleY = size.height / 164;

  path.moveTo(97.7947 * scaleX, 15.3629 * scaleY);
  path.cubicTo(
    117.325 * scaleX,
    -4.16746 * scaleY,
    148.99 * scaleX,
    -4.16731 * scaleY,
    168.52 * scaleX,
    15.3629 * scaleY,
  );
  path.cubicTo(
    188.051 * scaleX,
    34.8933 * scaleY,
    188.051 * scaleX,
    66.5581 * scaleY,
    168.52 * scaleX,
    86.0885 * scaleY,
  );
  path.lineTo(91.7254 * scaleX, 162.883 * scaleY);
  path.lineTo(91.7234 * scaleX, 162.881 * scaleY);
  path.lineTo(91.447 * scaleX, 163.158 * scaleY);
  path.lineTo(14.6443 * scaleX, 86.3551 * scaleY);
  path.cubicTo(
    -4.88145 * scaleX,
    66.8289 * scaleY,
    -4.88142 * scaleX,
    35.1712 * scaleY,
    14.6443 * scaleX,
    15.6451 * scaleY,
  );
  path.cubicTo(
    34.1705 * scaleX,
    -3.88112 * scaleY,
    65.829 * scaleX,
    -3.88112 * scaleY,
    85.3552 * scaleX,
    15.6451 * scaleY,
  );
  path.lineTo(91.4334 * scaleX, 21.7232 * scaleY);
  path.lineTo(97.7947 * scaleX, 15.3629 * scaleY);
  path.close();

  return path;
}
