import 'package:flutter/material.dart';

class BlobBottomShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const BlobBottomShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _BlobBottomShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _BlobBottomPainter(
            color: color,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
            gradient: gradient,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _BlobBottomClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _BlobBottomPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _BlobBottomPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildBlobBottomPath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => true;
}

class _BlobBottomShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _BlobBottomShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildBlobBottomPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _BlobBottomClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildBlobBottomPath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildBlobBottomPath(Size size) {
  Path path = Path();

  path.moveTo(0, size.height / 2);
  path.cubicTo(0, size.height * 0.22, size.width * 0.22, 0, size.width / 2, 0);
  path.cubicTo(
    size.width * 0.78,
    0,
    size.width,
    size.height * 0.22,
    size.width,
    size.height / 2,
  );
  path.cubicTo(
    size.width,
    size.height / 2,
    size.width,
    size.height * 0.83,
    size.width,
    size.height * 0.83,
  );
  path.cubicTo(
    size.width,
    size.height * 0.97,
    size.width * 0.83,
    size.height * 1.04,
    size.width * 0.72,
    size.height * 0.96,
  );
  path.cubicTo(
    size.width * 0.55,
    size.height * 0.84,
    size.width * 0.46,
    size.height * 0.84,
    size.width * 0.28,
    size.height * 0.97,
  );
  path.cubicTo(
    size.width * 0.16,
    size.height * 1.05,
    0,
    size.height * 0.97,
    0,
    size.height * 0.83,
  );
  path.cubicTo(0, size.height * 0.83, 0, size.height / 2, 0, size.height / 2);
  path.close();

  return path;
}
