import 'package:flutter/material.dart';

class JellybeanShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const JellybeanShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _JellybeanShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _JellybeanPainter(
            color: color,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
            gradient: gradient,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _JellybeanClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _JellybeanPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _JellybeanPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildJellybeanPath(size);

    final paint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      paint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, paint);

    if (isStroked) {
      final strokePaint = Paint()
        ..style = PaintingStyle.stroke
        ..color = strokeColor
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => true;
}

class _JellybeanShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _JellybeanShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildJellybeanPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _JellybeanClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildJellybeanPath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildJellybeanPath(Size size) {
  final path = Path();
  path.moveTo(size.width * 0.39, size.height * 0.08);
  path.cubicTo(
    size.width * 0.47,
    size.height * 0.03,
    size.width * 0.57,
    size.height * 0.03,
    size.width * 0.65,
    size.height * 0.08,
  );
  path.cubicTo(
    size.width * 0.88,
    size.height * 0.23,
    size.width * 0.93,
    size.height * 0.26,
    size.width * 0.98,
    size.height * 0.39,
  );
  path.cubicTo(
    size.width * 1.02,
    size.height * 0.69,
    size.width * 1.03,
    size.height * 0.79,
    size.width * 0.87,
    size.height * 0.92,
  );
  path.cubicTo(
    size.width * 0.61,
    size.height * 1.03,
    size.width * 0.55,
    size.height * 1.05,
    size.width * 0.43,
    size.height * 1.03,
  );
  path.cubicTo(
    size.width * 0.17,
    size.height * 0.92,
    size.width * 0.07,
    size.height * 0.89,
    size.width * 0.02,
    size.height * 0.69,
  );
  path.cubicTo(
    size.width * 0.05,
    size.height * 0.39,
    size.width * 0.06,
    size.height * 0.32,
    size.width * 0.16,
    size.height * 0.23,
  );
  path.cubicTo(
    size.width * 0.39,
    size.height * 0.08,
    size.width * 0.39,
    size.height * 0.08,
    size.width * 0.39,
    size.height * 0.08,
  );
  path.close();
  return path;
}
