import 'package:flutter/material.dart';

class CustomShape4 extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const CustomShape4({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _CustomShape4ShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _CustomShape4Painter(
            color: color,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
            gradient: gradient,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _CustomShape4Clipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _CustomShape4Painter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _CustomShape4Painter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildFourLeafCloverPath(size);

    final paint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      paint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, paint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => true;
}

class _CustomShape4ShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _CustomShape4ShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildFourLeafCloverPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _CustomShape4Clipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildFourLeafCloverPath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildFourLeafCloverPath(Size size) {
  final path = Path();

  path.moveTo(size.width / 2, size.height * 0.09);
  path.cubicTo(
    size.width * 0.61,
    -0.03,
    size.width * 0.8,
    -0.03,
    size.width * 0.91,
    size.height * 0.09,
  );
  path.cubicTo(
    size.width * 1.03,
    size.height / 5,
    size.width * 1.03,
    size.height * 0.39,
    size.width * 0.91,
    size.height / 2,
  );
  path.cubicTo(
    size.width * 1.03,
    size.height * 0.61,
    size.width * 1.03,
    size.height * 0.8,
    size.width * 0.91,
    size.height * 0.91,
  );
  path.cubicTo(
    size.width * 0.8,
    size.height * 1.03,
    size.width * 0.61,
    size.height * 1.03,
    size.width / 2,
    size.height * 0.91,
  );
  path.cubicTo(
    size.width * 0.39,
    size.height * 1.03,
    size.width / 5,
    size.height * 1.03,
    size.width * 0.09,
    size.height * 0.91,
  );
  path.cubicTo(
    -0.03,
    size.height * 0.8,
    -0.03,
    size.height * 0.61,
    size.width * 0.09,
    size.height / 2,
  );
  path.cubicTo(
    -0.03,
    size.height * 0.39,
    -0.03,
    size.height / 5,
    size.width * 0.09,
    size.height * 0.09,
  );
  path.cubicTo(
    size.width / 5,
    -0.03,
    size.width * 0.39,
    -0.03,
    size.width / 2,
    size.height * 0.09,
  );
  path.close();

  return path;
}
