import 'package:flutter/material.dart';

class GearFlowerShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const GearFlowerShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _GearFlowerShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _GearFlowerPainter(
            color: color,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
            gradient: gradient,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _GearFlowerClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _GearFlowerPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _GearFlowerPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildGearFlowerPath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => true;
}

class _GearFlowerShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _GearFlowerShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildGearFlowerPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _GearFlowerClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildGearFlowerPath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildGearFlowerPath(Size size) {
  Path path = Path();
  path.moveTo(size.width * 0.44, size.height * 0.04);
  path.cubicTo(
    size.width * 0.46,
    -0.01,
    size.width * 0.54,
    -0.01,
    size.width * 0.56,
    size.height * 0.04,
  );
  path.cubicTo(
    size.width * 0.56,
    size.height * 0.04,
    size.width * 0.62,
    size.height / 5,
    size.width * 0.62,
    size.height / 5,
  );
  path.cubicTo(
    size.width * 0.62,
    size.height / 5,
    size.width * 0.78,
    size.height * 0.13,
    size.width * 0.78,
    size.height * 0.13,
  );
  path.cubicTo(
    size.width * 0.84,
    size.height * 0.11,
    size.width * 0.89,
    size.height * 0.16,
    size.width * 0.87,
    size.height * 0.22,
  );
  path.cubicTo(
    size.width * 0.87,
    size.height * 0.22,
    size.width * 0.79,
    size.height * 0.38,
    size.width * 0.79,
    size.height * 0.38,
  );
  path.cubicTo(
    size.width * 0.79,
    size.height * 0.38,
    size.width * 0.96,
    size.height * 0.44,
    size.width * 0.96,
    size.height * 0.44,
  );
  path.cubicTo(
    size.width,
    size.height * 0.46,
    size.width,
    size.height * 0.54,
    size.width * 0.96,
    size.height * 0.56,
  );
  path.cubicTo(
    size.width * 0.96,
    size.height * 0.56,
    size.width * 0.79,
    size.height * 0.62,
    size.width * 0.79,
    size.height * 0.62,
  );
  path.cubicTo(
    size.width * 0.79,
    size.height * 0.62,
    size.width * 0.87,
    size.height * 0.78,
    size.width * 0.87,
    size.height * 0.78,
  );
  path.cubicTo(
    size.width * 0.89,
    size.height * 0.84,
    size.width * 0.84,
    size.height * 0.89,
    size.width * 0.78,
    size.height * 0.87,
  );
  path.cubicTo(
    size.width * 0.78,
    size.height * 0.87,
    size.width * 0.62,
    size.height * 0.79,
    size.width * 0.62,
    size.height * 0.79,
  );
  path.cubicTo(
    size.width * 0.62,
    size.height * 0.79,
    size.width * 0.56,
    size.height * 0.96,
    size.width * 0.56,
    size.height * 0.96,
  );
  path.cubicTo(
    size.width * 0.54,
    size.height,
    size.width * 0.46,
    size.height,
    size.width * 0.44,
    size.height * 0.96,
  );
  path.cubicTo(
    size.width * 0.44,
    size.height * 0.96,
    size.width * 0.38,
    size.height * 0.79,
    size.width * 0.38,
    size.height * 0.79,
  );
  path.cubicTo(
    size.width * 0.38,
    size.height * 0.79,
    size.width * 0.22,
    size.height * 0.87,
    size.width * 0.22,
    size.height * 0.87,
  );
  path.cubicTo(
    size.width * 0.16,
    size.height * 0.89,
    size.width * 0.11,
    size.height * 0.84,
    size.width * 0.13,
    size.height * 0.78,
  );
  path.cubicTo(
    size.width * 0.13,
    size.height * 0.78,
    size.width / 5,
    size.height * 0.62,
    size.width / 5,
    size.height * 0.62,
  );
  path.cubicTo(
    size.width / 5,
    size.height * 0.62,
    size.width * 0.04,
    size.height * 0.56,
    size.width * 0.04,
    size.height * 0.56,
  );
  path.cubicTo(
    -0.01,
    size.height * 0.54,
    -0.01,
    size.height * 0.46,
    size.width * 0.04,
    size.height * 0.44,
  );
  path.cubicTo(
    size.width * 0.04,
    size.height * 0.44,
    size.width / 5,
    size.height * 0.38,
    size.width / 5,
    size.height * 0.38,
  );
  path.cubicTo(
    size.width / 5,
    size.height * 0.38,
    size.width * 0.13,
    size.height * 0.22,
    size.width * 0.13,
    size.height * 0.22,
  );
  path.cubicTo(
    size.width * 0.11,
    size.height * 0.16,
    size.width * 0.16,
    size.height * 0.11,
    size.width * 0.22,
    size.height * 0.13,
  );
  path.cubicTo(
    size.width * 0.22,
    size.height * 0.13,
    size.width * 0.38,
    size.height / 5,
    size.width * 0.38,
    size.height / 5,
  );
  path.cubicTo(
    size.width * 0.38,
    size.height / 5,
    size.width * 0.44,
    size.height * 0.04,
    size.width * 0.44,
    size.height * 0.04,
  );
  path.close();

  return path;
}
