import 'package:flutter/material.dart';

class CapsuleShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const CapsuleShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final Widget capsule = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _CapsuleShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _CapsulePainter(
            color: color,
            gradient: gradient,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _CapsuleClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: capsule),
    );
  }
}

class _CapsulePainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _CapsulePainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildCapsulePath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => true;
}

class _CapsuleShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _CapsuleShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildCapsulePath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => false;
}

class _CapsuleClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildCapsulePath(size);

  @override
  bool shouldReclip(CustomClipper<Path> oldClipper) => false;
}

Path _buildCapsulePath(Size size) {
  Path path = Path();

  path.moveTo(0, size.height * 0.17);
  path.cubicTo(
    0,
    size.height * 0.07,
    size.width * 0.07,
    0,
    size.width * 0.17,
    0,
  );
  path.cubicTo(
    size.width * 0.63,
    0,
    size.width,
    size.height * 0.37,
    size.width,
    size.height * 0.83,
  );
  path.cubicTo(
    size.width,
    size.height * 0.93,
    size.width * 0.93,
    size.height,
    size.width * 0.83,
    size.height,
  );
  path.cubicTo(
    size.width * 0.83,
    size.height,
    size.width / 5,
    size.height,
    size.width / 5,
    size.height,
  );
  path.cubicTo(
    size.width * 0.09,
    size.height,
    0,
    size.height * 0.91,
    0,
    size.height * 0.8,
  );
  path.cubicTo(
    0,
    size.height * 0.8,
    0,
    size.height * 0.17,
    0,
    size.height * 0.17,
  );
  path.close();

  return path;
}
