import 'package:flutter/material.dart';

class RibbonShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const RibbonShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final Widget ribbon = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _RibbonShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _RibbonPainter(
            color: color,
            gradient: gradient,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _RibbonClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: ribbon),
    );
  }
}

class _RibbonPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _RibbonPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildRibbonPath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => true;
}

class _RibbonShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _RibbonShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildRibbonPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => false;
}

class _RibbonClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildRibbonPath(size);

  @override
  bool shouldReclip(CustomClipper<Path> oldClipper) => false;
}

Path _buildRibbonPath(Size size) {
  Path path = Path();

  path.moveTo(size.width * 0.3, size.height * 0.58);
  path.cubicTo(
    size.width * 0.27,
    size.height * 0.54,
    size.width * 0.27,
    size.height * 0.49,
    size.width * 0.31,
    size.height * 0.46,
  );
  path.cubicTo(
    size.width * 0.31,
    size.height * 0.46,
    size.width * 0.66,
    size.height * 0.12,
    size.width * 0.66,
    size.height * 0.12,
  );
  path.cubicTo(
    size.width * 0.72,
    size.height * 0.06,
    size.width * 0.84,
    size.height * 0.06,
    size.width * 0.9,
    size.height * 0.12,
  );
  path.cubicTo(
    size.width * 0.9,
    size.height * 0.12,
    size.width * 1.25,
    size.height * 0.46,
    size.width * 1.25,
    size.height * 0.46,
  );
  path.cubicTo(
    size.width * 1.29,
    size.height * 0.49,
    size.width * 1.29,
    size.height * 0.54,
    size.width * 1.26,
    size.height * 0.58,
  );
  path.cubicTo(
    size.width * 1.26,
    size.height * 0.58,
    size.width * 0.91,
    size.height * 1.02,
    size.width * 0.91,
    size.height * 1.02,
  );
  path.cubicTo(
    size.width * 0.85,
    size.height * 1.09,
    size.width * 0.71,
    size.height * 1.09,
    size.width * 0.65,
    size.height * 1.02,
  );
  path.cubicTo(
    size.width * 0.65,
    size.height * 1.02,
    size.width * 0.3,
    size.height * 0.58,
    size.width * 0.3,
    size.height * 0.58,
  );
  path.close();

  return path;
}
