import 'package:flutter/material.dart';

class BlobShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const BlobShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final Widget blob = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _BlobShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _BlobPainter(
            color: color,
            gradient: gradient,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _BlobClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: blob),
    );
  }
}

class _BlobPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _BlobPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildBlobPath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => true;
}

class _BlobShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _BlobShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildBlobPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => false;
}

class _BlobClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildBlobPath(size);

  @override
  bool shouldReclip(CustomClipper<Path> oldClipper) => false;
}

Path _buildBlobPath(Size size) {
  Path path = Path();

  path.moveTo(size.width * 0.05, size.height * 0.62);
  path.cubicTo(
    size.width * 0.02,
    size.height * 0.54,
    size.width * 0.02,
    size.height * 0.46,
    size.width * 0.05,
    size.height * 0.38,
  );
  path.cubicTo(
    size.width * 0.05,
    size.height * 0.38,
    size.width * 0.16,
    size.height * 0.12,
    size.width * 0.16,
    size.height * 0.12,
  );
  path.cubicTo(
    size.width * 0.19,
    size.height * 0.04,
    size.width * 0.24,
    0,
    size.width * 0.3,
    0,
  );
  path.cubicTo(size.width * 0.3, 0, size.width * 0.75, 0, size.width * 0.75, 0);
  path.cubicTo(
    size.width * 0.81,
    0,
    size.width * 0.86,
    size.height * 0.04,
    size.width * 0.89,
    size.height * 0.12,
  );
  path.cubicTo(
    size.width * 0.89,
    size.height * 0.12,
    size.width,
    size.height * 0.38,
    size.width,
    size.height * 0.38,
  );
  path.cubicTo(
    size.width * 1.03,
    size.height * 0.46,
    size.width * 1.03,
    size.height * 0.54,
    size.width,
    size.height * 0.62,
  );
  path.cubicTo(
    size.width,
    size.height * 0.62,
    size.width * 0.89,
    size.height * 0.88,
    size.width * 0.89,
    size.height * 0.88,
  );
  path.cubicTo(
    size.width * 0.86,
    size.height * 0.96,
    size.width * 0.81,
    size.height,
    size.width * 0.75,
    size.height,
  );
  path.cubicTo(
    size.width * 0.75,
    size.height,
    size.width * 0.3,
    size.height,
    size.width * 0.3,
    size.height,
  );
  path.cubicTo(
    size.width * 0.24,
    size.height,
    size.width * 0.19,
    size.height * 0.96,
    size.width * 0.16,
    size.height * 0.88,
  );
  path.cubicTo(
    size.width * 0.16,
    size.height * 0.88,
    size.width * 0.05,
    size.height * 0.62,
    size.width * 0.05,
    size.height * 0.62,
  );
  path.close();

  return path;
}
