import 'package:flutter/material.dart';

class SymmetricBlobShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const SymmetricBlobShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final Widget shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _SymmetricBlobShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _SymmetricBlobPainter(
            color: color,
            gradient: gradient,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _SymmetricBlobClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _SymmetricBlobPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _SymmetricBlobPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildSymmetricBlobPath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => true;
}

class _SymmetricBlobShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _SymmetricBlobShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildSymmetricBlobPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _SymmetricBlobClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildSymmetricBlobPath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildSymmetricBlobPath(Size size) {
  final path = Path();

  // Start at top-center
  path.moveTo(size.width / 2, 0);

  // Top-right curve
  path.cubicTo(
    size.width * 0.78,
    0,
    size.width,
    size.height * 0.22,
    size.width,
    size.height / 2,
  );

  // Bottom-right curve
  path.cubicTo(
    size.width,
    size.height * 0.78,
    size.width * 0.78,
    size.height,
    size.width / 2,
    size.height,
  );

  // Bottom-left curve
  path.cubicTo(
    size.width * 0.22,
    size.height,
    0,
    size.height * 0.78,
    0,
    size.height / 2,
  );

  // Top-left curve
  path.cubicTo(0, size.height * 0.22, size.width * 0.22, 0, size.width / 2, 0);

  path.close();
  return path;
}
