import 'package:flutter/material.dart';

class BunShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const BunShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _BunShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _BunPainter(
            color: color,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
            gradient: gradient,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _BunClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _BunPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _BunPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildBunPath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => true;
}

class _BunShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _BunShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final paint = shadow.toPaint();
    final path = _buildBunPath(size);
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(covariant CustomPainter oldDelegate) => false;
}

class _BunClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildBunPath(size);

  @override
  bool shouldReclip(covariant CustomClipper<Path> oldClipper) => false;
}

Path _buildBunPath(Size size) {
  Path path = Path();
  path.moveTo(size.width * 0.75, 0);
  path.cubicTo(
    size.width * 0.89,
    0,
    size.width,
    size.height * 0.12,
    size.width,
    size.height * 0.27,
  );
  path.cubicTo(
    size.width,
    size.height * 0.37,
    size.width * 0.95,
    size.height * 0.45,
    size.width * 0.87,
    size.height / 2,
  );
  path.cubicTo(
    size.width * 0.95,
    size.height * 0.55,
    size.width,
    size.height * 0.63,
    size.width,
    size.height * 0.73,
  );
  path.cubicTo(
    size.width,
    size.height * 0.88,
    size.width * 0.89,
    size.height,
    size.width * 0.75,
    size.height,
  );
  path.cubicTo(
    size.width * 0.75,
    size.height,
    size.width / 4,
    size.height,
    size.width / 4,
    size.height,
  );
  path.cubicTo(
    size.width * 0.11,
    size.height,
    0,
    size.height * 0.88,
    0,
    size.height * 0.73,
  );
  path.cubicTo(
    0,
    size.height * 0.63,
    size.width * 0.05,
    size.height * 0.55,
    size.width * 0.13,
    size.height / 2,
  );
  path.cubicTo(
    size.width * 0.05,
    size.height * 0.45,
    0,
    size.height * 0.37,
    0,
    size.height * 0.27,
  );
  path.cubicTo(0, size.height * 0.12, size.width * 0.11, 0, size.width / 4, 0);
  path.cubicTo(size.width / 4, 0, size.width * 0.75, 0, size.width * 0.75, 0);
  path.cubicTo(
    size.width * 0.75,
    0,
    size.width * 0.75,
    0,
    size.width * 0.75,
    0,
  );
  path.close();
  return path;
}
