import 'package:flutter/material.dart';

class MandalaNovaShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const MandalaNovaShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final shape = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _MandalaNovaShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _MandalaNovaPainter(
            color: color,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
            gradient: gradient,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _MandalaNovaClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: shape),
    );
  }
}

class _MandalaNovaPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _MandalaNovaPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildMandalaNovaPath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => true;
}

class _MandalaNovaShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _MandalaNovaShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildMandalaNovaPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => false;
}

class _MandalaNovaClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildMandalaNovaPath(size);

  @override
  bool shouldReclip(CustomClipper<Path> oldClipper) => false;
}

Path _buildMandalaNovaPath(Size size) {
  Path path = Path();
  path.moveTo(size.width / 2, 0);
  path.cubicTo(
    size.width / 2,
    0,
    size.width * 0.53,
    size.height * 0.28,
    size.width * 0.53,
    size.height * 0.28,
  );
  path.cubicTo(
    size.width * 0.53,
    size.height * 0.29,
    size.width * 0.55,
    size.height * 0.29,
    size.width * 0.55,
    size.height * 0.29,
  );
  path.cubicTo(
    size.width * 0.55,
    size.height * 0.29,
    size.width * 0.69,
    size.height * 0.04,
    size.width * 0.69,
    size.height * 0.04,
  );
  path.cubicTo(
    size.width * 0.69,
    size.height * 0.04,
    size.width * 0.61,
    size.height * 0.31,
    size.width * 0.61,
    size.height * 0.31,
  );
  path.cubicTo(
    size.width * 0.61,
    size.height * 0.32,
    size.width * 0.62,
    size.height / 3,
    size.width * 0.63,
    size.height * 0.32,
  );
  path.cubicTo(
    size.width * 0.63,
    size.height * 0.32,
    size.width * 0.85,
    size.height * 0.15,
    size.width * 0.85,
    size.height * 0.15,
  );
  path.cubicTo(
    size.width * 0.85,
    size.height * 0.15,
    size.width * 0.68,
    size.height * 0.37,
    size.width * 0.68,
    size.height * 0.37,
  );
  path.cubicTo(
    size.width * 0.67,
    size.height * 0.38,
    size.width * 0.68,
    size.height * 0.39,
    size.width * 0.69,
    size.height * 0.39,
  );
  path.cubicTo(
    size.width * 0.69,
    size.height * 0.39,
    size.width * 0.96,
    size.height * 0.31,
    size.width * 0.96,
    size.height * 0.31,
  );
  path.cubicTo(
    size.width * 0.96,
    size.height * 0.31,
    size.width * 0.71,
    size.height * 0.45,
    size.width * 0.71,
    size.height * 0.45,
  );
  path.cubicTo(
    size.width * 0.71,
    size.height * 0.45,
    size.width * 0.71,
    size.height * 0.47,
    size.width * 0.72,
    size.height * 0.47,
  );
  path.cubicTo(
    size.width * 0.72,
    size.height * 0.47,
    size.width,
    size.height / 2,
    size.width,
    size.height / 2,
  );
  path.cubicTo(
    size.width,
    size.height / 2,
    size.width * 0.72,
    size.height * 0.53,
    size.width * 0.72,
    size.height * 0.53,
  );
  path.cubicTo(
    size.width * 0.71,
    size.height * 0.53,
    size.width * 0.71,
    size.height * 0.55,
    size.width * 0.71,
    size.height * 0.55,
  );
  path.cubicTo(
    size.width * 0.71,
    size.height * 0.55,
    size.width * 0.96,
    size.height * 0.69,
    size.width * 0.96,
    size.height * 0.69,
  );
  path.cubicTo(
    size.width * 0.96,
    size.height * 0.69,
    size.width * 0.69,
    size.height * 0.61,
    size.width * 0.69,
    size.height * 0.61,
  );
  path.cubicTo(
    size.width * 0.68,
    size.height * 0.61,
    size.width * 0.67,
    size.height * 0.62,
    size.width * 0.68,
    size.height * 0.63,
  );
  path.cubicTo(
    size.width * 0.68,
    size.height * 0.63,
    size.width * 0.85,
    size.height * 0.85,
    size.width * 0.85,
    size.height * 0.85,
  );
  path.cubicTo(
    size.width * 0.85,
    size.height * 0.85,
    size.width * 0.63,
    size.height * 0.68,
    size.width * 0.63,
    size.height * 0.68,
  );
  path.cubicTo(
    size.width * 0.62,
    size.height * 0.67,
    size.width * 0.61,
    size.height * 0.68,
    size.width * 0.61,
    size.height * 0.69,
  );
  path.cubicTo(
    size.width * 0.61,
    size.height * 0.69,
    size.width * 0.69,
    size.height * 0.96,
    size.width * 0.69,
    size.height * 0.96,
  );
  path.cubicTo(
    size.width * 0.69,
    size.height * 0.96,
    size.width * 0.55,
    size.height * 0.71,
    size.width * 0.55,
    size.height * 0.71,
  );
  path.cubicTo(
    size.width * 0.55,
    size.height * 0.71,
    size.width * 0.53,
    size.height * 0.71,
    size.width * 0.53,
    size.height * 0.72,
  );
  path.cubicTo(
    size.width * 0.53,
    size.height * 0.72,
    size.width / 2,
    size.height,
    size.width / 2,
    size.height,
  );
  path.cubicTo(
    size.width / 2,
    size.height,
    size.width * 0.47,
    size.height * 0.72,
    size.width * 0.47,
    size.height * 0.72,
  );
  path.cubicTo(
    size.width * 0.47,
    size.height * 0.71,
    size.width * 0.45,
    size.height * 0.71,
    size.width * 0.45,
    size.height * 0.71,
  );
  path.cubicTo(
    size.width * 0.45,
    size.height * 0.71,
    size.width * 0.31,
    size.height * 0.96,
    size.width * 0.31,
    size.height * 0.96,
  );
  path.cubicTo(
    size.width * 0.31,
    size.height * 0.96,
    size.width * 0.39,
    size.height * 0.69,
    size.width * 0.39,
    size.height * 0.69,
  );
  path.cubicTo(
    size.width * 0.39,
    size.height * 0.68,
    size.width * 0.38,
    size.height * 0.67,
    size.width * 0.37,
    size.height * 0.68,
  );
  path.cubicTo(
    size.width * 0.37,
    size.height * 0.68,
    size.width * 0.15,
    size.height * 0.85,
    size.width * 0.15,
    size.height * 0.85,
  );
  path.cubicTo(
    size.width * 0.15,
    size.height * 0.85,
    size.width * 0.32,
    size.height * 0.63,
    size.width * 0.32,
    size.height * 0.63,
  );
  path.cubicTo(
    size.width / 3,
    size.height * 0.62,
    size.width * 0.32,
    size.height * 0.61,
    size.width * 0.31,
    size.height * 0.61,
  );
  path.cubicTo(
    size.width * 0.31,
    size.height * 0.61,
    size.width * 0.04,
    size.height * 0.69,
    size.width * 0.04,
    size.height * 0.69,
  );
  path.cubicTo(
    size.width * 0.04,
    size.height * 0.69,
    size.width * 0.29,
    size.height * 0.55,
    size.width * 0.29,
    size.height * 0.55,
  );
  path.cubicTo(
    size.width * 0.29,
    size.height * 0.55,
    size.width * 0.29,
    size.height * 0.53,
    size.width * 0.28,
    size.height * 0.53,
  );
  path.cubicTo(
    size.width * 0.28,
    size.height * 0.53,
    0,
    size.height / 2,
    0,
    size.height / 2,
  );
  path.cubicTo(
    0,
    size.height / 2,
    size.width * 0.28,
    size.height * 0.47,
    size.width * 0.28,
    size.height * 0.47,
  );
  path.cubicTo(
    size.width * 0.29,
    size.height * 0.47,
    size.width * 0.29,
    size.height * 0.45,
    size.width * 0.29,
    size.height * 0.45,
  );
  path.cubicTo(
    size.width * 0.29,
    size.height * 0.45,
    size.width * 0.04,
    size.height * 0.31,
    size.width * 0.04,
    size.height * 0.31,
  );
  path.cubicTo(
    size.width * 0.04,
    size.height * 0.31,
    size.width * 0.31,
    size.height * 0.39,
    size.width * 0.31,
    size.height * 0.39,
  );
  path.cubicTo(
    size.width * 0.32,
    size.height * 0.39,
    size.width / 3,
    size.height * 0.38,
    size.width * 0.32,
    size.height * 0.37,
  );
  path.cubicTo(
    size.width * 0.32,
    size.height * 0.37,
    size.width * 0.15,
    size.height * 0.15,
    size.width * 0.15,
    size.height * 0.15,
  );
  path.cubicTo(
    size.width * 0.15,
    size.height * 0.15,
    size.width * 0.37,
    size.height * 0.32,
    size.width * 0.37,
    size.height * 0.32,
  );
  path.cubicTo(
    size.width * 0.38,
    size.height / 3,
    size.width * 0.39,
    size.height * 0.32,
    size.width * 0.39,
    size.height * 0.31,
  );
  path.cubicTo(
    size.width * 0.39,
    size.height * 0.31,
    size.width * 0.31,
    size.height * 0.04,
    size.width * 0.31,
    size.height * 0.04,
  );
  path.cubicTo(
    size.width * 0.31,
    size.height * 0.04,
    size.width * 0.45,
    size.height * 0.29,
    size.width * 0.45,
    size.height * 0.29,
  );
  path.cubicTo(
    size.width * 0.45,
    size.height * 0.29,
    size.width * 0.47,
    size.height * 0.29,
    size.width * 0.47,
    size.height * 0.28,
  );
  path.cubicTo(
    size.width * 0.47,
    size.height * 0.28,
    size.width / 2,
    0,
    size.width / 2,
    0,
  );
  path.close();
  return path;
}
