import 'package:flutter/material.dart';

class WingShape extends StatelessWidget {
  final double size;
  final Color color;
  final bool isStroked;
  final Color? strokeColor;
  final double strokeWidth;
  final ImageProvider? image;
  final BoxFit imageFit;
  final Alignment imageAlignment;
  final double imageScale;
  final Color? overlayColor;
  final Gradient? gradient;
  final BoxShadow? shadow;
  final VoidCallback? onTap;

  const WingShape({
    super.key,
    required this.size,
    required this.color,
    this.isStroked = false,
    this.strokeColor,
    this.strokeWidth = 2.0,
    this.image,
    this.imageFit = BoxFit.cover,
    this.imageAlignment = Alignment.center,
    this.imageScale = 1.0,
    this.overlayColor,
    this.gradient,
    this.shadow,
    this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    final Widget wing = Stack(
      alignment: Alignment.center,
      children: [
        if (shadow != null)
          CustomPaint(
            size: Size(size, size),
            painter: _WingShadowPainter(shadow!),
          ),
        CustomPaint(
          size: Size(size, size),
          painter: _WingPainter(
            color: color,
            gradient: gradient,
            isStroked: isStroked,
            strokeColor: strokeColor ?? Colors.black,
            strokeWidth: strokeWidth,
          ),
        ),
        if (image != null)
          ClipPath(
            clipper: _WingClipper(),
            child: Stack(
              children: [
                Transform.scale(
                  scale: imageScale,
                  child: Image(
                    image: image!,
                    width: size,
                    height: size,
                    fit: imageFit,
                    alignment: imageAlignment,
                  ),
                ),
                if (overlayColor != null)
                  Container(width: size, height: size, color: overlayColor),
              ],
            ),
          ),
      ],
    );

    return GestureDetector(
      onTap: onTap,
      child: SizedBox(width: size, height: size, child: wing),
    );
  }
}

class _WingPainter extends CustomPainter {
  final Color color;
  final bool isStroked;
  final Color strokeColor;
  final double strokeWidth;
  final Gradient? gradient;

  const _WingPainter({
    required this.color,
    required this.isStroked,
    required this.strokeColor,
    required this.strokeWidth,
    this.gradient,
  });

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildWingPath(size);

    final fillPaint = Paint()
      ..style = PaintingStyle.fill
      ..color = color;

    if (gradient != null) {
      fillPaint.shader = gradient!.createShader(
        Rect.fromLTWH(0, 0, size.width, size.height),
      );
    }

    canvas.drawPath(path, fillPaint);

    if (isStroked) {
      final strokePaint = Paint()
        ..color = strokeColor
        ..style = PaintingStyle.stroke
        ..strokeWidth = strokeWidth;
      canvas.drawPath(path, strokePaint);
    }
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => true;
}

class _WingShadowPainter extends CustomPainter {
  final BoxShadow shadow;

  const _WingShadowPainter(this.shadow);

  @override
  void paint(Canvas canvas, Size size) {
    final path = _buildWingPath(size);
    final paint = shadow.toPaint();
    canvas.drawPath(path.shift(shadow.offset), paint);
  }

  @override
  bool shouldRepaint(CustomPainter oldDelegate) => false;
}

class _WingClipper extends CustomClipper<Path> {
  @override
  Path getClip(Size size) => _buildWingPath(size);

  @override
  bool shouldReclip(CustomClipper<Path> oldClipper) => false;
}

Path _buildWingPath(Size size) {
  Path path = Path();

  path.moveTo(size.width * 0.59, size.height * 0.19);
  path.cubicTo(
    size.width * 0.63,
    size.height * 0.1,
    size.width * 0.74,
    size.height * 0.1,
    size.width * 0.78,
    size.height * 0.19,
  );
  path.cubicTo(
    size.width * 0.78,
    size.height * 0.19,
    size.width * 1.17,
    size.height * 0.91,
    size.width * 1.17,
    size.height * 0.91,
  );
  path.cubicTo(
    size.width * 1.23,
    size.height * 1.02,
    size.width * 1.11,
    size.height * 1.15,
    size.width,
    size.height * 1.11,
  );
  path.cubicTo(
    size.width * 0.78,
    size.height * 1.04,
    size.width * 0.59,
    size.height * 1.04,
    size.width * 0.37,
    size.height * 1.12,
  );
  path.cubicTo(
    size.width * 0.26,
    size.height * 1.16,
    size.width * 0.14,
    size.height * 1.02,
    size.width / 5,
    size.height * 0.92,
  );
  path.cubicTo(
    size.width / 5,
    size.height * 0.92,
    size.width * 0.59,
    size.height * 0.19,
    size.width * 0.59,
    size.height * 0.19,
  );
  path.close();

  return path;
}
