// Copyright 2013 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'dart:math' as math;

import 'package:flutter_test/flutter_test.dart';
import 'package:vector_graphics_compiler/vector_graphics_compiler.dart';

void main() {
  test('SVG Path tests', () {
    Path path = parseSvgPathData(
        'M22.1595 3.80852C19.6789 1.35254 16.3807 -4.80966e-07 12.8727 '
        '-4.80966e-07C9.36452 -4.80966e-07 6.06642 1.35254 3.58579 3.80852C1.77297 5.60333 '
        '0.53896 7.8599 0.0171889 10.3343C-0.0738999 10.7666 0.206109 11.1901 0.64265 '
        '11.2803C1.07908 11.3706 1.50711 11.0934 1.5982 10.661C2.05552 8.49195 3.13775 6.51338 4.72783 '
        '4.9391C9.21893 0.492838 16.5262 0.492728 21.0173 4.9391C25.5082 9.38548 25.5082 16.6202 '
        '21.0173 21.0667C16.5265 25.5132 9.21893 25.5133 4.72805 21.0669C3.17644 19.5307 2.10538 '
        '17.6035 1.63081 15.4937C1.53386 15.0627 1.10252 14.7908 0.66697 14.887C0.231645 14.983 '
        '-0.0427272 15.4103 0.0542205 15.8413C0.595668 18.2481 1.81686 20.4461 3.5859 '
        '22.1976C6.14623 24.7325 9.50955 26 12.8727 26C16.236 26 19.5991 24.7326 22.1595 22.1976C27.2802 '
        '17.1277 27.2802 8.87841 22.1595 3.80852Z');
    expect(
      path.toFlutterString(),
      'Path()\n'
      '  ..moveTo(22.1595, 3.80852)\n'
      '  ..cubicTo(19.6789, 1.35254, 16.3807, -4.809659999999999e-7, 12.8727, -4.809659999999999e-7)\n'
      '  ..cubicTo(9.36452, -4.809659999999999e-7, 6.06642, 1.35254, 3.5857900000000003, 3.80852)\n'
      '  ..cubicTo(1.77297, 5.60333, 0.53896, 7.8599, 0.017188900000000007, 10.3343)\n'
      '  ..cubicTo(-0.0738999, 10.7666, 0.20610900000000001, 11.1901, 0.6426500000000002, 11.2803)\n'
      '  ..cubicTo(1.07908, 11.3706, 1.50711, 11.0934, 1.5982, 10.661)\n'
      '  ..cubicTo(2.05552, 8.49195, 3.13775, 6.51338, 4.72783, 4.9391)\n'
      '  ..cubicTo(9.21893, 0.49283800000000005, 16.5262, 0.49272800000000005, 21.0173, 4.9391)\n'
      '  ..cubicTo(25.5082, 9.38548, 25.5082, 16.6202, 21.0173, 21.0667)\n'
      '  ..cubicTo(16.5265, 25.5132, 9.21893, 25.5133, 4.72805, 21.0669)\n'
      '  ..cubicTo(3.17644, 19.5307, 2.10538, 17.6035, 1.63081, 15.4937)\n'
      '  ..cubicTo(1.53386, 15.0627, 1.10252, 14.7908, 0.6669700000000002, 14.887)\n'
      '  ..cubicTo(0.23164500000000002, 14.983, -0.04272720000000001, 15.4103, 0.05422050000000001, 15.8413)\n'
      '  ..cubicTo(0.5956680000000001, 18.2481, 1.8168600000000001, 20.4461, 3.5859, 22.1976)\n'
      '  ..cubicTo(6.14623, 24.7325, 9.50955, 26.0, 12.8727, 26.0)\n'
      '  ..cubicTo(16.236, 26.0, 19.5991, 24.7326, 22.1595, 22.1976)\n'
      '  ..cubicTo(27.2802, 17.1277, 27.2802, 8.87841, 22.1595, 3.80852)\n'
      '  ..close();',
    );

    path = parseSvgPathData('M10 10L20 20');

    expect(
      path.toFlutterString(),
      'Path()\n'
      '  ..moveTo(10.0, 10.0)\n'
      '  ..lineTo(20.0, 20.0);',
    );
  });

  test('addRect', () {
    final PathBuilder builder = PathBuilder()
      ..addRect(const Rect.fromLTRB(10, 10, 20, 20));

    expect(
      builder.toPath().toFlutterString(),
      'Path()\n'
      '  ..moveTo(10.0, 10.0)\n'
      '  ..lineTo(20.0, 10.0)\n'
      '  ..lineTo(20.0, 20.0)\n'
      '  ..lineTo(10.0, 20.0)\n'
      '  ..close();',
    );
  });

  test('addOval', () {
    final PathBuilder builder = PathBuilder()
      ..addOval(const Rect.fromLTRB(10, 10, 20, 20))
      ..addOval(const Rect.fromLTRB(50, 50, 80, 70));
    expect(
      builder.toPath().toFlutterString(),
      'Path()\n'
      '  ..moveTo(15.0, 10.0)\n'
      '  ..cubicTo(17.75957512247, 10.0, 20.0, 12.24042487753, 20.0, 15.0)\n'
      '  ..cubicTo(20.0, 17.75957512247, 17.75957512247, 20.0, 15.0, 20.0)\n'
      '  ..cubicTo(12.24042487753, 20.0, 10.0, 17.75957512247, 10.0, 15.0)\n'
      '  ..cubicTo(10.0, 12.24042487753, 12.24042487753, 10.0, 15.0, 10.0)\n'
      '  ..close()\n'
      '  ..moveTo(65.0, 50.0)\n'
      '  ..cubicTo(73.27872536741, 50.0, 80.0, 54.48084975506, 80.0, 60.0)\n'
      '  ..cubicTo(80.0, 65.51915024494, 73.27872536741, 70.0, 65.0, 70.0)\n'
      '  ..cubicTo(56.72127463259, 70.0, 50.0, 65.51915024494, 50.0, 60.0)\n'
      '  ..cubicTo(50.0, 54.48084975506, 56.72127463259, 50.0, 65.0, 50.0)\n'
      '  ..close();',
    );
  });

  test('addRRect', () {
    final PathBuilder builder = PathBuilder()
      ..addRRect(const Rect.fromLTRB(20, 20, 60, 60), 5, 5);
    expect(
      builder.toPath().toFlutterString(),
      'Path()\n'
      '  ..moveTo(25.0, 20.0)\n'
      '  ..lineTo(55.0, 20.0)\n'
      '  ..cubicTo(57.75957512247, 20.0, 60.0, 22.24042487753, 60.0, 25.0)\n'
      '  ..lineTo(60.0, 55.0)\n'
      '  ..cubicTo(60.0, 57.75957512247, 57.75957512247, 60.0, 55.0, 60.0)\n'
      '  ..lineTo(25.0, 60.0)\n'
      '  ..cubicTo(22.24042487753, 60.0, 20.0, 57.75957512247, 20.0, 55.0)\n'
      '  ..lineTo(20.0, 25.0)\n'
      '  ..cubicTo(20.0, 22.24042487753, 22.24042487753, 20.0, 25.0, 20.0)\n'
      '  ..close();',
    );
  });

  test('reset/no reset', () {
    final PathBuilder builder = PathBuilder()..lineTo(10, 10);

    final Path a = builder.toPath(reset: false);
    final Path b = builder.toPath();
    final Path c = builder.toPath();

    expect(a, b);
    expect(identical(a, b), false);
    expect(a != c, true);
    expect(c.isEmpty, true);
  });

  test('PathBuilder.fromPath', () {
    final PathBuilder builder = PathBuilder()..lineTo(10, 10);

    final Path a = builder.toPath();

    final PathBuilder builderA = PathBuilder.fromPath(a);
    final Path b = builderA.toPath();

    expect(a, b);
    expect(identical(a, b), false);
  });

  test('transforms', () {
    Path path = parseSvgPathData(
        'M22.1595 3.80852C19.6789 1.35254 16.3807 -4.80966e-07 12.8727 '
        '-4.80966e-07C9.36452 -4.80966e-07 6.06642 1.35254 3.58579 3.80852C1.77297 5.60333 '
        '0.53896 7.8599 0.0171889 10.3343C-0.0738999 10.7666 0.206109 11.1901 0.64265 '
        '11.2803C1.07908 11.3706 1.50711 11.0934 1.5982 10.661C2.05552 8.49195 3.13775 6.51338 4.72783 '
        '4.9391C9.21893 0.492838 16.5262 0.492728 21.0173 4.9391C25.5082 9.38548 25.5082 16.6202 '
        '21.0173 21.0667C16.5265 25.5132 9.21893 25.5133 4.72805 21.0669C3.17644 19.5307 2.10538 '
        '17.6035 1.63081 15.4937C1.53386 15.0627 1.10252 14.7908 0.66697 14.887C0.231645 14.983 '
        '-0.0427272 15.4103 0.0542205 15.8413C0.595668 18.2481 1.81686 20.4461 3.5859 '
        '22.1976C6.14623 24.7325 9.50955 26 12.8727 26C16.236 26 19.5991 24.7326 22.1595 22.1976C27.2802 '
        '17.1277 27.2802 8.87841 22.1595 3.80852Z');
    expect(
      path.transformed(AffineMatrix.identity).toFlutterString(),
      'Path()\n'
      '  ..moveTo(22.1595, 3.80852)\n'
      '  ..cubicTo(19.6789, 1.35254, 16.3807, -4.809659999999999e-7, 12.8727, -4.809659999999999e-7)\n'
      '  ..cubicTo(9.36452, -4.809659999999999e-7, 6.06642, 1.35254, 3.5857900000000003, 3.80852)\n'
      '  ..cubicTo(1.77297, 5.60333, 0.53896, 7.8599, 0.017188900000000007, 10.3343)\n'
      '  ..cubicTo(-0.0738999, 10.7666, 0.20610900000000001, 11.1901, 0.6426500000000002, 11.2803)\n'
      '  ..cubicTo(1.07908, 11.3706, 1.50711, 11.0934, 1.5982, 10.661)\n'
      '  ..cubicTo(2.05552, 8.49195, 3.13775, 6.51338, 4.72783, 4.9391)\n'
      '  ..cubicTo(9.21893, 0.49283800000000005, 16.5262, 0.49272800000000005, 21.0173, 4.9391)\n'
      '  ..cubicTo(25.5082, 9.38548, 25.5082, 16.6202, 21.0173, 21.0667)\n'
      '  ..cubicTo(16.5265, 25.5132, 9.21893, 25.5133, 4.72805, 21.0669)\n'
      '  ..cubicTo(3.17644, 19.5307, 2.10538, 17.6035, 1.63081, 15.4937)\n'
      '  ..cubicTo(1.53386, 15.0627, 1.10252, 14.7908, 0.6669700000000002, 14.887)\n'
      '  ..cubicTo(0.23164500000000002, 14.983, -0.04272720000000001, 15.4103, 0.05422050000000001, 15.8413)\n'
      '  ..cubicTo(0.5956680000000001, 18.2481, 1.8168600000000001, 20.4461, 3.5859, 22.1976)\n'
      '  ..cubicTo(6.14623, 24.7325, 9.50955, 26.0, 12.8727, 26.0)\n'
      '  ..cubicTo(16.236, 26.0, 19.5991, 24.7326, 22.1595, 22.1976)\n'
      '  ..cubicTo(27.2802, 17.1277, 27.2802, 8.87841, 22.1595, 3.80852)\n'
      '  ..close();',
    );

    expect(
      path
          .transformed(AffineMatrix.identity.rotated(math.pi / 2))
          .toFlutterString(),
      'Path()\n'
      '  ..moveTo(-3.808519999999999, 22.1595)\n'
      '  ..cubicTo(-1.352539999999999, 19.6789, 4.809660010030285e-7, 16.3807, 4.809660007882255e-7, 12.8727)\n'
      '  ..cubicTo(4.809660005734114e-7, 9.36452, -1.3525399999999996, 6.06642, -3.80852, 3.5857900000000007)\n'
      '  ..cubicTo(-5.60333, 1.7729700000000004, -7.8599, 0.5389600000000004, -10.3343, 0.01718890000000064)\n'
      '  ..cubicTo(-10.7666, -0.07389989999999934, -11.1901, 0.2061090000000007, -11.2803, 0.6426500000000008)\n'
      '  ..cubicTo(-11.3706, 1.0790800000000007, -11.0934, 1.5071100000000006, -10.661, 1.5982000000000007)\n'
      '  ..cubicTo(-8.49195, 2.0555200000000005, -6.51338, 3.1377500000000005, -4.9391, 4.72783)\n'
      '  ..cubicTo(-0.4928379999999995, 9.21893, -0.49272799999999906, 16.5262, -4.939099999999999, 21.0173)\n'
      '  ..cubicTo(-9.385479999999998, 25.5082, -16.6202, 25.5082, -21.0667, 21.0173)\n'
      '  ..cubicTo(-25.5132, 16.5265, -25.5133, 9.218930000000002, -21.0669, 4.7280500000000005)\n'
      '  ..cubicTo(-19.5307, 3.1764400000000013, -17.6035, 2.1053800000000007, -15.4937, 1.630810000000001)\n'
      '  ..cubicTo(-15.0627, 1.533860000000001, -14.7908, 1.1025200000000008, -14.887, 0.6669700000000011)\n'
      '  ..cubicTo(-14.983, 0.23164500000000093, -15.4103, -0.04272719999999906, -15.8413, 0.05422050000000098)\n'
      '  ..cubicTo(-18.2481, 0.5956680000000012, -20.4461, 1.8168600000000015, -22.1976, 3.5859000000000014)\n'
      '  ..cubicTo(-24.7325, 6.146230000000002, -26.0, 9.509550000000003, -26.0, 12.872700000000002)\n'
      '  ..cubicTo(-26.0, 16.236, -24.7326, 19.5991, -22.1976, 22.1595)\n'
      '  ..cubicTo(-17.1277, 27.2802, -8.878409999999999, 27.2802, -3.808519999999999, 22.1595)\n'
      '  ..close();',
    );

    path = parseSvgPathData('M10 10L20 20');

    expect(
      path
          .transformed(AffineMatrix.identity.translated(10, 10))
          .toFlutterString(),
      'Path()\n'
      '  ..moveTo(20.0, 20.0)\n'
      '  ..lineTo(30.0, 30.0);',
    );
  });

  test('Compute path bounds with rect', () {
    final PathBuilder builder = PathBuilder()
      ..addRect(const Rect.fromLTWH(5, 5, 95, 95))
      ..close();
    final Path path = builder.toPath();

    expect(path.bounds(), const Rect.fromLTWH(5, 5, 95, 95));
  });

  test('Compute path bounds with lines', () {
    final PathBuilder builder = PathBuilder()
      ..moveTo(0, 0)
      ..lineTo(25, 0)
      ..lineTo(25, 25)
      ..lineTo(0, 25)
      ..close();
    final Path path = builder.toPath();

    expect(path.bounds(), const Rect.fromLTWH(0, 0, 25, 25));
  });

  test('Compute path bounds with cubics', () {
    final PathBuilder builder = PathBuilder()
      ..moveTo(0, 0)
      ..cubicTo(10, 10, 20, 20, -10, -10)
      ..close();
    final Path path = builder.toPath();

    expect(path.bounds(), const Rect.fromLTRB(-10.0, -10.0, 20.0, 20.0));
  });

  test('Compute cubic bounds where R and B are negative', () {
    const Rect circle = Rect.fromCircle(-83.533, -122.753, 74.461);
    final Path path = PathBuilder().addOval(circle).toPath();
    expect(path.bounds(), circle);
  });

  test('Cubic length', () {
    // Value is very close to what Skia says for same input.
    const CubicToCommand command =
        CubicToCommand(1.0, 15.327, 15.326, 1.0, 33.0, 1.0);
    expect(command.computeLength(Point.zero), 38.16245134493276);

    // Trivially describes a line.
    const CubicToCommand command2 = CubicToCommand(0, 0, 0, 10, 0, 10);
    expect(command2.computeLength(Point.zero), 10);
  });

  test('Cubic splitting', () {
    expect(
      CubicToCommand.subdivide(
        Point.zero,
        const Point(1, 15),
        const Point(15, 1),
        const Point(33, 1),
        .4,
      ),
      const <Point>[
        Point.zero,
        Point(0.4, 6.0),
        Point(2.88, 7.359999999999999),
        Point(6.864, 6.832),
        Point(12.84, 6.04),
        Point(22.200000000000003, 1.0),
        Point(33.0, 1.0)
      ],
    );
  });

  test('Dashed path - cubic 1', () {
    final Path cubic1 = parseSvgPathData(
        'M65 33c0 17.673-14.326 32-32 32S1 50.673 1 33C1 15.327 15.326 1 33 1s32 14.327 32 32z');

    expect(
      cubic1.dashed(<double>[2, 5.94]),
      Path(
        commands: const <PathCommand>[
          MoveToCommand(65.0, 33.0),
          CubicToCommand(65.0, 33.70763536030907, 64.97703198598045,
              34.40990628009675, 64.93180068504239, 35.106107839678536),
          MoveToCommand(63.95590821783997, 41.138570321608114),
          CubicToCommand(
              63.78388200211823,
              41.79470101955482,
              63.591548343963595,
              42.442608960532986,
              63.37961933238859,
              43.08158186093479),
          MoveToCommand(60.9913809985287, 48.519833069956164),
          CubicToCommand(
              60.668492690302394,
              49.100962858508964,
              60.32803245363184,
              49.6709842212665,
              59.97071797737027,
              50.22917927324997),
          MoveToCommand(56.349259058467936, 54.88222449322346),
          CubicToCommand(
              55.900015721696754,
              55.36139696181796,
              55.43619093879707,
              55.82673409716467,
              54.958492499076485,
              56.27752791653569),
          MoveToCommand(50.36747808861274, 59.88172317266531),
          CubicToCommand(
              49.82068927796797,
              60.23570903270937,
              49.26241388099913,
              60.57346570952925,
              48.6933336310888,
              60.894311283442455),
          MoveToCommand(43.3292530683581, 63.29637356816345),
          CubicToCommand(
              42.71018120276065,
              63.50737099626945,
              42.082572102791204,
              63.69992743387707,
              41.447080444962786,
              63.87338802556734),
          MoveToCommand(35.630945421656286, 64.89340586107826),
          CubicToCommand(
              34.97191375239566,
              64.94703220278787,
              34.30723211065745,
              64.98069119320479,
              33.637506242045006,
              64.99377692119126),
          MoveToCommand(27.497686697670588, 64.52877424012252),
          CubicToCommand(
              26.821431561562925,
              64.41158787290307,
              26.152455351357716,
              64.27317574141253,
              25.49146906822908,
              64.11424904112303),
          MoveToCommand(19.829734138714148, 62.17288407887853),
          CubicToCommand(
              19.22254029238493,
              61.89834087441918,
              18.625447859490016,
              61.60539834563903,
              18.03916772219419,
              61.29476756896835),
          MoveToCommand(13.123453402340228, 58.08021292143201),
          CubicToCommand(
              12.606563474017639,
              57.67001905040603,
              12.102648327670325,
              57.244183807034666,
              11.612423197029361,
              56.80342262050364),
          MoveToCommand(7.634684007455509, 52.51189122452449),
          CubicToCommand(
              7.236391480757161,
              51.99487385892312,
              6.853553118859099,
              51.46533727229267,
              6.486860221302153,
              50.923972953088025),
          MoveToCommand(3.672998384894739, 45.82384400992106),
          CubicToCommand(
              3.410525278026412,
              45.22442217885262,
              3.1658907799562406,
              44.61540918759221,
              2.9397661639663557,
              43.997476492863385),
          MoveToCommand(1.4383854599561599, 38.30920094462239),
          CubicToCommand(
              1.3301369848492222,
              37.660898732125794,
              1.2413788962838397,
              37.00599786367147,
              1.1727338823578792,
              36.34512119752162),
          MoveToCommand(1.1140602215592654, 30.278918052611246),
          CubicToCommand(
              1.1724789949662897,
              29.58512444526761,
              1.2530318675342023,
              28.897630742351822,
              1.355010329306275,
              28.21714564743783),
          MoveToCommand(2.813895354454088, 22.352839675947116),
          CubicToCommand(
              3.037366664760353,
              21.719242963796532,
              3.2802211823279825,
              21.094791779246986,
              3.5417468824172644,
              20.48019834161591),
          MoveToCommand(6.34621623119101, 15.285435280579309),
          CubicToCommand(
              6.714032665445101,
              14.73310174588461,
              7.098560403588749,
              14.192851911447924,
              7.499077458587648,
              13.665407961603714),
          MoveToCommand(11.461835455942882, 9.332832232296191),
          CubicToCommand(
              11.945763342108913,
              8.892180194722316,
              12.44325896152358,
              8.466169418310185,
              12.953622282985162,
              8.055500125561714),
          MoveToCommand(17.807192192563488, 4.829368065777091),
          CubicToCommand(
              18.384146248584546,
              4.517560722634985,
              18.971766599642926,
              4.22296045700293,
              19.5693656981362,
              3.946255004372108),
          MoveToCommand(25.105444010192507, 1.9811251891028818),
          CubicToCommand(
              25.74028000900354,
              1.8200561533454764,
              26.382686089997996,
              1.6779627249499476,
              27.032023265982627,
              1.5554840657280102),
          MoveToCommand(32.94511841345392, 1.0000460546288212),
          CubicToCommand(32.96340867914606, 1.000015355604456,
              32.981702545388494, 1.0, 33.0, 1.0),
          CubicToCommand(33.6882562195718, 1.0, 34.37143533540402,
              1.0217263038461621, 35.04888905781622, 1.0645304446974477),
          MoveToCommand(41.084875326577645, 2.030039591233501),
          CubicToCommand(
              41.74147663234361,
              2.200961034160867,
              42.38987221052307,
              2.3922104375592332,
              43.029349965306906,
              2.6030755110223245),
          MoveToCommand(48.472612206885536, 4.982395187695522),
          CubicToCommand(
              49.05438116106162,
              5.304353344751444,
              49.62505841296779,
              5.643905011918012,
              50.18392615788486,
              6.000332188318815),
          MoveToCommand(54.84351930781497, 9.614505510093846),
          CubicToCommand(
              55.323479280858734,
              10.063008836851036,
              55.78961965045844,
              10.526114893814489,
              56.24123242758923,
              11.00311549848448),
          MoveToCommand(59.853140416091875, 15.58850419596532),
          CubicToCommand(
              60.20803756757811,
              16.134740492295958,
              60.54671980770702,
              16.69248333095327,
              60.868505145713,
              17.261050534801427),
          MoveToCommand(63.27947749389385, 22.621402004854463),
          CubicToCommand(
              63.49149442978085,
              23.240129771843495,
              63.68508101683114,
              23.867413675066253,
              63.859582179442924,
              24.50259845990592),
          MoveToCommand(64.88908365060104, 30.31651883115104),
          CubicToCommand(
              64.9437829347059,
              30.975372322751923,
              64.97852291408398,
              31.639891900169225,
              64.99269741829195,
              32.309471227309054),
        ],
      ),
    );
  });

  test('Dashed paths - cubic 2', () {
    final Path cubic2 = parseSvgPathData(
        'M20 39c10.493 0 19-8.507 19-19S30.493 1 20 1 1 9.507 1 20s8.507 19 19 19z');

    final Path dashed = cubic2.dashed(<double>[2, 6]);
    expect(
      dashed,
      Path(
        commands: const <PathCommand>[
          MoveToCommand(20.0, 39.0),
          CubicToCommand(20.707618776439247, 39.0, 21.4062056638081,
              38.96131204414812, 22.093760739330435, 38.88593605522049),
          MoveToCommand(27.941787133378586, 37.26550076491443),
          CubicToCommand(
              28.552828792067846,
              36.98395247399771,
              29.146376269127835,
              36.67092015975091,
              29.72035001637582,
              36.32848337035675),
          MoveToCommand(34.28872657525249, 32.52332807494088),
          CubicToCommand(
              34.72002359056755,
              32.03163198310322,
              35.12634784297375,
              31.51748652496024,
              35.50571809622244,
              30.9828729367606),
          MoveToCommand(38.15063297069716, 25.634183986854904),
          CubicToCommand(
              38.344405068145036,
              25.00930101075243,
              38.506794068808375,
              24.3706162148244,
              38.63595727148749,
              23.71997230027049),
          MoveToCommand(38.854396196856676, 17.636338403239797),
          CubicToCommand(
              38.76922952234208,
              16.950072134496082,
              38.647462472510895,
              16.275154057296998,
              38.49110936584479,
              15.613598490124222),
          MoveToCommand(36.156993272025055, 9.997799228160478),
          CubicToCommand(
              35.806275376209854,
              9.432491207486724,
              35.4267166755551,
              8.886947549472037,
              35.02037300554141,
              8.363224089597036),
          MoveToCommand(30.714494314385455, 4.307279443052996),
          CubicToCommand(
              30.170446598805572,
              3.9350931692753632,
              29.606060167556546,
              3.5904613343389835,
              29.023345848321203,
              3.2753947659266824),
          MoveToCommand(23.398670360690218, 1.303142250295933),
          CubicToCommand(
              22.75231244012423,
              1.1864084956688012,
              22.094586719261592,
              1.102371988982065,
              21.42726782093816,
              1.0528073530715703),
          MoveToCommand(15.312878715409806, 1.5825736677487203),
          CubicToCommand(
              14.645620930686587,
              1.7518831987472228,
              13.992442800092594,
              1.956503341694209,
              13.35545017026537,
              2.1943282499521333),
          MoveToCommand(8.126992325034307, 5.1659142014116854),
          CubicToCommand(7.612222375568647, 5.578470802327974, 7.11938698608979,
              6.017265988457743, 6.6505146434356375, 6.480271272963089),
          MoveToCommand(3.144680725594728, 11.22271551538195),
          CubicToCommand(
              2.841419027240009,
              11.803860948679425,
              2.5673312422713543,
              12.402645469100445,
              2.324360556848317,
              13.017125890485456),
          MoveToCommand(1.0338820704843192, 18.855919027319942),
          CubicToCommand(1.0114006643291984, 19.23440453004496, 1.0,
              19.61587149510607, 1.0, 20.0),
          CubicToCommand(1.0, 20.30265471646772, 1.0070773712819083,
              20.603657186624655, 1.0210756343203329, 20.902850930945412),
          MoveToCommand(2.285459600751064, 26.883674148689185),
          CubicToCommand(
              2.530950502455319,
              27.514930621208894,
              2.809211392982629,
              28.12982234128287,
              3.118152746709571,
              28.7262597832877),
          MoveToCommand(6.658084504103845, 33.527199707900266),
          CubicToCommand(
              7.124854343878676,
              33.98761626048461,
              7.615335906786609,
              34.424054910109746,
              8.127531169442356,
              34.8345176333904),
          MoveToCommand(13.322396345361739, 37.79329566146798),
          CubicToCommand(
              13.93468962982282,
              38.02320534530446,
              14.561997661390594,
              38.22241500364579,
              15.202445231678627,
              38.38904942810554),
        ],
      ),
    );
  });

  test('Dashed path - lines/closes', () {
    final Path path = parseSvgPathData('M1 20L20 20L20 39L30 30L1 26z');
    expect(
      path.dashed(<double>[5, 3, 5, 5]),
      Path(
        commands: const <PathCommand>[
          MoveToCommand(1.0, 20.0),
          LineToCommand(6.0, 20.0),
          MoveToCommand(9.0, 20.0),
          LineToCommand(13.999999999999998, 20.0),
          MoveToCommand(18.999999999999996, 20.0),
          LineToCommand(20.0, 20.0),
          LineToCommand(20.0, 24.0),
          MoveToCommand(20.0, 27.000000000000004),
          LineToCommand(20.0, 32.0),
          MoveToCommand(20.0, 37.0),
          LineToCommand(20.0, 39.0),
          LineToCommand(22.229882438741498, 36.99310580513265),
          MoveToCommand(24.459764877482996, 34.9862116102653),
          LineToCommand(28.17623560871883, 31.641387952153053),
          MoveToCommand(27.47750617803373, 29.65206981765983),
          LineToCommand(22.524400531816358, 28.96888283197467),
          MoveToCommand(19.55253714408593, 28.55897064056358),
          LineToCommand(14.599431497868558, 27.875783654878425),
          MoveToCommand(9.646325851651186, 27.19259666919327),
          LineToCommand(4.693220205433812, 26.509409683508114),
          MoveToCommand(1.7213568177033882, 26.09949749209702),
          LineToCommand(1.0, 26.0),
          LineToCommand(1.0, 21.72818638368261),
        ],
      ),
    );
  });
}
