package com.oodrive.downloadsfolder

import android.webkit.MimeTypeMap


/**
 * Gets the MIME type of a file based on its extension.
 *
 * @param extension The file extension (e.g., "pdf", "jpg").
 * @return The MIME type of the file, or null if the extension is not recognized.
 */
fun getMimeTypeFromExtension(extension: String?): String? {
    return if (extension != null) {
        // Check if the extension is not null.

        // Use MimeTypeMap to get the MIME type based on the extension.
        // Convert the extension to lowercase and remove any leading dots.
        val lowercasedExtension = extension.lowercase().replace(".", "")
        val mimeType = MimeTypeMap.getSingleton().getMimeTypeFromExtension(lowercasedExtension)

        // If the MIME type is found, return it. Otherwise, use a custom mapping.
        mimeType ?: mimeTypes[lowercasedExtension]
    } else {
        // If the extension is null, return null.
        null
    }
}


// Define a custom mapping of extensions to MIME types for cases where MimeTypeMap.getSingleton() doesn't provide a match.
private val mimeTypes = mapOf(
    "x3d" to "application/vnd.hzn-3d-crossword",
    "3gp" to "video/3gpp",
    "3g2" to "video/3gpp2",
    "mseq" to "application/vnd.mseq",
    "pwn" to "application/vnd.3m.post-it-notes",
    "plb" to "application/vnd.3gpp.pic-bw-large",
    "psb" to "application/vnd.3gpp.pic-bw-small",
    "pvb" to "application/vnd.3gpp.pic-bw-var",
    "tcap" to "application/vnd.3gpp2.tcap",
    "7z" to "application/x-7z-compressed",
    "abw" to "application/x-abiword",
    "ace" to "application/x-ace-compressed",
    "acc" to "application/vnd.americandynamics.acc",
    "acu" to "application/vnd.acucobol",
    "atc" to "application/vnd.acucorp",
    "adp" to "audio/adpcm",
    "aab" to "application/x-authorware-bin",
    "aam" to "application/x-authorware-map",
    "aas" to "application/x-authorware-seg",
    "air" to "application/vnd.adobe.air-application-installer-package+zip",
    "swf" to "application/x-shockwave-flash",
    "fxp" to "application/vnd.adobe.fxp",
    "pdf" to "application/pdf",
    "ppd" to "application/vnd.cups-ppd",
    "dir" to "application/x-director",
    "xdp" to "application/vnd.adobe.xdp+xml",
    "xfdf" to "application/vnd.adobe.xfdf",
    "aac" to "audio/x-aac",
    "ahead" to "application/vnd.ahead.space",
    "azf" to "application/vnd.airzip.filesecure.azf",
    "azs" to "application/vnd.airzip.filesecure.azs",
    "azw" to "application/vnd.amazon.ebook",
    "ami" to "application/vnd.amiga.ami",
    "N/A" to "application/andrew-inset",
    "apk" to "application/vnd.android.package-archive",
    "cii" to "application/vnd.anser-web-certificate-issue-initiation",
    "fti" to "application/vnd.anser-web-funds-transfer-initiation",
    "atx" to "application/vnd.antix.game-component",
    "dmg" to "application/x-apple-diskimage",
    "mpkg" to "application/vnd.apple.installer+xml",
    "aw" to "application/applixware",
    "mp3" to "audio/mpeg",
    "les" to "application/vnd.hhe.lesson-player",
    "swi" to "application/vnd.aristanetworks.swi",
    "s" to "text/x-asm",
    "atomcat" to "application/atomcat+xml",
    "atomsvc" to "application/atomsvc+xml",
    "atom" to "application/atom+xml",
    "ac" to "application/pkix-attr-cert",
    "aif" to "audio/x-aiff",
    "avi" to "video/x-msvideo",
    "aep" to "application/vnd.audiograph",
    "dxf" to "image/vnd.dxf",
    "dwf" to "model/vnd.dwf",
    "par" to "text/plain-bas",
    "bcpio" to "application/x-bcpio",
    "bin" to "application/octet-stream",
    "bmp" to "image/bmp",
    "torrent" to "application/x-bittorrent",
    "cod" to "application/vnd.rim.cod",
    "mpm" to "application/vnd.blueice.multipass",
    "bmi" to "application/vnd.bmi",
    "sh" to "application/x-sh",
    "btif" to "image/prs.btif",
    "rep" to "application/vnd.businessobjects",
    "bz" to "application/x-bzip",
    "bz2" to "application/x-bzip2",
    "csh" to "application/x-csh",
    "c" to "text/x-c",
    "cdxml" to "application/vnd.chemdraw+xml",
    "css" to "text/css",
    "cdx" to "chemical/x-cdx",
    "cml" to "chemical/x-cml",
    "csml" to "chemical/x-csml",
    "cdbcmsg" to "application/vnd.contact.cmsg",
    "cla" to "application/vnd.claymore",
    "c4g" to "application/vnd.clonk.c4group",
    "sub" to "image/vnd.dvb.subtitle",
    "cdmia" to "application/cdmi-capability",
    "cdmic" to "application/cdmi-container",
    "cdmid" to "application/cdmi-domain",
    "cdmio" to "application/cdmi-object",
    "cdmiq" to "application/cdmi-queue",
    "c11amc" to "application/vnd.cluetrust.cartomobile-config",
    "c11amz" to "application/vnd.cluetrust.cartomobile-config-pkg",
    "ras" to "image/x-cmu-raster",
    "dae" to "model/vnd.collada+xml",
    "csv" to "text/csv",
    "cpt" to "application/mac-compactpro",
    "wmlc" to "application/vnd.wap.wmlc",
    "cgm" to "image/cgm",
    "ice" to "x-conference/x-cooltalk",
    "cmx" to "image/x-cmx",
    "xar" to "application/vnd.xara",
    "cmc" to "application/vnd.cosmocaller",
    "cpio" to "application/x-cpio",
    "clkx" to "application/vnd.crick.clicker",
    "clkk" to "application/vnd.crick.clicker.keyboard",
    "clkp" to "application/vnd.crick.clicker.palette",
    "clkt" to "application/vnd.crick.clicker.template",
    "clkw" to "application/vnd.crick.clicker.wordbank",
    "wbs" to "application/vnd.criticaltools.wbs+xml",
    "cryptonote" to "application/vnd.rig.cryptonote",
    "cif" to "chemical/x-cif",
    "cmdf" to "chemical/x-cmdf",
    "cu" to "application/cu-seeme",
    "cww" to "application/prs.cww",
    "curl" to "text/vnd.curl",
    "dcurl" to "text/vnd.curl.dcurl",
    "mcurl" to "text/vnd.curl.mcurl",
    "scurl" to "text/vnd.curl.scurl",
    "car" to "application/vnd.curl.car",
    "pcurl" to "application/vnd.curl.pcurl",
    "cmp" to "application/vnd.yellowriver-custom-menu",
    "dssc" to "application/dssc+der",
    "xdssc" to "application/dssc+xml",
    "deb" to "application/x-debian-package",
    "uva" to "audio/vnd.dece.audio",
    "uvi" to "image/vnd.dece.graphic",
    "uvh" to "video/vnd.dece.hd",
    "uvm" to "video/vnd.dece.mobile",
    "uvu" to "video/vnd.uvvu.mp4",
    "uvp" to "video/vnd.dece.pd",
    "uvs" to "video/vnd.dece.sd",
    "uvv" to "video/vnd.dece.video",
    "dvi" to "application/x-dvi",
    "seed" to "application/vnd.fdsn.seed",
    "dtb" to "application/x-dtbook+xml",
    "res" to "application/x-dtbresource+xml",
    "ait" to "application/vnd.dvb.ait",
    "svc" to "application/vnd.dvb.service",
    "eol" to "audio/vnd.digital-winds",
    "djvu" to "image/vnd.djvu",
    "dtd" to "application/xml-dtd",
    "mlp" to "application/vnd.dolby.mlp",
    "wad" to "application/x-doom",
    "dpg" to "application/vnd.dpgraph",
    "dra" to "audio/vnd.dra",
    "dfac" to "application/vnd.dreamfactory",
    "dts" to "audio/vnd.dts",
    "dtshd" to "audio/vnd.dts.hd",
    "dwg" to "image/vnd.dwg",
    "geo" to "application/vnd.dynageo",
    "es" to "application/ecmascript",
    "mag" to "application/vnd.ecowin.chart",
    "mmr" to "image/vnd.fujixerox.edmics-mmr",
    "rlc" to "image/vnd.fujixerox.edmics-rlc",
    "exi" to "application/exi",
    "mgz" to "application/vnd.proteus.magazine",
    "epub" to "application/epub+zip",
    "eml" to "message/rfc822",
    "nml" to "application/vnd.enliven",
    "xpr" to "application/vnd.is-xpr",
    "xif" to "image/vnd.xiff",
    "xfdl" to "application/vnd.xfdl",
    "emma" to "application/emma+xml",
    "ez2" to "application/vnd.ezpix-album",
    "ez3" to "application/vnd.ezpix-package",
    "fst" to "image/vnd.fst",
    "fvt" to "video/vnd.fvt",
    "fbs" to "image/vnd.fastbidsheet",
    "fe_launch" to "application/vnd.denovo.fcselayout-link",
    "f4v" to "video/x-f4v",
    "flv" to "video/x-flv",
    "fpx" to "image/vnd.fpx",
    "npx" to "image/vnd.net-fpx",
    "flx" to "text/vnd.fmi.flexstor",
    "fli" to "video/x-fli",
    "ftc" to "application/vnd.fluxtime.clip",
    "fdf" to "application/vnd.fdf",
    "f" to "text/x-fortran",
    "mif" to "application/vnd.mif",
    "fm" to "application/vnd.framemaker",
    "fh" to "image/x-freehand",
    "fsc" to "application/vnd.fsc.weblaunch",
    "fnc" to "application/vnd.frogans.fnc",
    "ltf" to "application/vnd.frogans.ltf",
    "ddd" to "application/vnd.fujixerox.ddd",
    "xdw" to "application/vnd.fujixerox.docuworks",
    "xbd" to "application/vnd.fujixerox.docuworks.binder",
    "oas" to "application/vnd.fujitsu.oasys",
    "oa2" to "application/vnd.fujitsu.oasys2",
    "oa3" to "application/vnd.fujitsu.oasys3",
    "fg5" to "application/vnd.fujitsu.oasysgp",
    "bh2" to "application/vnd.fujitsu.oasysprs",
    "spl" to "application/x-futuresplash",
    "fzs" to "application/vnd.fuzzysheet",
    "g3" to "image/g3fax",
    "gmx" to "application/vnd.gmx",
    "gtw" to "model/vnd.gtw",
    "txd" to "application/vnd.genomatix.tuxedo",
    "ggb" to "application/vnd.geogebra.file",
    "ggt" to "application/vnd.geogebra.tool",
    "gdl" to "model/vnd.gdl",
    "gex" to "application/vnd.geometry-explorer",
    "gxt" to "application/vnd.geonext",
    "g2w" to "application/vnd.geoplan",
    "g3w" to "application/vnd.geospace",
    "gsf" to "application/x-font-ghostscript",
    "bdf" to "application/x-font-bdf",
    "gtar" to "application/x-gtar",
    "texinfo" to "application/x-texinfo",
    "gnumeric" to "application/x-gnumeric",
    "kml" to "application/vnd.google-earth.kml+xml",
    "kmz" to "application/vnd.google-earth.kmz",
    "gqf" to "application/vnd.grafeq",
    "gif" to "image/gif",
    "gv" to "text/vnd.graphviz",
    "gac" to "application/vnd.groove-account",
    "ghf" to "application/vnd.groove-help",
    "gim" to "application/vnd.groove-identity-message",
    "grv" to "application/vnd.groove-injector",
    "gtm" to "application/vnd.groove-tool-message",
    "tpl" to "application/vnd.groove-tool-template",
    "vcg" to "application/vnd.groove-vcard",
    "h261" to "video/h261",
    "h263" to "video/h263",
    "h264" to "video/h264",
    "hpid" to "application/vnd.hp-hpid",
    "hps" to "application/vnd.hp-hps",
    "hdf" to "application/x-hdf",
    "rip" to "audio/vnd.rip",
    "hbci" to "application/vnd.hbci",
    "jlt" to "application/vnd.hp-jlyt",
    "pcl" to "application/vnd.hp-pcl",
    "hpgl" to "application/vnd.hp-hpgl",
    "hvs" to "application/vnd.yamaha.hv-script",
    "hvd" to "application/vnd.yamaha.hv-dic",
    "hvp" to "application/vnd.yamaha.hv-voice",
    "sfd-hdstx" to "application/vnd.hydrostatix.sof-data",
    "stk" to "application/hyperstudio",
    "hal" to "application/vnd.hal+xml",
    "html" to "text/html",
    "irm" to "application/vnd.ibm.rights-management",
    "sc" to "application/vnd.ibm.secure-container",
    "ics" to "text/calendar",
    "icc" to "application/vnd.iccprofile",
    "ico" to "image/x-icon",
    "igl" to "application/vnd.igloader",
    "ief" to "image/ief",
    "ivp" to "application/vnd.immervision-ivp",
    "ivu" to "application/vnd.immervision-ivu",
    "rif" to "application/reginfo+xml",
    "3dml" to "text/vnd.in3d.3dml",
    "spot" to "text/vnd.in3d.spot",
    "igs" to "model/iges",
    "i2g" to "application/vnd.intergeo",
    "cdy" to "application/vnd.cinderella",
    "xpw" to "application/vnd.intercon.formnet",
    "fcs" to "application/vnd.isac.fcs",
    "ipfix" to "application/ipfix",
    "cer" to "application/pkix-cert",
    "pki" to "application/pkixcmp",
    "crl" to "application/pkix-crl",
    "pkipath" to "application/pkix-pkipath",
    "igm" to "application/vnd.insors.igm",
    "rcprofile" to "application/vnd.ipunplugged.rcprofile",
    "irp" to "application/vnd.irepository.package+xml",
    "jad" to "text/vnd.sun.j2me.app-descriptor",
    "jar" to "application/java-archive",
    "class" to "application/java-vm",
    "jnlp" to "application/x-java-jnlp-file",
    "ser" to "application/java-serialized-object",
    "java" to "text/x-java-source,java",
    "js" to "application/javascript",
    "json" to "application/json",
    "joda" to "application/vnd.joost.joda-archive",
    "jpm" to "video/jpm",
    "jpeg" to "image/x-citrix-jpeg",
    "jpg" to "image/x-citrix-jpeg",
    "pjpeg" to "image/pjpeg",
    "jpgv" to "video/jpeg",
    "ktz" to "application/vnd.kahootz",
    "mmd" to "application/vnd.chipnuts.karaoke-mmd",
    "karbon" to "application/vnd.kde.karbon",
    "chrt" to "application/vnd.kde.kchart",
    "kfo" to "application/vnd.kde.kformula",
    "flw" to "application/vnd.kde.kivio",
    "kon" to "application/vnd.kde.kontour",
    "kpr" to "application/vnd.kde.kpresenter",
    "ksp" to "application/vnd.kde.kspread",
    "kwd" to "application/vnd.kde.kword",
    "htke" to "application/vnd.kenameaapp",
    "kia" to "application/vnd.kidspiration",
    "kne" to "application/vnd.kinar",
    "sse" to "application/vnd.kodak-descriptor",
    "lasxml" to "application/vnd.las.las+xml",
    "latex" to "application/x-latex",
    "lbd" to "application/vnd.llamagraphics.life-balance.desktop",
    "lbe" to "application/vnd.llamagraphics.life-balance.exchange+xml",
    "jam" to "application/vnd.jam",
    "123" to "application/vnd.lotus-1-2-3",
    "apr" to "application/vnd.lotus-approach",
    "pre" to "application/vnd.lotus-freelance",
    "nsf" to "application/vnd.lotus-notes",
    "org" to "application/vnd.lotus-organizer",
    "scm" to "application/vnd.lotus-screencam",
    "lwp" to "application/vnd.lotus-wordpro",
    "lvp" to "audio/vnd.lucent.voice",
    "m3u" to "audio/x-mpegurl",
    "m4v" to "video/x-m4v",
    "hqx" to "application/mac-binhex40",
    "portpkg" to "application/vnd.macports.portpkg",
    "mgp" to "application/vnd.osgeo.mapguide.package",
    "mrc" to "application/marc",
    "mrcx" to "application/marcxml+xml",
    "mxf" to "application/mxf",
    "nbp" to "application/vnd.wolfram.player",
    "ma" to "application/mathematica",
    "mathml" to "application/mathml+xml",
    "mbox" to "application/mbox",
    "mc1" to "application/vnd.medcalcdata",
    "mscml" to "application/mediaservercontrol+xml",
    "cdkey" to "application/vnd.mediastation.cdkey",
    "mwf" to "application/vnd.mfer",
    "mfm" to "application/vnd.mfmp",
    "msh" to "model/mesh",
    "mads" to "application/mads+xml",
    "mets" to "application/mets+xml",
    "mods" to "application/mods+xml",
    "meta4" to "application/metalink4+xml",
    "mcd" to "application/vnd.mcd",
    "flo" to "application/vnd.micrografx.flo",
    "igx" to "application/vnd.micrografx.igx",
    "es3" to "application/vnd.eszigno3+xml",
    "mdb" to "application/x-msaccess",
    "asf" to "video/x-ms-asf",
    "exe" to "application/x-msdownload",
    "cil" to "application/vnd.ms-artgalry",
    "cab" to "application/vnd.ms-cab-compressed",
    "ims" to "application/vnd.ms-ims",
    "application" to "application/x-ms-application",
    "clp" to "application/x-msclip",
    "mdi" to "image/vnd.ms-modi",
    "eot" to "application/vnd.ms-fontobject",
    "xls" to "application/vnd.ms-excel",
    "xlam" to "application/vnd.ms-excel.addin.macroenabled.12",
    "xlsb" to "application/vnd.ms-excel.sheet.binary.macroenabled.12",
    "xltm" to "application/vnd.ms-excel.template.macroenabled.12",
    "xlsm" to "application/vnd.ms-excel.sheet.macroenabled.12",
    "chm" to "application/vnd.ms-htmlhelp",
    "crd" to "application/x-mscardfile",
    "lrm" to "application/vnd.ms-lrm",
    "mvb" to "application/x-msmediaview",
    "mny" to "application/x-msmoney",
    "pptx" to "application/vnd.openxmlformats-officedocument.presentationml.presentation",
    "sldx" to "application/vnd.openxmlformats-officedocument.presentationml.slide",
    "ppsx" to "application/vnd.openxmlformats-officedocument.presentationml.slideshow",
    "potx" to "application/vnd.openxmlformats-officedocument.presentationml.template",
    "xlsx" to "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
    "xltx" to "application/vnd.openxmlformats-officedocument.spreadsheetml.template",
    "docx" to "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
    "dotx" to "application/vnd.openxmlformats-officedocument.wordprocessingml.template",
    "obd" to "application/x-msbinder",
    "thmx" to "application/vnd.ms-officetheme",
    "onetoc" to "application/onenote",
    "pya" to "audio/vnd.ms-playready.media.pya",
    "pyv" to "video/vnd.ms-playready.media.pyv",
    "ppt" to "application/vnd.ms-powerpoint",
    "ppam" to "application/vnd.ms-powerpoint.addin.macroenabled.12",
    "sldm" to "application/vnd.ms-powerpoint.slide.macroenabled.12",
    "pptm" to "application/vnd.ms-powerpoint.presentation.macroenabled.12",
    "ppsm" to "application/vnd.ms-powerpoint.slideshow.macroenabled.12",
    "potm" to "application/vnd.ms-powerpoint.template.macroenabled.12",
    "mpp" to "application/vnd.ms-project",
    "pub" to "application/x-mspublisher",
    "scd" to "application/x-msschedule",
    "xap" to "application/x-silverlight-app",
    "stl" to "application/vnd.ms-pki.stl",
    "cat" to "application/vnd.ms-pki.seccat",
    "vsd" to "application/vnd.visio",
    "vsdx" to "application/vnd.visio2013",
    "wm" to "video/x-ms-wm",
    "wma" to "audio/x-ms-wma",
    "wax" to "audio/x-ms-wax",
    "wmx" to "video/x-ms-wmx",
    "wmd" to "application/x-ms-wmd",
    "wpl" to "application/vnd.ms-wpl",
    "wmz" to "application/x-ms-wmz",
    "wmv" to "video/x-ms-wmv",
    "wvx" to "video/x-ms-wvx",
    "wmf" to "application/x-msmetafile",
    "trm" to "application/x-msterminal",
    "doc" to "application/msword",
    "docm" to "application/vnd.ms-word.document.macroenabled.12",
    "dotm" to "application/vnd.ms-word.template.macroenabled.12",
    "wri" to "application/x-mswrite",
    "wps" to "application/vnd.ms-works",
    "xbap" to "application/x-ms-xbap",
    "xps" to "application/vnd.ms-xpsdocument",
    "mid" to "audio/midi",
    "mpy" to "application/vnd.ibm.minipay",
    "afp" to "application/vnd.ibm.modcap",
    "rms" to "application/vnd.jcp.javame.midlet-rms",
    "tmo" to "application/vnd.tmobile-livetv",
    "prc" to "application/x-mobipocket-ebook",
    "mbk" to "application/vnd.mobius.mbk",
    "dis" to "application/vnd.mobius.dis",
    "plc" to "application/vnd.mobius.plc",
    "mqy" to "application/vnd.mobius.mqy",
    "msl" to "application/vnd.mobius.msl",
    "txf" to "application/vnd.mobius.txf",
    "daf" to "application/vnd.mobius.daf",
    "fly" to "text/vnd.fly",
    "mpc" to "application/vnd.mophun.certificate",
    "mpn" to "application/vnd.mophun.application",
    "mj2" to "video/mj2",
    "mpga" to "audio/mpeg",
    "mxu" to "video/vnd.mpegurl",
    "mpeg" to "video/mpeg",
    "m21" to "application/mp21",
    "mp4a" to "audio/mp4",
    "mp4" to "video/mp4",
    "m3u8" to "application/vnd.apple.mpegurl",
    "mus" to "application/vnd.musician",
    "msty" to "application/vnd.muvee.style",
    "mxml" to "application/xv+xml",
    "ngdat" to "application/vnd.nokia.n-gage.data",
    "n-gage" to "application/vnd.nokia.n-gage.symbian.install",
    "ncx" to "application/x-dtbncx+xml",
    "nc" to "application/x-netcdf",
    "nlu" to "application/vnd.neurolanguage.nlu",
    "dna" to "application/vnd.dna",
    "nnd" to "application/vnd.noblenet-directory",
    "nns" to "application/vnd.noblenet-sealer",
    "nnw" to "application/vnd.noblenet-web",
    "rpst" to "application/vnd.nokia.radio-preset",
    "rpss" to "application/vnd.nokia.radio-presets",
    "n3" to "text/n3",
    "edm" to "application/vnd.novadigm.edm",
    "edx" to "application/vnd.novadigm.edx",
    "ext" to "application/vnd.novadigm.ext",
    "gph" to "application/vnd.flographit",
    "ecelp4800" to "audio/vnd.nuera.ecelp4800",
    "ecelp7470" to "audio/vnd.nuera.ecelp7470",
    "ecelp9600" to "audio/vnd.nuera.ecelp9600",
    "oda" to "application/oda",
    "ogx" to "application/ogg",
    "oga" to "audio/ogg",
    "ogv" to "video/ogg",
    "dd2" to "application/vnd.oma.dd2+xml",
    "oth" to "application/vnd.oasis.opendocument.text-web",
    "opf" to "application/oebps-package+xml",
    "qbo" to "application/vnd.intu.qbo",
    "oxt" to "application/vnd.openofficeorg.extension",
    "osf" to "application/vnd.yamaha.openscoreformat",
    "weba" to "audio/webm",
    "webm" to "video/webm",
    "odc" to "application/vnd.oasis.opendocument.chart",
    "otc" to "application/vnd.oasis.opendocument.chart-template",
    "odb" to "application/vnd.oasis.opendocument.database",
    "odf" to "application/vnd.oasis.opendocument.formula",
    "odft" to "application/vnd.oasis.opendocument.formula-template",
    "odg" to "application/vnd.oasis.opendocument.graphics",
    "otg" to "application/vnd.oasis.opendocument.graphics-template",
    "odi" to "application/vnd.oasis.opendocument.image",
    "oti" to "application/vnd.oasis.opendocument.image-template",
    "odp" to "application/vnd.oasis.opendocument.presentation",
    "otp" to "application/vnd.oasis.opendocument.presentation-template",
    "ods" to "application/vnd.oasis.opendocument.spreadsheet",
    "ots" to "application/vnd.oasis.opendocument.spreadsheet-template",
    "odt" to "application/vnd.oasis.opendocument.text",
    "odm" to "application/vnd.oasis.opendocument.text-master",
    "ott" to "application/vnd.oasis.opendocument.text-template",
    "ktx" to "image/ktx",
    "sxc" to "application/vnd.sun.xml.calc",
    "stc" to "application/vnd.sun.xml.calc.template",
    "sxd" to "application/vnd.sun.xml.draw",
    "std" to "application/vnd.sun.xml.draw.template",
    "sxi" to "application/vnd.sun.xml.impress",
    "sti" to "application/vnd.sun.xml.impress.template",
    "sxm" to "application/vnd.sun.xml.math",
    "sxw" to "application/vnd.sun.xml.writer",
    "sxg" to "application/vnd.sun.xml.writer.global",
    "stw" to "application/vnd.sun.xml.writer.template",
    "otf" to "application/x-font-otf",
    "osfpvg" to "application/vnd.yamaha.openscoreformat.osfpvg+xml",
    "dp" to "application/vnd.osgi.dp",
    "pdb" to "application/vnd.palm",
    "p" to "text/x-pascal",
    "paw" to "application/vnd.pawaafile",
    "pclxl" to "application/vnd.hp-pclxl",
    "efif" to "application/vnd.picsel",
    "pcx" to "image/x-pcx",
    "psd" to "image/vnd.adobe.photoshop",
    "prf" to "application/pics-rules",
    "pic" to "image/x-pict",
    "chat" to "application/x-chat",
    "p10" to "application/pkcs10",
    "p12" to "application/x-pkcs12",
    "p7m" to "application/pkcs7-mime",
    "p7s" to "application/pkcs7-signature",
    "p7r" to "application/x-pkcs7-certreqresp",
    "p7b" to "application/x-pkcs7-certificates",
    "p8" to "application/pkcs8",
    "plf" to "application/vnd.pocketlearn",
    "pnm" to "image/x-portable-anymap",
    "pbm" to "image/x-portable-bitmap",
    "pcf" to "application/x-font-pcf",
    "pfr" to "application/font-tdpfr",
    "pgn" to "application/x-chess-pgn",
    "pgm" to "image/x-portable-graymap",
    "png" to "image/png",
    "ppm" to "image/x-portable-pixmap",
    "pskcxml" to "application/pskc+xml",
    "pml" to "application/vnd.ctc-posml",
    "ai" to "application/postscript",
    "pfa" to "application/x-font-type1",
    "pbd" to "application/vnd.powerbuilder6",
    "pgp" to "application/pgp-signature",
    "box" to "application/vnd.previewsystems.box",
    "ptid" to "application/vnd.pvi.ptid1",
    "pls" to "application/pls+xml",
    "str" to "application/vnd.pg.format",
    "ei6" to "application/vnd.pg.osasli",
    "dsc" to "text/prs.lines.tag",
    "psf" to "application/x-font-linux-psf",
    "qps" to "application/vnd.publishare-delta-tree",
    "wg" to "application/vnd.pmi.widget",
    "qxd" to "application/vnd.quark.quarkxpress",
    "esf" to "application/vnd.epson.esf",
    "msf" to "application/vnd.epson.msf",
    "ssf" to "application/vnd.epson.ssf",
    "qam" to "application/vnd.epson.quickanime",
    "qfx" to "application/vnd.intu.qfx",
    "qt" to "video/quicktime",
    "rar" to "application/x-rar-compressed",
    "ram" to "audio/x-pn-realaudio",
    "rmp" to "audio/x-pn-realaudio-plugin",
    "rsd" to "application/rsd+xml",
    "rm" to "application/vnd.rn-realmedia",
    "bed" to "application/vnd.realvnc.bed",
    "mxl" to "application/vnd.recordare.musicxml",
    "musicxml" to "application/vnd.recordare.musicxml+xml",
    "rnc" to "application/relax-ng-compact-syntax",
    "rdz" to "application/vnd.data-vision.rdz",
    "rdf" to "application/rdf+xml",
    "rp9" to "application/vnd.cloanto.rp9",
    "jisp" to "application/vnd.jisp",
    "rtf" to "application/rtf",
    "rtx" to "text/richtext",
    "link66" to "application/vnd.route66.link66+xml",
    "rss" to "application/rss+xml",
    "shf" to "application/shf+xml",
    "st" to "application/vnd.sailingtracker.track",
    "svg" to "image/svg+xml",
    "sus" to "application/vnd.sus-calendar",
    "sru" to "application/sru+xml",
    "setpay" to "application/set-payment-initiation",
    "setreg" to "application/set-registration-initiation",
    "sema" to "application/vnd.sema",
    "semd" to "application/vnd.semd",
    "semf" to "application/vnd.semf",
    "see" to "application/vnd.seemail",
    "snf" to "application/x-font-snf",
    "spq" to "application/scvp-vp-request",
    "spp" to "application/scvp-vp-response",
    "scq" to "application/scvp-cv-request",
    "scs" to "application/scvp-cv-response",
    "sdp" to "application/sdp",
    "etx" to "text/x-setext",
    "movie" to "video/x-sgi-movie",
    "ifm" to "application/vnd.shana.informed.formdata",
    "itp" to "application/vnd.shana.informed.formtemplate",
    "iif" to "application/vnd.shana.informed.interchange",
    "ipk" to "application/vnd.shana.informed.package",
    "tfi" to "application/thraud+xml",
    "shar" to "application/x-shar",
    "rgb" to "image/x-rgb",
    "slt" to "application/vnd.epson.salt",
    "aso" to "application/vnd.accpac.simply.aso",
    "imp" to "application/vnd.accpac.simply.imp",
    "twd" to "application/vnd.simtech-mindmapper",
    "csp" to "application/vnd.commonspace",
    "saf" to "application/vnd.yamaha.smaf-audio",
    "mmf" to "application/vnd.smaf",
    "spf" to "application/vnd.yamaha.smaf-phrase",
    "teacher" to "application/vnd.smart.teacher",
    "svd" to "application/vnd.svd",
    "rq" to "application/sparql-query",
    "srx" to "application/sparql-results+xml",
    "gram" to "application/srgs",
    "grxml" to "application/srgs+xml",
    "ssml" to "application/ssml+xml",
    "skp" to "application/vnd.koan",
    "sgml" to "text/sgml",
    "sdc" to "application/vnd.stardivision.calc",
    "sda" to "application/vnd.stardivision.draw",
    "sdd" to "application/vnd.stardivision.impress",
    "smf" to "application/vnd.stardivision.math",
    "sdw" to "application/vnd.stardivision.writer",
    "sgl" to "application/vnd.stardivision.writer-global",
    "sm" to "application/vnd.stepmania.stepchart",
    "sit" to "application/x-stuffit",
    "sitx" to "application/x-stuffitx",
    "sdkm" to "application/vnd.solent.sdkm+xml",
    "xo" to "application/vnd.olpc-sugar",
    "au" to "audio/basic",
    "wqd" to "application/vnd.wqd",
    "sis" to "application/vnd.symbian.install",
    "smi" to "application/smil+xml",
    "xsm" to "application/vnd.syncml+xml",
    "bdm" to "application/vnd.syncml.dm+wbxml",
    "xdm" to "application/vnd.syncml.dm+xml",
    "sv4cpio" to "application/x-sv4cpio",
    "sv4crc" to "application/x-sv4crc",
    "sbml" to "application/sbml+xml",
    "tsv" to "text/tab-separated-values",
    "tiff" to "image/tiff",
    "tao" to "application/vnd.tao.intent-module-archive",
    "tar" to "application/x-tar",
    "tcl" to "application/x-tcl",
    "tex" to "application/x-tex",
    "tfm" to "application/x-tex-tfm",
    "tei" to "application/tei+xml",
    "txt" to "text/plain",
    "dxp" to "application/vnd.spotfire.dxp",
    "sfs" to "application/vnd.spotfire.sfs",
    "tsd" to "application/timestamped-data",
    "tpt" to "application/vnd.trid.tpt",
    "mxs" to "application/vnd.triscape.mxs",
    "t" to "text/troff",
    "tra" to "application/vnd.trueapp",
    "ttf" to "application/x-font-ttf",
    "ttl" to "text/turtle",
    "umj" to "application/vnd.umajin",
    "uoml" to "application/vnd.uoml+xml",
    "unityweb" to "application/vnd.unity",
    "ufd" to "application/vnd.ufdl",
    "uri" to "text/uri-list",
    "utz" to "application/vnd.uiq.theme",
    "ustar" to "application/x-ustar",
    "uu" to "text/x-uuencode",
    "vcs" to "text/x-vcalendar",
    "vcf" to "text/x-vcard",
    "vcd" to "application/x-cdlink",
    "vsf" to "application/vnd.vsf",
    "wrl" to "model/vrml",
    "vcx" to "application/vnd.vcx",
    "mts" to "model/vnd.mts",
    "vtu" to "model/vnd.vtu",
    "vis" to "application/vnd.visionary",
    "viv" to "video/vnd.vivo",
    "ccxml" to "application/ccxml+xml,",
    "vxml" to "application/voicexml+xml",
    "src" to "application/x-wais-source",
    "wbxml" to "application/vnd.wap.wbxml",
    "wbmp" to "image/vnd.wap.wbmp",
    "wav" to "audio/x-wav",
    "davmount" to "application/davmount+xml",
    "woff" to "application/x-font-woff",
    "wspolicy" to "application/wspolicy+xml",
    "webp" to "image/webp",
    "wtb" to "application/vnd.webturbo",
    "wgt" to "application/widget",
    "hlp" to "application/winhlp",
    "wml" to "text/vnd.wap.wml",
    "wmls" to "text/vnd.wap.wmlscript",
    "wmlsc" to "application/vnd.wap.wmlscriptc",
    "wpd" to "application/vnd.wordperfect",
    "stf" to "application/vnd.wt.stf",
    "wsdl" to "application/wsdl+xml",
    "xbm" to "image/x-xbitmap",
    "xpm" to "image/x-xpixmap",
    "xwd" to "image/x-xwindowdump",
    "der" to "application/x-x509-ca-cert",
    "fig" to "application/x-xfig",
    "xhtml" to "application/xhtml+xml",
    "xml" to "application/xml",
    "xdf" to "application/xcap-diff+xml",
    "xenc" to "application/xenc+xml",
    "xer" to "application/patch-ops-error+xml",
    "rl" to "application/resource-lists+xml",
    "rs" to "application/rls-services+xml",
    "rld" to "application/resource-lists-diff+xml",
    "xslt" to "application/xslt+xml",
    "xop" to "application/xop+xml",
    "xpi" to "application/x-xpinstall",
    "xspf" to "application/xspf+xml",
    "xul" to "application/vnd.mozilla.xul+xml",
    "xyz" to "chemical/x-xyz",
    "yaml" to "text/yaml",
    "yang" to "application/yang",
    "yin" to "application/yin+xml",
    "zir" to "application/vnd.zul",
    "zip" to "application/zip",
    "zmm" to "application/vnd.handheld-entertainment+xml",
    "zaz" to "application/vnd.zzazz.deck+xml"
)