package com.github.andreyasadchy.xtra.util.chat

import com.github.andreyasadchy.xtra.util.WebSocket
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import java.util.Timer
import javax.net.ssl.X509TrustManager
import kotlin.concurrent.schedule
import kotlin.random.Random

class ChatReadWebSocket(
    private val loggedIn: Boolean,
    private val channelName: String,
    private val onConnect: (() -> Unit)? = null,
    private val onDisconnect: ((String, String) -> Unit)? = null,
    private val onChatMessage: ((String, Boolean) -> Unit)? = null,
    private val onClearMessage: ((String) -> Unit)? = null,
    private val onClearChat: ((String) -> Unit)? = null,
    private val onNotice: ((String) -> Unit)? = null,
    private val onRoomState: ((String) -> Unit)? = null,
    private val trustManager: X509TrustManager?,
    private val coroutineScope: CoroutineScope,
) {
    private var webSocket: WebSocket? = null
    internal var pingTimer: Timer? = null
    internal var pongTimer: Timer? = null
    val isActive: Boolean?
        get() = webSocket?.isActive

    fun connect() {
        webSocket = WebSocket("wss://irc-ws.chat.twitch.tv", trustManager, ChatReadWebSocketListener())
        coroutineScope.launch {
            webSocket?.start()
        }
    }

    suspend fun disconnect() {
        pingTimer?.cancel()
        pongTimer?.cancel()
        webSocket?.stop()
    }

    internal fun startPingTimer() {
        pingTimer = Timer().apply {
            schedule(270000) {
                coroutineScope.launch {
                    webSocket?.write("PING")
                }
                startPongTimer()
            }
        }
    }

    private fun startPongTimer() {
        pongTimer = Timer().apply {
            schedule(10000) {
                coroutineScope.launch {
                    webSocket?.disconnect()
                }
            }
        }
    }

    private inner class ChatReadWebSocketListener: WebSocket.Listener {
        override fun onOpen(webSocket: WebSocket) {
            coroutineScope.launch {
                webSocket.write("CAP REQ :twitch.tv/tags twitch.tv/commands")
                webSocket.write("NICK justinfan${Random.nextInt(1000, 10000)}")
                webSocket.write("JOIN #$channelName")
            }
            onConnect?.invoke()
            pingTimer?.cancel()
            pongTimer?.cancel()
            startPingTimer()
        }

        override fun onMessage(webSocket: WebSocket, message: String) {
            message.removeSuffix("\r\n").split("\r\n").forEach {
                it.run {
                    when {
                        contains("PRIVMSG") -> onChatMessage?.invoke(this, false)
                        contains("USERNOTICE") -> onChatMessage?.invoke(this, true)
                        contains("CLEARMSG") -> onClearMessage?.invoke(this)
                        contains("CLEARCHAT") -> onClearChat?.invoke(this)
                        contains("NOTICE") -> {
                            if (!loggedIn) {
                                onNotice?.invoke(this)
                            }
                        }
                        contains("ROOMSTATE") -> onRoomState?.invoke(this)
                        startsWith("PING") -> {
                            coroutineScope.launch {
                                webSocket.write("PONG")
                            }
                        }
                        startsWith("PONG") -> {
                            pingTimer?.cancel()
                            pongTimer?.cancel()
                            startPingTimer()
                        }
                        startsWith("RECONNECT") -> {
                            pingTimer?.cancel()
                            pongTimer?.cancel()
                            coroutineScope.launch {
                                webSocket.disconnect()
                            }
                        }
                    }
                }
            }
        }

        override fun onFailure(webSocket: WebSocket, throwable: Throwable) {
            onDisconnect?.invoke(throwable.toString(), throwable.stackTraceToString())
        }
    }
}