// Copyright (c) 2019, the Dart project authors. Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

import 'package:analyzer/file_system/file_system.dart';
import 'package:analyzer/src/error/codes.dart';
import 'package:analyzer/src/generated/parser.dart';
import 'package:analyzer/utilities/package_config_file_builder.dart';
import 'package:test_reflective_loader/test_reflective_loader.dart';

import '../dart/resolution/context_collection_resolution.dart';

main() {
  defineReflectiveSuite(() {
    defineReflectiveTests(DeprecatedMemberUse_BlazeWorkspaceTest);
    defineReflectiveTests(DeprecatedMemberUse_GnWorkspaceTest);
    defineReflectiveTests(DeprecatedMemberUse_PackageBuildWorkspaceTest);
    defineReflectiveTests(DeprecatedMemberUse_PackageConfigWorkspaceTest);
  });
}

@reflectiveTest
class DeprecatedMemberUse_BlazeWorkspaceTest
    extends BlazeWorkspaceResolutionTest {
  test_dart() async {
    newFile('$workspaceRootPath/foo/bar/lib/a.dart', r'''
@deprecated
class A {}
''');

    await assertErrorsInCode(
      r'''
import 'package:foo.bar/a.dart';

void f(A a) {}
''',
      [error(HintCode.deprecatedMemberUse, 41, 1)],
    );
  }

  test_thirdPartyDart() async {
    newFile('$workspaceThirdPartyDartPath/aaa/lib/a.dart', r'''
@deprecated
class A {}
''');

    assertBlazeWorkspaceFor(testFile);

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f(A a) {}
''',
      [error(HintCode.deprecatedMemberUse, 37, 1)],
    );
  }
}

@reflectiveTest
class DeprecatedMemberUse_GnWorkspaceTest extends ContextResolutionTest {
  @override
  List<String> get collectionIncludedPaths => [workspaceRootPath];

  String get genPath => '$outPath/dartlang/gen';

  String get myPackageLibPath => '$myPackageRootPath/lib';

  String get myPackageRootPath => '$workspaceRootPath/my';

  Folder get outFolder => getFolder(outPath);

  String get outPath => '$workspaceRootPath/out/default';

  @override
  File get testFile => getFile('$myPackageLibPath/my.dart');

  String get workspaceRootPath => '/workspace';

  @override
  void setUp() {
    super.setUp();
    newFolder('$workspaceRootPath/.jiri_root');

    newFile('$workspaceRootPath/.fx-build-dir', '''
${outFolder.path}
''');

    newBuildGnFile(myPackageRootPath, '');
  }

  test_differentPackage() async {
    newBuildGnFile('$workspaceRootPath/aaa', '');

    var myPackageConfig = getFile('$genPath/my/my_package_config.json');
    _writeWorkspacePackagesFile(myPackageConfig, {
      'aaa': '$workspaceRootPath/aaa/lib',
      'my': myPackageLibPath,
    });

    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
@deprecated
class A {}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f(A a) {}
''',
      [error(HintCode.deprecatedMemberUse, 37, 1)],
    );
  }

  @override
  void verifyCreatedCollection() {
    super.verifyCreatedCollection();
    assertGnWorkspaceFor(testFile);
  }

  void _writeWorkspacePackagesFile(
    File file,
    Map<String, String> nameToLibPath,
  ) {
    var packages = nameToLibPath.entries.map(
      (entry) =>
          '''{
    "languageVersion": "2.2",
    "name": "${entry.key}",
    "packageUri": ".",
    "rootUri": "${toUriStr(entry.value)}"
  }''',
    );

    newFile(file.path, '''{
  "configVersion": 2,
  "packages": [ ${packages.join(', ')} ]
}''');
  }
}

@reflectiveTest
class DeprecatedMemberUse_PackageBuildWorkspaceTest
    extends _PackageConfigWorkspaceBase {
  test_generated() async {
    writeTestPackageConfig(
      PackageConfigFileBuilder()
        ..add(name: 'aaa', rootPath: '$workspaceRootPath/aaa'),
    );

    newPubspecYamlFile(testPackageRootPath, 'name: test');
    _newTestPackageGeneratedFile(
      packageName: 'aaa',
      pathInLib: 'a.dart',
      content: r'''
@deprecated
class A {}
''',
    );

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f(A a) {}
''',
      [error(HintCode.deprecatedMemberUse, 37, 1)],
    );
  }

  test_lib() async {
    writeTestPackageConfig(
      PackageConfigFileBuilder()
        ..add(name: 'aaa', rootPath: '$workspaceRootPath/aaa'),
    );

    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
@deprecated
class A {}
''');

    newPubspecYamlFile(testPackageRootPath, 'name: test');
    _createTestPackageBuildMarker();

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f(A a) {}
''',
      [error(HintCode.deprecatedMemberUse, 37, 1)],
    );
  }
}

@reflectiveTest
class DeprecatedMemberUse_PackageConfigWorkspaceTest
    extends PubPackageResolutionTest {
  String get externalLibPath => '$workspaceRootPath/aaa/lib/a.dart';

  String get externalLibUri => 'package:aaa/a.dart';

  Future<void> assertErrorsInCode2(
    List<ExpectedDiagnostic> expectedDiagnostics, {
    required String externalCode,
    required String code,
  }) async {
    newFile(externalLibPath, externalCode);

    await assertErrorsInCode('''
import '$externalLibUri';
$code
''', expectedDiagnostics);
  }

  Future<void> assertNoErrorsInCode2({
    required String externalCode,
    required String code,
  }) async {
    newFile(externalLibPath, externalCode);

    await assertNoErrorsInCode('''
import '$externalLibUri';
$code
''');
  }

  @override
  void setUp() {
    super.setUp();

    writeTestPackageConfig(
      PackageConfigFileBuilder()
        ..add(name: 'aaa', rootPath: '$workspaceRootPath/aaa'),
    );
  }

  test_assignmentExpression_compound_deprecatedGetter() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int get x => 0;

set x(int _) {}
''',
      code: r'''
void f() {
  x += 2;
}
''',
      [error(HintCode.deprecatedMemberUse, 42, 1)],
    );
  }

  test_assignmentExpression_compound_deprecatedSetter() async {
    await assertErrorsInCode2(
      externalCode: r'''
int get x => 0;

@deprecated
set x(int _) {}
''',
      code: r'''
void f() {
  x += 2;
}
''',
      [error(HintCode.deprecatedMemberUse, 42, 1)],
    );
  }

  test_assignmentExpression_simple_deprecatedGetter() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
int get x => 0;

set x(int _) {}
''',
      code: r'''
void f() {
  x = 0;
}
''',
    );
  }

  test_assignmentExpression_simple_deprecatedGetterSetter() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int x = 1;
''',
      code: r'''
void f() {
  x = 0;
}
''',
      [error(HintCode.deprecatedMemberUse, 42, 1)],
    );
  }

  test_assignmentExpression_simple_deprecatedSetter() async {
    await assertErrorsInCode2(
      externalCode: r'''
int get x => 0;

@deprecated
set x(int _) {}
''',
      code: r'''
void f() {
  x = 0;
}
''',
      [error(HintCode.deprecatedMemberUse, 42, 1)],
    );
  }

  test_call() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  call() {}
}
''',
      code: r'''
void f(A a) {
  a();
}
''',
      [error(HintCode.deprecatedMemberUse, 45, 3)],
    );
  }

  test_class() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
class A {}
''',
      code: r'''
void f(A a) {}
''',
      [error(HintCode.deprecatedMemberUse, 36, 1)],
    );
  }

  test_class_inDeprecatedFunctionTypeAlias() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
@deprecated
class A {}
''');

    await assertNoErrorsInCode(r'''
import 'package:aaa/a.dart';

@deprecated
typedef A T();
''');
  }

  test_class_inDeprecatedGenericTypeAlias() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
@deprecated
class A {}
''');

    await assertNoErrorsInCode(r'''
import 'package:aaa/a.dart';

@deprecated
typedef T = A Function();
''');
  }

  test_compoundAssignment() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  A operator+(A a) { return a; }
}
''',
      code: r'''
f(A a, A b) {
  a += b;
}
''',
      [error(HintCode.deprecatedMemberUse, 45, 6)],
    );
  }

  test_deprecatedField_inObjectPattern_explicitName() async {
    await assertErrorsInCode2(
      externalCode: r'''
class C {
  @Deprecated('')
  final int foo = 0;
}
''',
      code: '''
int g(Object s) =>
  switch (s) {
    C(foo: var f) => f,
    _ => 7,
  };
''',
      [error(HintCode.deprecatedMemberUse, 69, 3)],
    );
  }

  test_deprecatedField_inObjectPattern_inferredName() async {
    await assertErrorsInCode2(
      externalCode: r'''
class C {
  @Deprecated('')
  final int foo = 0;
}
''',
      code: '''
int g(Object s) =>
  switch (s) {
    C(:var foo) => foo,
    _ => 7,
  };
''',
      [error(HintCode.deprecatedMemberUse, 74, 3)],
    );
  }

  test_dotShorthandConstructorInvocation_deprecatedClass_deprecatedConstructor() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
@deprecated
class A {
  @deprecated
  A();
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f() {
  A a = .new();
}
''',
      [
        error(HintCode.deprecatedMemberUse, 43, 1),
        error(WarningCode.unusedLocalVariable, 45, 1),
        error(HintCode.deprecatedMemberUse, 50, 3),
      ],
    );
  }

  test_dotShorthandConstructorInvocation_deprecatedClass_undeprecatedConstructor() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
@deprecated
class A {
  A();
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f() {
  A a = .new();
}
''',
      [
        error(HintCode.deprecatedMemberUse, 43, 1),
        error(WarningCode.unusedLocalVariable, 45, 1),
      ],
    );
  }

  test_dotShorthandConstructorInvocation_deprecatedClass_undeprecatedNamedConstructor() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
@deprecated
class A {
  A.a();
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f() {
  A a = .a();
}
''',
      [
        error(HintCode.deprecatedMemberUse, 43, 1),
        error(WarningCode.unusedLocalVariable, 45, 1),
      ],
    );
  }

  test_dotShorthandConstructorInvocation_namedConstructor() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  A.named(int i) {}
}
''',
      code: r'''
f() {
  A a = .named(1);
}
''',
      [
        error(WarningCode.unusedLocalVariable, 39, 1),
        error(
          HintCode.deprecatedMemberUse,
          44,
          5,
          messageContains: ["'A.named' is deprecated and shouldn't be used."],
        ),
      ],
    );
  }

  test_dotShorthandConstructorInvocation_undeprecatedClass_deprecatedConstructor() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  @deprecated
  A();
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f() {
  A a = .new();
}
''',
      [
        error(WarningCode.unusedLocalVariable, 45, 1),
        error(HintCode.deprecatedMemberUse, 50, 3),
      ],
    );
  }

  test_dotShorthandInvocation_deprecatedMethod() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  @deprecated
  static A m() => A();
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f() {
  A a = .m();
}
''',
      [
        error(WarningCode.unusedLocalVariable, 45, 1),
        error(HintCode.deprecatedMemberUse, 50, 1),
      ],
    );
  }

  test_export() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
@deprecated
library a;
''');

    await assertErrorsInCode(
      '''
export 'package:aaa/a.dart';
''',
      [error(HintCode.deprecatedMemberUse, 0, 28)],
    );
  }

  test_export_fromSamePackage() async {
    newFile('$testPackageLibPath/lib2.dart', r'''
@deprecated
library a;
''');

    await assertNoErrorsInCode('''
export 'lib2.dart';
''');
  }

  test_extensionOverride() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
extension E on int {
  int get foo => 0;
}
''',
      code: '''
void f() {
  E(0).foo;
}
''',
      [error(HintCode.deprecatedMemberUse, 42, 1)],
    );
  }

  test_field_implicitGetter() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  @deprecated
  int foo = 0;
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f(A a) {
  a.foo;
}
''',
      [error(HintCode.deprecatedMemberUse, 48, 3)],
    );
  }

  test_field_implicitSetter() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  @deprecated
  int foo = 0;
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f(A a) {
  a.foo = 0;
}
''',
      [error(HintCode.deprecatedMemberUse, 48, 3)],
    );
  }

  test_field_inDeprecatedConstructor() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  @deprecated
  int x = 0;
}
''');

    await assertNoErrorsInCode(r'''
import 'package:aaa/a.dart';

class B extends A {
  @deprecated
  B() {
    x;
    x = 1;
  }
}
''');
  }

  test_hideCombinator() async {
    newFile(externalLibPath, r'''
@deprecated
class A {}
''');
    await assertNoErrorsInCode('''
// ignore: unused_import
import '$externalLibUri' hide A;
''');
  }

  test_import() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
@deprecated
library a;
''');

    await assertErrorsInCode(
      r'''
// ignore:unused_import
import 'package:aaa/a.dart';
''',
      [
        error(
          HintCode.deprecatedMemberUse,
          24,
          28,
          messageContains: ['package:aaa/a.dart'],
        ),
      ],
    );
  }

  test_incorrectlyNestedNamedParameterDeclaration() async {
    // This is a regression test; previously this code would cause an analyzer
    // crash in DeprecatedMemberUseVerifier.
    await assertErrorsInCode(
      r'''
class C {
  final String x;
  final bool y;

  const C({
    required this.x,
    {this.y = false}
  });
}

const z = C(x: '');
''',
      [
        error(CompileTimeErrorCode.finalNotInitializedConstructor1, 53, 1),
        error(ParserErrorCode.missingIdentifier, 82, 1),
        error(ParserErrorCode.expectedToken, 82, 1),
      ],
    );
  }

  test_inDeprecatedClass() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
f() {}
''',
      code: r'''
@deprecated
class C {
  m() {
    f();
  }
}
''',
    );
  }

  test_inDeprecatedDefaultFormalParameter() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
class C {
  const C();
}
''',
      code: r'''
f({@deprecated C? c = const C()}) {}
''',
    );
  }

  test_inDeprecatedEnum() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
void f() {}
''',
      code: r'''
@deprecated
enum E {
  one, two;

  void m() {
    f();
  }
}
''',
    );
  }

  test_inDeprecatedExtension() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
void f() {}
''',
      code: r'''
@deprecated
extension E on int {
  void m() {
    f();
  }
}
''',
    );
  }

  test_inDeprecatedExtensionType() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
f() {}
''',
      code: '''
@deprecated
extension type E(int i) {
  m() {
    f();
  }
}
''',
    );
  }

  test_inDeprecatedField() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
class C {}
''',
      code: r'''
class X {
  @deprecated
  C f;
  X(this.f);
}
''',
    );
  }

  test_inDeprecatedFieldFormalParameter() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
class C {}
''',
      code: r'''
class A {
  Object? o;
  A({@deprecated C? this.o});
}
''',
    );
  }

  test_inDeprecatedFunction() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
f() {}
''',
      code: '''
@deprecated
g() {
  f();
}
''',
    );
  }

  test_inDeprecatedFunctionTypedFormalParameter() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
class C {}
''',
      code: r'''
f({@deprecated C? callback()?}) {}
''',
    );
  }

  test_inDeprecatedLibrary() async {
    newFile(externalLibPath, r'''
@deprecated
f() {}
''');
    await assertNoErrorsInCode('''
@deprecated
library lib;
import '$externalLibUri';

class C {
  m() {
    f();
  }
}
''');
  }

  test_inDeprecatedMethod() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
f() {}
''',
      code: '''
class C {
  @deprecated
  m() {
    f();
  }
}
''',
    );
  }

  test_inDeprecatedMethod_inDeprecatedClass() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
f() {}
''',
      code: '''
@deprecated
class C {
  @deprecated
  m() {
    f();
  }
}
''',
    );
  }

  test_inDeprecatedMixin() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
f() {}
''',
      code: '''
@deprecated
mixin M {
  m() {
    f();
  }
}
''',
    );
  }

  test_inDeprecatedSimpleFormalParameter() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
class C {}
''',
      code: r'''
f({@deprecated C? c}) {}
''',
    );
  }

  test_inDeprecatedSuperFormalParameter() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
class C {}
''',
      code: r'''
class A {
  A({Object? o});
}
class B extends A {
  B({@deprecated C? super.o});
}
''',
    );
  }

  test_inDeprecatedTopLevelVariable() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
@deprecated
class C {}
var x = C();
''',
      code: r'''
@deprecated
C v = x;
''',
    );
  }

  test_indexExpression() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  operator[](int i) {}
}
''',
      code: r'''
void f(A a) {
  return a[1];
}
''',
      [error(HintCode.deprecatedMemberUse, 52, 4)],
    );
  }

  test_inEnum() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
void f() {}
''',
      code: r'''
enum E {
  one, two;

  void m() {
    f();
  }
}
''',
      [error(HintCode.deprecatedMemberUse, 68, 1)],
    );
  }

  test_inExtension() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
void f() {}
''',
      code: r'''
extension E on int {
  void m() {
    f();
  }
}
''',
      [error(HintCode.deprecatedMemberUse, 67, 1)],
    );
  }

  test_instanceCreation_deprecatedClass_deprecatedConstructor() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
@deprecated
class A {
  @deprecated
  A();
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f() {
  A();
}
''',
      [error(HintCode.deprecatedMemberUse, 43, 1)],
    );
  }

  test_instanceCreation_deprecatedClass_undeprecatedConstructor() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
@deprecated
class A {
  A();
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f() {
  A();
}
''',
      [error(HintCode.deprecatedMemberUse, 43, 1)],
    );
  }

  test_instanceCreation_deprecatedClass_undeprecatedNamedConstructor() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
@deprecated
class A {
  A.a();
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f() {
  A.a();
}
''',
      [
        // https://github.com/dart-lang/linter/issues/4752
        // Highlights `A`.
        error(HintCode.deprecatedMemberUse, 43, 1),
      ],
    );
  }

  test_instanceCreation_namedConstructor() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  A.named(int i) {}
}
''',
      code: r'''
f() {
  return new A.named(1);
}
''',
      [
        error(
          HintCode.deprecatedMemberUse,
          48,
          7,
          messageContains: ["'A.named' is deprecated and shouldn't be used."],
        ),
      ],
    );
  }

  test_instanceCreation_undeprecatedClass_deprecatedConstructor() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  @deprecated
  A();
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f() {
  A();
}
''',
      [error(HintCode.deprecatedMemberUse, 43, 1)],
    );
  }

  test_instanceCreation_unnamedConstructor() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  A(int i) {}
}
''',
      code: r'''
f() {
  return new A(1);
}
''',
      [error(HintCode.deprecatedMemberUse, 48, 1)],
    );
  }

  test_methodInvocation() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  @deprecated
  void foo() {}
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f(A a) {
  a.foo();
}
''',
      [error(HintCode.deprecatedMemberUse, 48, 3)],
    );
  }

  test_methodInvocation_constantAnnotation() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int f() => 0;
''',
      code: r'''
var x = f();
''',
      [error(HintCode.deprecatedMemberUse, 37, 1)],
    );
  }

  test_methodInvocation_constructorAnnotation() async {
    await assertErrorsInCode2(
      externalCode: r'''
@Deprecated('0.9')
int f() => 0;
''',
      code: r'''
var x = f();
''',
      [
        error(
          HintCode.deprecatedMemberUseWithMessage,
          37,
          1,
          text: "'f' is deprecated and shouldn't be used. 0.9.",
        ),
      ],
    );
  }

  test_methodInvocation_constructorAnnotation_dot() async {
    await assertErrorsInCode2(
      externalCode: r'''
@Deprecated('0.9.')
int f() => 0;
''',
      code: r'''
var x = f();
''',
      [
        error(
          HintCode.deprecatedMemberUseWithMessage,
          37,
          1,
          text: "'f' is deprecated and shouldn't be used. 0.9.",
        ),
      ],
    );
  }

  test_methodInvocation_constructorAnnotation_exclamationMark() async {
    await assertErrorsInCode2(
      externalCode: r'''
@Deprecated(' Really! ')
int f() => 0;
''',
      code: r'''
var x = f();
''',
      [
        error(
          HintCode.deprecatedMemberUseWithMessage,
          37,
          1,
          text: "'f' is deprecated and shouldn't be used. Really!",
        ),
      ],
    );
  }

  test_methodInvocation_constructorAnnotation_onlyDot() async {
    await assertErrorsInCode2(
      externalCode: r'''
@Deprecated('.')
int f() => 0;
''',
      code: r'''
var x = f();
''',
      [
        error(
          HintCode.deprecatedMemberUse,
          37,
          1,
          text: "'f' is deprecated and shouldn't be used.",
        ),
      ],
    );
  }

  test_methodInvocation_constructorAnnotation_questionMark() async {
    await assertErrorsInCode2(
      externalCode: r'''
@Deprecated('Are you sure?')
int f() => 0;
''',
      code: r'''
var x = f();
''',
      [
        error(
          HintCode.deprecatedMemberUseWithMessage,
          37,
          1,
          text: "'f' is deprecated and shouldn't be used. Are you sure?",
        ),
      ],
    );
  }

  test_methodInvocation_fromSamePackage() async {
    newFile('$testPackageLibPath/lib2.dart', r'''
class A {
  @deprecated
  void foo() {}
}
''');

    await assertNoErrorsInCode(r'''
import 'lib2.dart';

void f(A a) {
  a.foo();
}
''');
  }

  test_methodInvocation_inDeprecatedConstructor() async {
    await assertNoErrorsInCode(r'''
class A {
  @deprecated
  A() {
    foo();
  }

  @deprecated
  void foo() {}
}
''');
  }

  test_methodInvocation_nonUseKind() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  @Deprecated.extend()
  void foo() {}
}
''');

    await assertNoErrorsInCode(r'''
import 'package:aaa/a.dart';

void f(A a) {
  a.foo();
}
''');
  }

  test_methodInvocation_unrelatedAnnotation() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  @override
  void foo() {}
}
''');

    await assertNoErrorsInCode(r'''
import 'package:aaa/a.dart';

void f(A a) {
  a.foo();
}
''');
  }

  test_methodInvocation_withMessage() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  @Deprecated('0.9')
  void foo() {}
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f(A a) {
  a.foo();
}
''',
      [
        error(
          HintCode.deprecatedMemberUseWithMessage,
          48,
          3,
          text: "'foo' is deprecated and shouldn't be used. 0.9.",
        ),
      ],
    );
  }

  test_mixin_inDeprecatedClassTypeAlias() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
@deprecated
mixin A {}
''');

    await assertNoErrorsInCode(r'''
import 'package:aaa/a.dart';

@deprecated
class B = Object with A;
''');
  }

  test_namedParameterMissingName() async {
    // This is a regression test; previously this code would cause an analyzer
    // crash in DeprecatedMemberUseVerifier.
    await assertErrorsInCode(
      r'''
class C {
  const C({this.});
}
var z = C(x: '');
''',
      [
        error(
          CompileTimeErrorCode.initializingFormalForNonExistentField,
          21,
          5,
        ),
        error(ParserErrorCode.missingIdentifier, 26, 1),
        error(CompileTimeErrorCode.undefinedNamedParameter, 42, 1),
      ],
    );
  }

  test_operator() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  operator+(A a) {}
}
''',
      code: r'''
f(A a, A b) {
  return a + b;
}
''',
      [error(HintCode.deprecatedMemberUse, 52, 5)],
    );
  }

  test_parameter_named() async {
    await assertErrorsInCode2(
      externalCode: r'''
void f({@deprecated int x = 0}) {}
''',
      code: r'''
void g() => f(x: 1);
''',
      [error(HintCode.deprecatedMemberUse, 43, 1)],
    );
  }

  test_parameter_named_inDefiningConstructor_asFieldFormalParameter() async {
    await assertNoErrorsInCode(r'''
class C {
  int x;
  C({@deprecated this.x = 0});
}
''');
  }

  test_parameter_named_inDefiningConstructor_assertInitializer() async {
    await assertNoErrorsInCode(r'''
class C {
  C({@deprecated int y = 0}) : assert(y > 0);
}
''');
  }

  test_parameter_named_inDefiningConstructor_fieldInitializer() async {
    await assertNoErrorsInCode(r'''
class C {
  int x;
  C({@deprecated int y = 0}) : x = y;
}
''');
  }

  test_parameter_named_inDefiningConstructor_inFieldFormalParameter_notName() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {}

@deprecated
class B extends A {
  const B();
}

const B instance = const B();
''',
      code: r'''
class C {
  final A a;
  C({B this.a = instance});
}
''',
      [error(HintCode.deprecatedMemberUse, 57, 1)],
    );
  }

  test_parameter_named_inDefiningFunction() async {
    await assertNoErrorsInCode(r'''
f({@deprecated int x = 0}) => x;
''');
  }

  test_parameter_named_inDefiningLocalFunction() async {
    await assertNoErrorsInCode(r'''
class C {
  m() {
    f({@deprecated int x = 0}) {
      return x;
    }
    return f();
  }
}
''');
  }

  test_parameter_named_inDefiningMethod() async {
    await assertNoErrorsInCode(r'''
class C {
  m({@deprecated int x = 0}) {
    return x;
  }
}
''');
  }

  test_parameter_named_inNestedLocalFunction() async {
    await assertNoErrorsInCode(r'''
class C {
  m({@deprecated int x = 0}) {
    f() {
      return x;
    }
    return f();
  }
}
''');
  }

  test_parameter_named_ofFunction() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
void foo({@deprecated int a}) {}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f() {
  foo(a: 0);
}
''',
      [error(HintCode.deprecatedMemberUse, 47, 1)],
    );
  }

  test_parameter_named_ofMethod() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  void foo({@deprecated int a}) {}
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f(A a) {
  a.foo(a: 0);
}
''',
      [error(HintCode.deprecatedMemberUse, 52, 1)],
    );
  }

  test_parameter_positionalOptional() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {
  void foo([@deprecated int x = 0]) {}
}
''',
      code: '''
void f(A a) {
  a.foo(0);
}
''',
      [error(HintCode.deprecatedMemberUse, 51, 1)],
    );
  }

  test_parameter_positionalOptional_inDeprecatedConstructor() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
void foo([@deprecated int x = 0]) {}
''',
      code: r'''
class A {
  @deprecated
  A() {
    foo(0);
  }
}
''',
    );
  }

  test_parameter_positionalOptional_inDeprecatedFunction() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
class A {
  void foo([@deprecated int x = 0]) {}
}
''',
      code: r'''
@deprecated
void f(A a) {
  a.foo(0);
}
''',
    );
  }

  test_parameter_positionalRequired() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
class A {
  void foo(@deprecated int x) {}
}
''',
      code: r'''
void f(A a) {
  a.foo(0);
}
''',
    );
  }

  test_parameterInSuper_explicitInvocation() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  A([@deprecated int? p]);
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

class B extends A {
  B() : super(7);
}
''',
      [error(HintCode.deprecatedMemberUse, 64, 1)],
    );
  }

  test_parameterInSuper_explicitInvocation_namedParameter() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  A({@deprecated int? p});
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

class B extends A {
  B() : super(p: 7);
}
''',
      [error(HintCode.deprecatedMemberUse, 64, 1)],
    );
  }

  test_parameterInSuper_implicitArgument() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  A([@deprecated int? p]);
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

class B extends A {
  B([super.p]);
}
''',
      [error(HintCode.deprecatedMemberUse, 55, 7)],
    );
  }

  test_parameterInSuper_implicitArgument_alsoDeprecated() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  A([@deprecated int? p]);
}
''');

    await assertNoErrorsInCode(r'''
import 'package:aaa/a.dart';

class B extends A {
  B([@deprecated super.p]);
}
''');
  }

  test_parameterInSuper_implicitArgument_explicitInvocation() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  A.named([@deprecated int? p]);
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

class B extends A {
  B([super.p]) : super.named();
}
''',
      [error(HintCode.deprecatedMemberUse, 55, 7)],
    );
  }

  test_parameterInSuper_implicitInvocation_namedParameter() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  A({@deprecated int? p});
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

class B extends A {
  B({super.p});
}
''',
      [error(HintCode.deprecatedMemberUse, 55, 7)],
    );
  }

  test_postfixExpression_deprecatedGetter() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int get x => 0;

set x(int _) {}
''',
      code: r'''
void f() {
  x++;
}
''',
      [error(HintCode.deprecatedMemberUse, 42, 1)],
    );
  }

  test_postfixExpression_deprecatedNothing() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
int get x => 0;

set x(int _) {}
''',
      code: r'''
void f() {
  x++;
}
''',
    );
  }

  test_postfixExpression_deprecatedSetter() async {
    await assertErrorsInCode2(
      externalCode: r'''
int get x => 0;

@deprecated
set x(int _) {}
''',
      code: r'''
void f() {
  x++;
}
''',
      [error(HintCode.deprecatedMemberUse, 42, 1)],
    );
  }

  test_prefixedIdentifier_identifier() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  static const foo = 0;
}
''',
      code: r'''
void f() {
  A.foo;
}
''',
      [error(HintCode.deprecatedMemberUse, 44, 3)],
    );
  }

  test_prefixedIdentifier_prefix() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
class A {
  static const foo = 0;
}
''',
      code: r'''
void f() {
  A.foo;
}
''',
      [error(HintCode.deprecatedMemberUse, 42, 1)],
    );
  }

  test_prefixExpression_deprecatedGetter() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int get x => 0;

set x(int _) {}
''',
      code: r'''
void f() {
  ++x;
}
''',
      [error(HintCode.deprecatedMemberUse, 44, 1)],
    );
  }

  test_prefixExpression_deprecatedSetter() async {
    await assertErrorsInCode2(
      externalCode: r'''
int get x => 0;

@deprecated
set x(int _) {}
''',
      code: r'''
void f() {
  ++x;
}
''',
      [error(HintCode.deprecatedMemberUse, 44, 1)],
    );
  }

  test_propertyAccess_super() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  int get foo => 0;
}
''',
      code: r'''
class B extends A {
  void bar() {
    super.foo;
  }
}
''',
      [error(HintCode.deprecatedMemberUse, 74, 3)],
    );
  }

  test_redirectedConstructor_fromFactoryConstructor() async {
    await assertErrorsInCode2(
      externalCode: r'''
import 'package:test/test.dart';
class B extends A {
  @deprecated
  B();
}
''',
      code: r'''
class A {
  factory A.two() = B;
}
''',
      [error(HintCode.deprecatedMemberUse, 59, 1)],
    );
  }

  test_redirectedParameter_redirectingFactoryConstructor() async {
    await assertErrorsInCode2(
      externalCode: r'''
import 'package:test/test.dart';
class B extends A {
  B([@deprecated int? p]);
}
''',
      code: r'''
class A {
  factory A.two([int? p]) = B;
}
''',
      [error(HintCode.deprecatedMemberUse, 56, 6)],
    );
  }

  test_redirectedParameter_redirectingFactoryConstructor_deprecatedFunctionTypedParameter() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
import 'package:test/test.dart';
class B extends A {
  B([@deprecated void p()?]);
}
''',
      code: r'''
class A {
  factory A.two([@deprecated void p()?]) = B;
}
''',
    );
  }

  test_redirectedParameter_redirectingFactoryConstructor_deprecatedParameter() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
import 'package:test/test.dart';
class B extends A {
  B([@deprecated int? p]);
}
''',
      code: r'''
class A {
  factory A.two([@deprecated int? p]) = B;
}
''',
    );
  }

  test_redirectedParameter_redirectingFactoryConstructor_functionTypedParameter() async {
    await assertErrorsInCode2(
      externalCode: r'''
import 'package:test/test.dart';
class B extends A {
  B([@deprecated void p()?]);
}
''',
      code: r'''
class A {
  factory A.two([void p()?]) = B;
}
''',
      [error(HintCode.deprecatedMemberUse, 56, 9)],
    );
  }

  test_redirectedParameter_redirectingFactoryConstructor_named() async {
    await assertErrorsInCode2(
      externalCode: r'''
import 'package:test/test.dart';
class B extends A {
  B({@deprecated int? p});
}
''',
      code: r'''
class A {
  factory A.two({int? p}) = B;
}
''',
      [error(HintCode.deprecatedMemberUse, 56, 6)],
    );
  }

  test_redirectingConstructorInvocation_namedParameter() async {
    await assertNoErrorsInCode(r'''
class A {
  A({@deprecated int a = 0}) {}
  A.named() : this(a: 0);
}
''');
  }

  test_setterInvocation() async {
    newFile('$workspaceRootPath/aaa/lib/a.dart', r'''
class A {
  @deprecated
  set foo(int _) {}
}
''');

    await assertErrorsInCode(
      r'''
import 'package:aaa/a.dart';

void f(A a) {
  a.foo = 0;
}
''',
      [error(HintCode.deprecatedMemberUse, 48, 3)],
    );
  }

  test_showCombinator() async {
    newFile(externalLibPath, r'''
@deprecated
class A {}
''');
    await assertErrorsInCode(
      '''
// ignore: unused_import
import '$externalLibUri' show A;
''',
      [error(HintCode.deprecatedMemberUse, 58, 1)],
    );
  }

  test_superConstructor_factoryConstructor() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  A();
  A.two();
}
''',
      code: r'''
class B extends A {
  factory B() => B.two();
  B.two() : super.two();
}
''',
    );
  }

  test_superConstructor_implicitCall() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  A();
}
''',
      code: r'''
class B extends A {
  B();
}
''',
      [error(HintCode.deprecatedMemberUse, 51, 4)],
    );
  }

  test_superConstructor_namedConstructor() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  A.named() {}
}
''',
      code: r'''
class B extends A {
  B() : super.named() {}
}
''',
      [
        error(
          HintCode.deprecatedMemberUse,
          57,
          13,
          text: "'A.named' is deprecated and shouldn't be used.",
        ),
      ],
    );
  }

  test_superConstructor_redirectingConstructor() async {
    await assertNoErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  A();
  A.two();
}
''',
      code: r'''
class B extends A {
  B() : this.two();
  B.two() : super.two();
}
''',
    );
  }

  test_superConstructor_unnamedConstructor() async {
    await assertErrorsInCode2(
      externalCode: r'''
class A {
  @deprecated
  A() {}
}
''',
      code: r'''
class B extends A {
  B() : super() {}
}
''',
      [
        error(
          HintCode.deprecatedMemberUse,
          57,
          7,
          text: "'A' is deprecated and shouldn't be used.",
        ),
      ],
    );
  }

  test_topLevelVariable_argument() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int x = 1;
''',
      code: r'''
void f() {
  print(x);
}
''',
      [error(HintCode.deprecatedMemberUse, 48, 1)],
    );
  }

  test_topLevelVariable_assignment_right() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int x = 1;
''',
      code: r'''
void f(int a) {
  a = x;
}
''',
      [error(HintCode.deprecatedMemberUse, 51, 1)],
    );
  }

  test_topLevelVariable_binaryExpression() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int x = 1;
''',
      code: r'''
void f() {
  x + 1;
}
''',
      [error(HintCode.deprecatedMemberUse, 42, 1)],
    );
  }

  test_topLevelVariable_constructorFieldInitializer() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
const int x = 1;
''',
      code: r'''
class A {
  final int f;
  A() : f = x;
}
''',
      [error(HintCode.deprecatedMemberUse, 66, 1)],
    );
  }

  test_topLevelVariable_expressionFunctionBody() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int x = 1;
''',
      code: r'''
int f() => x;
''',
      [error(HintCode.deprecatedMemberUse, 40, 1)],
    );
  }

  test_topLevelVariable_expressionStatement() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int x = 1;
''',
      code: r'''
void f() {
  x;
}
''',
      [error(HintCode.deprecatedMemberUse, 42, 1)],
    );
  }

  test_topLevelVariable_forElement_condition() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
var x = true;
''',
      code: r'''
void f() {
  [for (;x;) 0];
}
''',
      [error(HintCode.deprecatedMemberUse, 49, 1)],
    );
  }

  test_topLevelVariable_forStatement_condition() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
var x = true;
''',
      code: r'''
void f() {
  for (;x;) {}
}
''',
      [error(HintCode.deprecatedMemberUse, 48, 1)],
    );
  }

  test_topLevelVariable_ifElement_condition() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
var x = true;
''',
      code: r'''
void f() {
  [if (x) 0];
}
''',
      [error(HintCode.deprecatedMemberUse, 47, 1)],
    );
  }

  test_topLevelVariable_ifStatement_condition() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
var x = true;
''',
      code: r'''
void f() {
  if (x) {}
}
''',
      [error(HintCode.deprecatedMemberUse, 46, 1)],
    );
  }

  test_topLevelVariable_listLiteral() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int x = 1;
''',
      code: r'''
void f() {
  [x];
}
''',
      [error(HintCode.deprecatedMemberUse, 43, 1)],
    );
  }

  test_topLevelVariable_mapLiteralEntry() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int x = 1;
''',
      code: r'''
void f() {
  ({0: x, x: 0});
}
''',
      [
        error(HintCode.deprecatedMemberUse, 47, 1),
        error(HintCode.deprecatedMemberUse, 50, 1),
      ],
    );
  }

  test_topLevelVariable_namedExpression() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int x = 1;
''',
      code: r'''
void g({int a = 0}) {}
void f() {
  g(a: x);
}
''',
      [error(HintCode.deprecatedMemberUse, 70, 1)],
    );
  }

  test_topLevelVariable_returnStatement() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int x = 1;
''',
      code: r'''
int f() {
  return x;
}
''',
      [error(HintCode.deprecatedMemberUse, 48, 1)],
    );
  }

  test_topLevelVariable_setLiteral() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int x = 1;
''',
      code: r'''
void f() {
  ({x});
}
''',
      [error(HintCode.deprecatedMemberUse, 44, 1)],
    );
  }

  test_topLevelVariable_spreadElement() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
var x = [0];
''',
      code: r'''
void f() {
  [...x];
}
''',
      [error(HintCode.deprecatedMemberUse, 46, 1)],
    );
  }

  test_topLevelVariable_switchCase() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
const int x = 1;
''',
      code: r'''
void f(int a) {
  switch (a) {
    case x:
      break;
  }
}
''',
      [error(HintCode.deprecatedMemberUse, 69, 1)],
    );
  }

  test_topLevelVariable_switchCase_language219() async {
    newFile(externalLibPath, r'''
@deprecated
const int x = 1;
''');
    await assertErrorsInCode(
      '''
// @dart = 2.19
import '$externalLibUri';
void f(int a) {
  switch (a) {
    case x:
      break;
  }
}
''',
      [error(HintCode.deprecatedMemberUse, 85, 1)],
    );
  }

  test_topLevelVariable_switchStatement() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int x = 1;
''',
      code: r'''
void f() {
  switch (x) {}
}
''',
      [error(HintCode.deprecatedMemberUse, 50, 1)],
    );
  }

  test_topLevelVariable_switchStatement_language219() async {
    newFile(externalLibPath, r'''
@deprecated
int x = 1;
''');
    await assertErrorsInCode(
      '''
// @dart = 2.19
import '$externalLibUri';
void f() {
  switch (x) {}
}
''',
      [error(HintCode.deprecatedMemberUse, 66, 1)],
    );
  }

  test_topLevelVariable_unaryExpression() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
int x = 1;
''',
      code: r'''
void f() {
  -x;
}
''',
      [error(HintCode.deprecatedMemberUse, 43, 1)],
    );
  }

  test_topLevelVariable_variableDeclaration_initializer() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
var x = 1;
''',
      code: r'''
void f() {
  // ignore: unused_local_variable
  var v = x;
}
''',
      [error(HintCode.deprecatedMemberUse, 85, 1)],
    );
  }

  test_topLevelVariable_whileStatement_condition() async {
    await assertErrorsInCode2(
      externalCode: r'''
@deprecated
var x = true;
''',
      code: r'''
void f() {
  while (x) {}
}
''',
      [error(HintCode.deprecatedMemberUse, 49, 1)],
    );
  }

  @override
  void verifyCreatedCollection() {
    super.verifyCreatedCollection();
    assertPackageConfigWorkspaceFor(testFile);
  }
}

class _PackageConfigWorkspaceBase extends PubPackageResolutionTest {
  String get testPackageGeneratedPath {
    return '$testPackageRootPath/.dart_tool/build/generated';
  }

  @override
  void verifyCreatedCollection() {
    super.verifyCreatedCollection();
    assertPackageConfigWorkspaceFor(testFile);
  }

  void _createTestPackageBuildMarker() {
    newFolder(testPackageGeneratedPath);
  }

  void _newTestPackageGeneratedFile({
    required String packageName,
    required String pathInLib,
    required String content,
  }) {
    newFile('$testPackageGeneratedPath/$packageName/lib/$pathInLib', content);
  }
}
