// Copyright 2014 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'package:args/command_runner.dart';
import 'package:file/memory.dart';
import 'package:flutter_tools/src/artifacts.dart';
import 'package:flutter_tools/src/base/file_system.dart';
import 'package:flutter_tools/src/base/logger.dart';
import 'package:flutter_tools/src/base/platform.dart';
import 'package:flutter_tools/src/build_info.dart';
import 'package:flutter_tools/src/build_system/build_system.dart';
import 'package:flutter_tools/src/cache.dart';
import 'package:flutter_tools/src/commands/build_ios_framework.dart';
import 'package:flutter_tools/src/commands/build_macos_framework.dart';
import 'package:flutter_tools/src/version.dart';

import '../../src/common.dart';
import '../../src/context.dart';
import '../../src/fake_process_manager.dart';
import '../../src/fakes.dart';
import '../../src/test_build_system.dart';
import '../../src/test_flutter_command_runner.dart';

void main() {
  late MemoryFileSystem memoryFileSystem;
  late Directory outputDirectory;
  late FakePlatform fakePlatform;

  setUpAll(() {
    Cache.disableLocking();
  });

  const storageBaseUrl = 'https://fake.googleapis.com';
  setUp(() {
    memoryFileSystem = MemoryFileSystem.test();
    fakePlatform = FakePlatform(
      operatingSystem: 'macos',
      environment: <String, String>{'FLUTTER_STORAGE_BASE_URL': storageBaseUrl},
    );

    outputDirectory =
        memoryFileSystem.systemTempDirectory
            .createTempSync('flutter_build_framework_test_output.')
            .childDirectory('Debug')
          ..createSync();
  });

  group('build ios-framework', () {
    group('podspec', () {
      const engineRevision = '0123456789abcdef';
      late Cache cache;

      setUp(() {
        final Directory rootOverride = memoryFileSystem.directory('cache');
        cache = Cache.test(
          rootOverride: rootOverride,
          platform: fakePlatform,
          fileSystem: memoryFileSystem,
          processManager: FakeProcessManager.any(),
        );
        rootOverride.childDirectory('bin').childDirectory('cache').childFile('engine.stamp')
          ..createSync(recursive: true)
          ..writeAsStringSync(engineRevision);
      });

      testUsingContext(
        'version unknown',
        () async {
          const frameworkVersion = '0.0.0-unknown';
          final fakeFlutterVersion = FakeFlutterVersion(frameworkVersion: frameworkVersion);

          final command = BuildIOSFrameworkCommand(
            logger: BufferLogger.test(),
            buildSystem: TestBuildSystem.all(BuildResult(success: true)),
            platform: fakePlatform,
            flutterVersion: fakeFlutterVersion,
            cache: cache,
            verboseHelp: false,
          );

          expect(
            () => command.produceFlutterPodspec(BuildMode.debug, outputDirectory),
            throwsToolExit(
              message:
                  '--cocoapods is only supported on the beta or stable channel. Detected version is $frameworkVersion',
            ),
          );
        },
        overrides: <Type, Generator>{
          FileSystem: () => memoryFileSystem,
          ProcessManager: () => FakeProcessManager.any(),
        },
      );

      testUsingContext(
        'throws when not on a released version',
        () async {
          const frameworkVersion = 'v1.13.10+hotfix-pre.2';
          const gitTagVersion = GitTagVersion(
            x: 1,
            y: 13,
            z: 10,
            hotfix: 13,
            commits: 2,
            hash: '',
            gitTag: frameworkVersion,
          );
          final fakeFlutterVersion = FakeFlutterVersion(
            gitTagVersion: gitTagVersion,
            frameworkVersion: frameworkVersion,
          );

          final command = BuildIOSFrameworkCommand(
            logger: BufferLogger.test(),
            buildSystem: TestBuildSystem.all(BuildResult(success: true)),
            platform: fakePlatform,
            flutterVersion: fakeFlutterVersion,
            cache: cache,
            verboseHelp: false,
          );

          expect(
            () => command.produceFlutterPodspec(BuildMode.debug, outputDirectory),
            throwsToolExit(
              message:
                  '--cocoapods is only supported on the beta or stable channel. Detected version is $frameworkVersion',
            ),
          );
        },
        overrides: <Type, Generator>{
          FileSystem: () => memoryFileSystem,
          ProcessManager: () => FakeProcessManager.any(),
        },
      );

      testUsingContext(
        'throws when license not found',
        () async {
          final fakeFlutterVersion = FakeFlutterVersion(
            gitTagVersion: const GitTagVersion(
              x: 1,
              y: 13,
              z: 10,
              hotfix: 13,
              commits: 0,
              hash: '',
              gitTag: '1.13.10+hotfix.14.0',
            ),
          );

          final command = BuildIOSFrameworkCommand(
            logger: BufferLogger.test(),
            buildSystem: TestBuildSystem.all(BuildResult(success: true)),
            platform: fakePlatform,
            flutterVersion: fakeFlutterVersion,
            cache: cache,
            verboseHelp: false,
          );

          expect(
            () => command.produceFlutterPodspec(BuildMode.debug, outputDirectory),
            throwsToolExit(message: 'Could not find license'),
          );
        },
        overrides: <Type, Generator>{
          FileSystem: () => memoryFileSystem,
          ProcessManager: () => FakeProcessManager.any(),
        },
      );

      group('is created', () {
        const frameworkVersion = 'v1.13.11+hotfix.14';
        const licenseText = 'This is the license!';

        setUp(() {
          // cache.getLicenseFile() relies on the flutter root being set.
          Cache.flutterRoot ??= getFlutterRoot();
          cache.getLicenseFile()
            ..createSync(recursive: true)
            ..writeAsStringSync(licenseText);
        });

        group('on master channel', () {
          testUsingContext(
            'created when forced',
            () async {
              const frameworkVersionWithCommits = '$frameworkVersion.pre.100';
              const gitTagVersion = GitTagVersion(
                x: 1,
                y: 13,
                z: 11,
                hotfix: 13,
                commits: 100,
                hash: '',
                gitTag: frameworkVersionWithCommits,
              );
              final fakeFlutterVersion = FakeFlutterVersion(
                gitTagVersion: gitTagVersion,
                frameworkVersion: frameworkVersionWithCommits,
              );

              final command = BuildIOSFrameworkCommand(
                logger: BufferLogger.test(),
                buildSystem: TestBuildSystem.all(BuildResult(success: true)),
                platform: fakePlatform,
                flutterVersion: fakeFlutterVersion,
                cache: cache,
                verboseHelp: false,
              );
              command.produceFlutterPodspec(BuildMode.debug, outputDirectory, force: true);

              final File expectedPodspec = outputDirectory.childFile('Flutter.podspec');
              expect(expectedPodspec.existsSync(), isTrue);
            },
            overrides: <Type, Generator>{
              FileSystem: () => memoryFileSystem,
              ProcessManager: () => FakeProcessManager.any(),
            },
          );
        });

        group('not on master channel', () {
          late FakeFlutterVersion fakeFlutterVersion;
          setUp(() {
            const frameworkVersionWithCommits = '$frameworkVersion.pre.0';
            const gitTagVersion = GitTagVersion(
              x: 1,
              y: 13,
              z: 11,
              hotfix: 13,
              commits: 0,
              hash: '',
              gitTag: frameworkVersionWithCommits,
            );
            fakeFlutterVersion = FakeFlutterVersion(
              gitTagVersion: gitTagVersion,
              frameworkVersion: frameworkVersionWithCommits,
            );
          });

          testUsingContext(
            'contains license and version',
            () async {
              final command = BuildIOSFrameworkCommand(
                logger: BufferLogger.test(),
                buildSystem: TestBuildSystem.all(BuildResult(success: true)),
                platform: fakePlatform,
                flutterVersion: fakeFlutterVersion,
                cache: cache,
                verboseHelp: false,
              );
              command.produceFlutterPodspec(BuildMode.debug, outputDirectory);

              final File expectedPodspec = outputDirectory.childFile('Flutter.podspec');
              final String podspecContents = expectedPodspec.readAsStringSync();
              expect(podspecContents, contains("'1.13.1113'"));
              expect(podspecContents, contains('# $frameworkVersion'));
              expect(podspecContents, contains(licenseText));
            },
            overrides: <Type, Generator>{
              FileSystem: () => memoryFileSystem,
              ProcessManager: () => FakeProcessManager.any(),
            },
          );

          testUsingContext(
            'debug URL',
            () async {
              final command = BuildIOSFrameworkCommand(
                logger: BufferLogger.test(),
                buildSystem: TestBuildSystem.all(BuildResult(success: true)),
                platform: fakePlatform,
                flutterVersion: fakeFlutterVersion,
                cache: cache,
                verboseHelp: false,
              );
              command.produceFlutterPodspec(BuildMode.debug, outputDirectory);

              final File expectedPodspec = outputDirectory.childFile('Flutter.podspec');
              final String podspecContents = expectedPodspec.readAsStringSync();
              expect(
                podspecContents,
                contains(
                  "'$storageBaseUrl/flutter_infra_release/flutter/$engineRevision/ios/artifacts.zip'",
                ),
              );
            },
            overrides: <Type, Generator>{
              FileSystem: () => memoryFileSystem,
              ProcessManager: () => FakeProcessManager.any(),
            },
          );

          testUsingContext(
            'profile URL',
            () async {
              final command = BuildIOSFrameworkCommand(
                logger: BufferLogger.test(),
                buildSystem: TestBuildSystem.all(BuildResult(success: true)),
                platform: fakePlatform,
                flutterVersion: fakeFlutterVersion,
                cache: cache,
                verboseHelp: false,
              );
              command.produceFlutterPodspec(BuildMode.profile, outputDirectory);

              final File expectedPodspec = outputDirectory.childFile('Flutter.podspec');
              final String podspecContents = expectedPodspec.readAsStringSync();
              expect(
                podspecContents,
                contains(
                  "'$storageBaseUrl/flutter_infra_release/flutter/$engineRevision/ios-profile/artifacts.zip'",
                ),
              );
            },
            overrides: <Type, Generator>{
              FileSystem: () => memoryFileSystem,
              ProcessManager: () => FakeProcessManager.any(),
            },
          );

          testUsingContext(
            'release URL',
            () async {
              final command = BuildIOSFrameworkCommand(
                logger: BufferLogger.test(),
                buildSystem: TestBuildSystem.all(BuildResult(success: true)),
                platform: fakePlatform,
                flutterVersion: fakeFlutterVersion,
                cache: cache,
                verboseHelp: false,
              );
              command.produceFlutterPodspec(BuildMode.release, outputDirectory);

              final File expectedPodspec = outputDirectory.childFile('Flutter.podspec');
              final String podspecContents = expectedPodspec.readAsStringSync();
              expect(
                podspecContents,
                contains(
                  "'$storageBaseUrl/flutter_infra_release/flutter/$engineRevision/ios-release/artifacts.zip'",
                ),
              );
            },
            overrides: <Type, Generator>{
              FileSystem: () => memoryFileSystem,
              ProcessManager: () => FakeProcessManager.any(),
            },
          );
        });
      });
    });
  });

  group('native assets', () {
    late FakeFlutterVersion fakeFlutterVersion;
    late Cache cache;

    setUp(() {
      fakeFlutterVersion = FakeFlutterVersion(
        frameworkVersion: '1.13.11',
        gitTagVersion: const GitTagVersion(
          x: 1,
          y: 13,
          z: 11,
          commits: 0,
          hash: 'abcdef',
          gitTag: '1.13.11',
        ),
      );
      final Directory rootOverride = memoryFileSystem.directory('cache');
      cache = Cache.test(
        rootOverride: rootOverride,
        platform: fakePlatform,
        fileSystem: memoryFileSystem,
        processManager: FakeProcessManager.any(),
      );
    });

    testUsingContext(
      'throws if simulator contains extra assets',
      () async {
        final Directory projectDir = memoryFileSystem.directory('project')..createSync();
        projectDir.childDirectory('ios').createSync();
        projectDir.childDirectory('ios').childDirectory('Pods').createSync();
        projectDir.childDirectory('lib').childFile('main.dart').createSync(recursive: true);
        projectDir.childDirectory('.dart_tool').childFile('package_config.json')
          ..createSync(recursive: true)
          ..writeAsStringSync(
            '{"configVersion": 2, "packages": [{"name": "project", "rootUri": "../", "packageUri": "lib/", "languageVersion": "3.0"}]}',
          );
        projectDir
            .childDirectory('.dart_tool')
            .childFile('package_graph.json')
            .writeAsStringSync(
              '{"configVersion": 1, "packages": [{"name": "project", "rootUri": "..", "packageUri": "lib/", "dependencies": []}]}',
            );
        projectDir.childFile('pubspec.yaml').writeAsStringSync('name: project');
        projectDir.childFile('.metadata').createSync();
        memoryFileSystem.currentDirectory = projectDir;

        final command = BuildIOSFrameworkCommand(
          logger: BufferLogger.test(),
          buildSystem: TestBuildSystem.all(BuildResult(success: true), (
            Target target,
            Environment environment,
          ) {
            final Directory output = environment.outputDir;
            output.childDirectory('App.framework').childFile('App').createSync(recursive: true);
            final File manifest = output
                .childDirectory('App.framework')
                .childDirectory('flutter_assets')
                .childFile('NativeAssetsManifest.json');
            manifest.createSync(recursive: true);
            if (output.path.contains('iphoneos')) {
              manifest.writeAsStringSync('{"format-version": [1, 0, 0], "native-assets": {}}');
            } else {
              manifest.writeAsStringSync(
                '{"format-version": [1, 0, 0], "native-assets": {"ios_x64": {"package:project/asset1": ["absolute", "Foo.framework/Foo"]}}}',
              );
            }
          }),
          platform: fakePlatform,
          flutterVersion: fakeFlutterVersion,
          cache: cache,
          verboseHelp: false,
        );

        // Mock engine artifacts. _TestArtifacts uses a string like this for getArtifactPath.
        memoryFileSystem
            .directory('Artifact.flutterXcframework.TargetPlatform.ios.debug')
            .createSync(recursive: true);

        final Directory buildDir =
            projectDir.childDirectory('.dart_tool').childDirectory('flutter_build')
              ..createSync(recursive: true);
        buildDir
            .childFile('dart_build_result.json')
            .writeAsStringSync('{"codeAssets": [], "dataAssets": [], "dependencies": []}');

        final CommandRunner<void> runner = createTestCommandRunner(command);
        await expectLater(
          () => runner.run(<String>[
            'ios-framework',
            '--no-pub',
            '--no-plugins',
            '--no-profile',
            '--no-release',
          ]),
          throwsToolExit(
            message:
                'The simulator build contains a code asset "package:project/asset1" that is not present in the physical device build.',
          ),
        );
      },
      overrides: <Type, Generator>{
        FileSystem: () => memoryFileSystem,
        ProcessManager: () => FakeProcessManager.any(),
        Artifacts: () => Artifacts.test(fileSystem: memoryFileSystem),
      },
    );

    testUsingContext(
      'throws if framework names are inconsistent',
      () async {
        final Directory projectDir = memoryFileSystem.directory('project')..createSync();
        projectDir.childDirectory('ios').createSync();
        projectDir.childDirectory('ios').childDirectory('Pods').createSync();
        projectDir.childDirectory('lib').childFile('main.dart').createSync(recursive: true);
        projectDir.childDirectory('.dart_tool').childFile('package_config.json')
          ..createSync(recursive: true)
          ..writeAsStringSync(
            '{"configVersion": 2, "packages": [{"name": "project", "rootUri": "../", "packageUri": "lib/", "languageVersion": "3.0"}]}',
          );
        projectDir
            .childDirectory('.dart_tool')
            .childFile('package_graph.json')
            .writeAsStringSync(
              '{"configVersion": 1, "packages": [{"name": "project", "rootUri": "..", "packageUri": "lib/", "dependencies": []}]}',
            );
        projectDir.childFile('pubspec.yaml').writeAsStringSync('name: project');
        projectDir.childFile('.metadata').createSync();
        memoryFileSystem.currentDirectory = projectDir;

        final command = BuildIOSFrameworkCommand(
          logger: BufferLogger.test(),
          buildSystem: TestBuildSystem.all(BuildResult(success: true), (
            Target target,
            Environment environment,
          ) {
            final Directory output = environment.outputDir;
            output.childDirectory('App.framework').childFile('App').createSync(recursive: true);
            final File manifest = output
                .childDirectory('App.framework')
                .childDirectory('flutter_assets')
                .childFile('NativeAssetsManifest.json');
            manifest.createSync(recursive: true);
            if (output.path.contains('iphoneos')) {
              manifest.writeAsStringSync(
                '{"format-version": [1, 0, 0], "native-assets": {"ios_arm64": {"package:project/asset1": ["absolute", "Foo.framework/Foo"]}}}',
              );
            } else {
              manifest.writeAsStringSync(
                '{"format-version": [1, 0, 0], "native-assets": {"ios_x64": {"package:project/asset1": ["absolute", "Bar.framework/Bar"]}}}',
              );
            }
          }),
          platform: fakePlatform,
          flutterVersion: fakeFlutterVersion,
          cache: cache,
          verboseHelp: false,
        );

        // Mock engine artifacts
        memoryFileSystem
            .directory('Artifact.flutterXcframework.TargetPlatform.ios.debug')
            .createSync(recursive: true);

        final Directory buildDir =
            projectDir.childDirectory('.dart_tool').childDirectory('flutter_build')
              ..createSync(recursive: true);
        buildDir
            .childFile('dart_build_result.json')
            .writeAsStringSync('{"codeAssets": [], "dataAssets": [], "dependencies": []}');

        final CommandRunner<void> runner = createTestCommandRunner(command);
        await expectLater(
          () => runner.run(<String>[
            'ios-framework',
            '--no-pub',
            '--no-plugins',
            '--no-profile',
            '--no-release',
          ]),
          throwsToolExit(
            message:
                'Consistent code asset framework names are required for XCFramework creation.\n'
                'The asset "package:project/asset1" has different framework paths across platforms:',
          ),
        );
      },
      overrides: <Type, Generator>{
        FileSystem: () => memoryFileSystem,
        ProcessManager: () => FakeProcessManager.any(),
        Artifacts: () => Artifacts.test(fileSystem: memoryFileSystem),
      },
    );
  });

  group('build macos-framework', () {
    group('podspec', () {
      const engineRevision = '0123456789abcdef';
      late Cache cache;

      setUp(() {
        final Directory rootOverride = memoryFileSystem.directory('cache');
        cache = Cache.test(
          rootOverride: rootOverride,
          platform: fakePlatform,
          fileSystem: memoryFileSystem,
          processManager: FakeProcessManager.any(),
        );
        rootOverride.childDirectory('bin').childDirectory('cache').childFile('engine.stamp')
          ..createSync(recursive: true)
          ..writeAsStringSync(engineRevision);
      });

      testUsingContext(
        'version unknown',
        () async {
          const frameworkVersion = '0.0.0-unknown';
          final fakeFlutterVersion = FakeFlutterVersion(frameworkVersion: frameworkVersion);

          final command = BuildMacOSFrameworkCommand(
            logger: BufferLogger.test(),
            buildSystem: TestBuildSystem.all(BuildResult(success: true)),
            platform: fakePlatform,
            flutterVersion: fakeFlutterVersion,
            cache: cache,
            verboseHelp: false,
          );

          expect(
            () => command.produceFlutterPodspec(BuildMode.debug, outputDirectory),
            throwsToolExit(
              message:
                  '--cocoapods is only supported on the beta or stable channel. Detected version is $frameworkVersion',
            ),
          );
        },
        overrides: <Type, Generator>{
          FileSystem: () => memoryFileSystem,
          ProcessManager: () => FakeProcessManager.any(),
        },
      );

      testUsingContext(
        'throws when not on a released version',
        () async {
          const frameworkVersion = 'v1.13.10+hotfix.14.pre.2';
          const gitTagVersion = GitTagVersion(
            x: 1,
            y: 13,
            z: 10,
            hotfix: 13,
            commits: 2,
            hash: '',
            gitTag: frameworkVersion,
          );
          final fakeFlutterVersion = FakeFlutterVersion(
            gitTagVersion: gitTagVersion,
            frameworkVersion: frameworkVersion,
          );

          final command = BuildMacOSFrameworkCommand(
            logger: BufferLogger.test(),
            buildSystem: TestBuildSystem.all(BuildResult(success: true)),
            platform: fakePlatform,
            flutterVersion: fakeFlutterVersion,
            cache: cache,
            verboseHelp: false,
          );

          expect(
            () => command.produceFlutterPodspec(BuildMode.debug, outputDirectory),
            throwsToolExit(
              message:
                  '--cocoapods is only supported on the beta or stable channel. Detected version is $frameworkVersion',
            ),
          );
        },
        overrides: <Type, Generator>{
          FileSystem: () => memoryFileSystem,
          ProcessManager: () => FakeProcessManager.any(),
        },
      );

      testUsingContext(
        'throws when license not found',
        () async {
          final fakeFlutterVersion = FakeFlutterVersion(
            gitTagVersion: const GitTagVersion(
              x: 1,
              y: 13,
              z: 10,
              hotfix: 13,
              commits: 0,
              hash: '',
              gitTag: '1.13.10+hotfix.14.pre.0',
            ),
          );

          final command = BuildMacOSFrameworkCommand(
            logger: BufferLogger.test(),
            buildSystem: TestBuildSystem.all(BuildResult(success: true)),
            platform: fakePlatform,
            flutterVersion: fakeFlutterVersion,
            cache: cache,
            verboseHelp: false,
          );

          expect(
            () => command.produceFlutterPodspec(BuildMode.debug, outputDirectory),
            throwsToolExit(message: 'Could not find license'),
          );
        },
        overrides: <Type, Generator>{
          FileSystem: () => memoryFileSystem,
          ProcessManager: () => FakeProcessManager.any(),
        },
      );

      group('is created', () {
        const frameworkVersion = 'v1.13.11+hotfix.13';
        const licenseText = 'This is the license!';

        setUp(() {
          // cache.getLicenseFile() relies on the flutter root being set.
          Cache.flutterRoot ??= getFlutterRoot();
          cache.getLicenseFile()
            ..createSync(recursive: true)
            ..writeAsStringSync(licenseText);
        });

        group('on master channel', () {
          testUsingContext(
            'created when forced',
            () async {
              const gitTagVersion = GitTagVersion(
                x: 1,
                y: 13,
                z: 11,
                hotfix: 13,
                commits: 100,
                hash: '',
                gitTag: '$frameworkVersion.pre.100',
              );
              final fakeFlutterVersion = FakeFlutterVersion(
                gitTagVersion: gitTagVersion,
                frameworkVersion: frameworkVersion,
              );

              final command = BuildMacOSFrameworkCommand(
                logger: BufferLogger.test(),
                buildSystem: TestBuildSystem.all(BuildResult(success: true)),
                platform: fakePlatform,
                flutterVersion: fakeFlutterVersion,
                cache: cache,
                verboseHelp: false,
              );
              command.produceFlutterPodspec(BuildMode.debug, outputDirectory, force: true);

              final File expectedPodspec = outputDirectory.childFile('FlutterMacOS.podspec');
              expect(expectedPodspec.existsSync(), isTrue);
            },
            overrides: <Type, Generator>{
              FileSystem: () => memoryFileSystem,
              ProcessManager: () => FakeProcessManager.any(),
            },
          );
        });

        group('not on master channel', () {
          late FakeFlutterVersion fakeFlutterVersion;
          setUp(() {
            const gitTagVersion = GitTagVersion(
              x: 1,
              y: 13,
              z: 11,
              hotfix: 13,
              commits: 0,
              hash: '',
              gitTag: '$frameworkVersion.pre.0',
            );
            fakeFlutterVersion = FakeFlutterVersion(
              gitTagVersion: gitTagVersion,
              frameworkVersion: frameworkVersion,
            );
          });

          testUsingContext(
            'contains license and version',
            () async {
              final command = BuildMacOSFrameworkCommand(
                logger: BufferLogger.test(),
                buildSystem: TestBuildSystem.all(BuildResult(success: true)),
                platform: fakePlatform,
                flutterVersion: fakeFlutterVersion,
                cache: cache,
                verboseHelp: false,
              );
              command.produceFlutterPodspec(BuildMode.debug, outputDirectory);

              final File expectedPodspec = outputDirectory.childFile('FlutterMacOS.podspec');
              final String podspecContents = expectedPodspec.readAsStringSync();
              expect(podspecContents, contains("'1.13.1113'"));
              expect(podspecContents, contains('# $frameworkVersion'));
              expect(podspecContents, contains(licenseText));
            },
            overrides: <Type, Generator>{
              FileSystem: () => memoryFileSystem,
              ProcessManager: () => FakeProcessManager.any(),
            },
          );

          testUsingContext(
            'debug URL',
            () async {
              final command = BuildMacOSFrameworkCommand(
                logger: BufferLogger.test(),
                buildSystem: TestBuildSystem.all(BuildResult(success: true)),
                platform: fakePlatform,
                flutterVersion: fakeFlutterVersion,
                cache: cache,
                verboseHelp: false,
              );
              command.produceFlutterPodspec(BuildMode.debug, outputDirectory);

              final File expectedPodspec = outputDirectory.childFile('FlutterMacOS.podspec');
              final String podspecContents = expectedPodspec.readAsStringSync();
              expect(
                podspecContents,
                contains(
                  "'$storageBaseUrl/flutter_infra_release/flutter/$engineRevision/darwin-x64/FlutterMacOS.framework.zip'",
                ),
              );
            },
            overrides: <Type, Generator>{
              FileSystem: () => memoryFileSystem,
              ProcessManager: () => FakeProcessManager.any(),
            },
          );

          testUsingContext(
            'profile URL',
            () async {
              final command = BuildMacOSFrameworkCommand(
                logger: BufferLogger.test(),
                buildSystem: TestBuildSystem.all(BuildResult(success: true)),
                platform: fakePlatform,
                flutterVersion: fakeFlutterVersion,
                cache: cache,
                verboseHelp: false,
              );
              command.produceFlutterPodspec(BuildMode.profile, outputDirectory);

              final File expectedPodspec = outputDirectory.childFile('FlutterMacOS.podspec');
              final String podspecContents = expectedPodspec.readAsStringSync();
              expect(
                podspecContents,
                contains(
                  "'$storageBaseUrl/flutter_infra_release/flutter/$engineRevision/darwin-x64-profile/FlutterMacOS.framework.zip'",
                ),
              );
            },
            overrides: <Type, Generator>{
              FileSystem: () => memoryFileSystem,
              ProcessManager: () => FakeProcessManager.any(),
            },
          );

          testUsingContext(
            'release URL',
            () async {
              final command = BuildMacOSFrameworkCommand(
                logger: BufferLogger.test(),
                buildSystem: TestBuildSystem.all(BuildResult(success: true)),
                platform: fakePlatform,
                flutterVersion: fakeFlutterVersion,
                cache: cache,
                verboseHelp: false,
              );
              command.produceFlutterPodspec(BuildMode.release, outputDirectory);

              final File expectedPodspec = outputDirectory.childFile('FlutterMacOS.podspec');
              final String podspecContents = expectedPodspec.readAsStringSync();
              expect(
                podspecContents,
                contains(
                  "'$storageBaseUrl/flutter_infra_release/flutter/$engineRevision/darwin-x64-release/FlutterMacOS.framework.zip'",
                ),
              );
            },
            overrides: <Type, Generator>{
              FileSystem: () => memoryFileSystem,
              ProcessManager: () => FakeProcessManager.any(),
            },
          );
        });
      });
    });
  });

  group('XCFrameworks', () {
    late MemoryFileSystem fileSystem;
    late FakeProcessManager fakeProcessManager;

    setUp(() {
      fileSystem = MemoryFileSystem.test();
      fakeProcessManager = FakeProcessManager.empty();
    });

    testWithoutContext('created', () async {
      final Directory frameworkA = fileSystem.directory('FrameworkA.framework')..createSync();
      final Directory frameworkB = fileSystem.directory('FrameworkB.framework')..createSync();
      final Directory output = fileSystem.directory('output');

      fakeProcessManager.addCommand(
        FakeCommand(
          command: <String>[
            'xcrun',
            'xcodebuild',
            '-create-xcframework',
            '-framework',
            frameworkA.path,
            '-framework',
            frameworkB.path,
            '-output',
            output.childDirectory('Combine.xcframework').path,
          ],
        ),
      );
      await BuildFrameworkCommand.produceXCFramework(
        <Directory>[frameworkA, frameworkB],
        'Combine',
        output,
        fakeProcessManager,
      );
      expect(fakeProcessManager, hasNoRemainingExpectations);
    });

    testWithoutContext('created with symbols', () async {
      final Directory parentA = fileSystem.directory('FrameworkA')..createSync();
      final File dSYMA = parentA.childFile('FrameworkA.framework.dSYM')..createSync();
      final Directory frameworkA = parentA.childDirectory('FrameworkA.framework')..createSync();
      // Flutter.framework.dSYM should be correctly filtered out.
      parentA.childFile('Flutter.framework.dSYM').createSync();

      final Directory parentB = fileSystem.directory('FrameworkB')..createSync();
      final File dSYMB = parentB.childFile('FrameworkB.framework.dSYM')..createSync();
      final Directory frameworkB = parentB.childDirectory('FrameworkB.framework')..createSync();
      final Directory output = fileSystem.directory('output');

      fakeProcessManager.addCommand(
        FakeCommand(
          command: <String>[
            'xcrun',
            'xcodebuild',
            '-create-xcframework',
            '-framework',
            frameworkA.path,
            '-debug-symbols',
            dSYMA.path,
            '-framework',
            frameworkB.path,
            '-debug-symbols',
            dSYMB.path,
            '-output',
            output.childDirectory('Combine.xcframework').path,
          ],
        ),
      );
      await BuildFrameworkCommand.produceXCFramework(
        <Directory>[frameworkA, frameworkB],
        'Combine',
        output,
        fakeProcessManager,
      );
      expect(fakeProcessManager, hasNoRemainingExpectations);
    });
  });

  group('findCodeAssetFrameworkNames', () {
    late MemoryFileSystem fileSystem;

    setUp(() {
      fileSystem = MemoryFileSystem.test();
    });

    testWithoutContext('finds frameworks in native_assets', () {
      final Directory output = fileSystem.directory('output')..createSync();
      final Directory nativeAssets = output.childDirectory('native_assets')..createSync();
      nativeAssets.childDirectory('foo.framework').createSync();
      nativeAssets.childDirectory('bar.framework').createSync();
      nativeAssets.childDirectory('baz.framework.dSYM').createSync(); // Should be ignored
      nativeAssets.childFile('something_else').createSync(); // Should be ignored

      final Iterable<String> names = BuildFrameworkCommand.findCodeAssetFrameworkNames(output);
      expect(names, unorderedEquals(<String>['foo.framework', 'bar.framework']));
    });

    testWithoutContext('returns empty if native_assets does not exist', () {
      final Directory output = fileSystem.directory('output')..createSync();
      final Iterable<String> names = BuildFrameworkCommand.findCodeAssetFrameworkNames(output);
      expect(names, isEmpty);
    });

    testWithoutContext('parseNativeAssetsManifest', () {
      final Directory output = fileSystem.directory('output')..createSync();
      final File manifest = output
          .childDirectory('App.framework')
          .childDirectory('flutter_assets')
          .childFile('NativeAssetsManifest.json');
      manifest.createSync(recursive: true);
      manifest.writeAsStringSync('''
{
  "format-version": [1, 0, 0],
  "native-assets": {
    "ios_arm64": {
      "package:project/asset1": ["absolute", "Foo.framework/Foo"],
      "package:project/asset2": ["absolute", "Bar.framework/Bar"]
    },
    "ios_x64": {
      "package:project/asset1": ["absolute", "Foo.framework/Foo"]
    }
  }
}
''');

      final Map<String, String> assets = BuildFrameworkCommand.parseNativeAssetsManifest(output);
      expect(assets, <String, String>{
        'package:project/asset1': 'Foo.framework/Foo',
        'package:project/asset2': 'Bar.framework/Bar',
      });
    });
  });
}
