// Copyright 2014 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import '../artifacts.dart';
import '../base/error_handling_io.dart';
import '../base/file_system.dart';
import '../base/template.dart';
import '../base/version.dart';
import '../build_info.dart';
import '../darwin/darwin.dart';
import '../plugins.dart';
import '../xcode_project.dart';
import 'swift_packages.dart';

/// The name of the Swift package that's generated by the Flutter tool to add
/// dependencies on Flutter plugin swift packages.
const kFlutterGeneratedPluginSwiftPackageName = 'FlutterGeneratedPluginSwiftPackage';

/// The name of the Swift pacakge that's generated by the Flutter tool to add
/// a dependency on the Flutter/FlutterMacOS framework.
const kFlutterGeneratedFrameworkSwiftPackageTargetName = 'FlutterFramework';

/// Swift Package Manager is a dependency management solution for iOS and macOS
/// applications.
///
/// See also:
///   * https://www.swift.org/documentation/package-manager/ - documentation on
///     Swift Package Manager.
///   * https://developer.apple.com/documentation/packagedescription/package -
///     documentation on Swift Package Manager manifest file, Package.swift.
class SwiftPackageManager {
  const SwiftPackageManager({
    required Artifacts artifacts,
    required FileSystem fileSystem,
    required TemplateRenderer templateRenderer,
  }) : _artifacts = artifacts,
       _fileSystem = fileSystem,
       _templateRenderer = templateRenderer;

  final Artifacts _artifacts;
  final FileSystem _fileSystem;
  final TemplateRenderer _templateRenderer;

  /// Creates a Swift Package called 'FlutterGeneratedPluginSwiftPackage' that
  /// has dependencies on Flutter plugins that are compatible with Swift
  /// Package Manager.
  Future<void> generatePluginsSwiftPackage(
    List<Plugin> plugins,
    FlutterDarwinPlatform platform,
    XcodeBasedProject project, {
    bool flutterAsADependency = true,
  }) async {
    final Directory symlinkDirectory = project.relativeSwiftPackagesDirectory;
    ErrorHandlingFileSystem.deleteIfExists(symlinkDirectory, recursive: true);
    symlinkDirectory.createSync(recursive: true);

    final (
      List<SwiftPackagePackageDependency> packageDependencies,
      List<SwiftPackageTargetDependency> targetDependencies,
    ) = _dependenciesForPlugins(
      plugins: plugins,
      platform: platform,
      symlinkDirectory: symlinkDirectory,
      pathRelativeTo: project.flutterPluginSwiftPackageDirectory.path,
    );

    // If there aren't any Swift Package plugins and the project hasn't been
    // migrated yet, don't generate a Swift package or migrate the app since
    // it's not needed. If the project has already been migrated, regenerate
    // the Package.swift even if there are no dependencies in case there
    // were dependencies previously.
    if (packageDependencies.isEmpty && !project.flutterPluginSwiftPackageInProjectSettings) {
      return;
    }

    // Add Flutter framework Swift package dependency
    if (flutterAsADependency) {
      final (
        SwiftPackagePackageDependency flutterFrameworkPackageDependency,
        SwiftPackageTargetDependency flutterFrameworkTargetDependency,
      ) = _dependencyForFlutterFramework(
        pathRelativeTo: project.flutterPluginSwiftPackageDirectory.path,
        platform: platform,
        project: project,
      );
      packageDependencies.add(flutterFrameworkPackageDependency);
      targetDependencies.add(flutterFrameworkTargetDependency);
    }

    // FlutterGeneratedPluginSwiftPackage must be statically linked to ensure
    // any dynamic dependencies are linked to Runner and prevent undefined symbols.
    final generatedProduct = SwiftPackageProduct(
      name: kFlutterGeneratedPluginSwiftPackageName,
      targets: <String>[kFlutterGeneratedPluginSwiftPackageName],
      libraryType: SwiftPackageLibraryType.static,
    );

    final generatedTarget = SwiftPackageTarget.defaultTarget(
      name: kFlutterGeneratedPluginSwiftPackageName,
      dependencies: targetDependencies,
    );

    final pluginsPackage = SwiftPackage(
      manifest: project.flutterPluginSwiftPackageManifest,
      name: kFlutterGeneratedPluginSwiftPackageName,
      platforms: <SwiftPackageSupportedPlatform>[platform.supportedPackagePlatform],
      products: <SwiftPackageProduct>[generatedProduct],
      dependencies: packageDependencies,
      targets: <SwiftPackageTarget>[generatedTarget],
      templateRenderer: _templateRenderer,
    );
    pluginsPackage.createSwiftPackage();
  }

  (List<SwiftPackagePackageDependency>, List<SwiftPackageTargetDependency>)
  _dependenciesForPlugins({
    required List<Plugin> plugins,
    required FlutterDarwinPlatform platform,
    required Directory symlinkDirectory,
    required String pathRelativeTo,
  }) {
    final packageDependencies = <SwiftPackagePackageDependency>[];
    final targetDependencies = <SwiftPackageTargetDependency>[];

    for (final plugin in plugins) {
      final String? pluginSwiftPackageManifestPath = plugin.pluginSwiftPackageManifestPath(
        _fileSystem,
        platform.name,
      );
      String? packagePath = plugin.pluginSwiftPackagePath(_fileSystem, platform.name);
      if (plugin.platforms[platform.name] == null ||
          pluginSwiftPackageManifestPath == null ||
          packagePath == null ||
          !_fileSystem.file(pluginSwiftPackageManifestPath).existsSync()) {
        continue;
      }

      final Link pluginSymlink = symlinkDirectory.childLink(plugin.name);
      ErrorHandlingFileSystem.deleteIfExists(pluginSymlink);
      pluginSymlink.createSync(packagePath);
      packagePath = pluginSymlink.path;
      packagePath = _fileSystem.path.relative(packagePath, from: pathRelativeTo);

      packageDependencies.add(SwiftPackagePackageDependency(name: plugin.name, path: packagePath));

      // The target dependency product name is hyphen separated because it's
      // the dependency's library name, which Swift Package Manager will
      // automatically use as the CFBundleIdentifier if linked dynamically. The
      // CFBundleIdentifier cannot contain underscores.
      targetDependencies.add(
        SwiftPackageTargetDependency.product(
          name: plugin.name.replaceAll('_', '-'),
          packageName: plugin.name,
        ),
      );
    }
    return (packageDependencies, targetDependencies);
  }

  /// Returns Flutter framework dependencies for the `FlutterGeneratedPluginSwiftPackage`.
  (SwiftPackagePackageDependency, SwiftPackageTargetDependency) _dependencyForFlutterFramework({
    required String pathRelativeTo,
    required FlutterDarwinPlatform platform,
    required XcodeBasedProject project,
  }) {
    createFlutterFrameworkSwiftPackage(platform: platform, project: project);
    return (
      SwiftPackagePackageDependency(
        name: kFlutterGeneratedFrameworkSwiftPackageTargetName,
        path: _fileSystem.path.relative(
          project.flutterFrameworkSwiftPackageDirectory.path,
          from: pathRelativeTo,
        ),
      ),
      SwiftPackageTargetDependency.product(
        name: kFlutterGeneratedFrameworkSwiftPackageTargetName,
        packageName: kFlutterGeneratedFrameworkSwiftPackageTargetName,
      ),
    );
  }

  /// Creates a Swift package called [kFlutterGeneratedFrameworkSwiftPackageTargetName] that vends the
  /// Flutter/FlutterMacOS framework as a binary target. The Flutter framework is symlinked within
  /// the package since binary targets must be relative.
  void createFlutterFrameworkSwiftPackage({
    required XcodeBasedProject project,
    required FlutterDarwinPlatform platform,
  }) {
    final String frameworkName = platform.binaryName;

    _symlinkFlutterFramework(platform: platform, project: project, frameworkName: frameworkName);
    final flutterFrameworkPackage = SwiftPackage(
      manifest: project.flutterFrameworkSwiftPackageDirectory.childFile('Package.swift'),
      name: kFlutterGeneratedFrameworkSwiftPackageTargetName,
      platforms: <SwiftPackageSupportedPlatform>[],
      products: <SwiftPackageProduct>[
        SwiftPackageProduct(
          name: kFlutterGeneratedFrameworkSwiftPackageTargetName,
          targets: <String>[kFlutterGeneratedFrameworkSwiftPackageTargetName],
        ),
      ],
      dependencies: <SwiftPackagePackageDependency>[],
      targets: <SwiftPackageTarget>[
        SwiftPackageTarget.defaultTarget(
          name: kFlutterGeneratedFrameworkSwiftPackageTargetName,
          dependencies: [],
        ),
      ],
      templateRenderer: _templateRenderer,
    );
    flutterFrameworkPackage.createSwiftPackage();
  }

  /// Creates a subdirectory in [XcodeBasedProject.flutterFrameworkSwiftPackageDirectory] for each
  /// mode in [buildModes] and symlinks the corresponding Flutter/FlutterMacOS xcframework from
  /// the engine artifact cache. Also creates a symlink directly in
  /// [XcodeBasedProject.flutterFrameworkSwiftPackageDirectory] that links to first build mode
  /// subdirectory's xcframework.
  ///
  /// When Xcode builds the project, it'll use the xcframework symlink directly in
  /// [XcodeBasedProject.flutterFrameworkSwiftPackageDirectory]. The symlink is updated during the
  /// build pre-action.
  ///
  /// Example:
  /// ```txt
  /// FlutterFramework/Debug/Flutter.xcframework -> [path to engine cache]/ios/Flutter.xcframework
  /// FlutterFramework/Profile/Flutter.xcframework -> [path to engine cache]/ios-profile/Flutter.xcframework
  /// FlutterFramework/Release/Flutter.xcframework -> [path to engine cache]/ios-release/Flutter.xcframework
  /// FlutterFramework/Flutter.xcframework -> ./Debug/Flutter.xcframework
  /// ```
  void _symlinkFlutterFramework({
    List<BuildMode> buildModes = const <BuildMode>[
      BuildMode.debug,
      BuildMode.profile,
      BuildMode.release,
    ],
    required XcodeBasedProject project,
    required FlutterDarwinPlatform platform,
    required String frameworkName,
  }) {
    for (final buildMode in buildModes) {
      final String frameworkArtifactPath = _artifacts.getArtifactPath(
        platform.xcframeworkArtifact,
        platform: platform.targetPlatform,
        mode: buildMode,
      );
      final Directory buildModeDirectory = project.flutterFrameworkSwiftPackageDirectory
          .childDirectory(buildMode.uppercaseName);
      final Link frameworkLink = _fileSystem.link(
        buildModeDirectory.childDirectory('$frameworkName.xcframework').path,
      );
      frameworkLink.createSync(frameworkArtifactPath, recursive: true);
    }
    updateFlutterFrameworkSymlink(
      buildMode: buildModes.first,
      fileSystem: _fileSystem,
      platform: platform,
      project: project,
      createIfNotFound: true,
    );
  }

  /// Update the symlink for the Flutter framework dependency to use the correct [buildMode].
  static void updateFlutterFrameworkSymlink({
    required BuildMode buildMode,
    required FileSystem fileSystem,
    required FlutterDarwinPlatform platform,
    required XcodeBasedProject project,
    bool createIfNotFound = false,
  }) {
    final String frameworkName = platform.binaryName;
    final Link frameworkLink = fileSystem.link(
      project.flutterFrameworkSwiftPackageDirectory
          .childDirectory('$frameworkName.xcframework')
          .path,
    );
    if (frameworkLink.existsSync()) {
      frameworkLink.updateSync('./${buildMode.uppercaseName}/$frameworkName.xcframework');
    } else if (createIfNotFound) {
      frameworkLink.createSync(
        './${buildMode.uppercaseName}/$frameworkName.xcframework',
        recursive: true,
      );
    }
  }

  /// If the project's IPHONEOS_DEPLOYMENT_TARGET/MACOSX_DEPLOYMENT_TARGET is
  /// higher than the FlutterGeneratedPluginSwiftPackage's default
  /// SupportedPlatform, increase the SupportedPlatform to match the project's
  /// deployment target.
  ///
  /// This is done for the use case of a plugin requiring a higher iOS/macOS
  /// version than FlutterGeneratedPluginSwiftPackage.
  ///
  /// Swift Package Manager emits an error if a dependency isn’t compatible
  /// with the top-level package’s deployment version. The deployment target of
  /// a package’s dependencies must be lower than or equal to the top-level
  /// package’s deployment target version for a particular platform.
  ///
  /// To still be able to use the plugin, the user can increase the Xcode
  /// project's iOS/macOS deployment target and this will then increase the
  /// deployment target for FlutterGeneratedPluginSwiftPackage.
  static void updateMinimumDeployment({
    required XcodeBasedProject project,
    required FlutterDarwinPlatform platform,
    required String deploymentTarget,
  }) {
    final Version? projectDeploymentTargetVersion = Version.parse(deploymentTarget);
    final SwiftPackageSupportedPlatform defaultPlatform = platform.supportedPackagePlatform;
    final SwiftPackagePlatform packagePlatform = platform.swiftPackagePlatform;

    if (projectDeploymentTargetVersion == null ||
        projectDeploymentTargetVersion <= defaultPlatform.version ||
        !project.flutterPluginSwiftPackageManifest.existsSync()) {
      return;
    }

    final String manifestContents = project.flutterPluginSwiftPackageManifest.readAsStringSync();
    final String oldSupportedPlatform = defaultPlatform.format();
    final String newSupportedPlatform = SwiftPackageSupportedPlatform(
      platform: packagePlatform,
      version: projectDeploymentTargetVersion,
    ).format();

    project.flutterPluginSwiftPackageManifest.writeAsStringSync(
      manifestContents.replaceFirst(oldSupportedPlatform, newSupportedPlatform),
    );
  }
}
