// Copyright 2014 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// Shared logic between iOS and macOS implementations of native assets.

import 'package:code_assets/code_assets.dart';
import 'package:hooks_runner/hooks_runner.dart';

import '../../../base/common.dart';
import '../../../base/file_system.dart';
import '../../../base/process.dart';
import '../../../build_info.dart';
import '../../../build_system/targets/darwin.dart';
import '../../../globals.dart' as globals;
import '../native_assets.dart';

/// Create an `Info.plist` in [target] for a framework with a single dylib.
///
/// The framework must be named [name].framework and the dylib [name].
Future<void> createInfoPlist(String name, Directory target, {String? minimumIOSVersion}) async {
  final File infoPlistFile = target.childFile('Info.plist');
  final String bundleIdentifier = 'io.flutter.flutter.native_assets.$name'.replaceAll('_', '-');
  await infoPlistFile.writeAsString(
    <String>[
      '''
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">
<dict>
	<key>CFBundleDevelopmentRegion</key>
	<string>en</string>
	<key>CFBundleExecutable</key>
	<string>$name</string>
	<key>CFBundleIdentifier</key>
	<string>$bundleIdentifier</string>
	<key>CFBundleInfoDictionaryVersion</key>
	<string>6.0</string>
	<key>CFBundleName</key>
	<string>$name</string>
	<key>CFBundlePackageType</key>
	<string>FMWK</string>
	<key>CFBundleShortVersionString</key>
	<string>1.0</string>
	<key>CFBundleSignature</key>
	<string>????</string>
	<key>CFBundleVersion</key>
	<string>1.0</string>
''',
      if (minimumIOSVersion != null)
        '''
	<key>MinimumOSVersion</key>
	<string>$minimumIOSVersion</string>
''',
      '''
</dict>
</plist>''',
    ].join(),
  );
}

/// Combines dylibs from [sources] into a fat binary in [target].
///
/// The dylibs must have different architectures. E.g. a dylib targeting
/// arm64 ios simulator cannot be combined with a dylib targeting arm64
/// ios device or macos arm64.
Future<void> lipoDylibs(File target, List<File> sources) async {
  final RunResult lipoResult = await globals.processUtils.run(<String>[
    'xcrun',
    'lipo',
    '-create',
    '-output',
    target.path,
    for (final File source in sources) source.path,
  ]);
  if (lipoResult.exitCode != 0) {
    throwToolExit('Failed to create universal binary:\n${lipoResult.stderr}');
  }
}

/// Sets the install names in a dylib with a Mach-O format.
///
/// On macOS and iOS, opening a dylib at runtime fails if the path inside the
/// dylib itself does not correspond to the path that the file is at. Therefore,
/// native assets copied into their final location also need their install name
/// updated with the `install_name_tool`.
///
/// [oldToNewInstallNames] is a map from old to new install names, that should
/// be applied to the dependencies of the dylib. Entries in this map for
/// dependencies that are not present in the dylib are ignored. The install
/// name of a dependencies needs to be updated if the location of the dependency
/// has changed.
Future<void> setInstallNamesDylib(
  File dylibFile,
  String newInstallName,
  Map<String, String> oldToNewInstallNames,
) async {
  final RunResult setInstallNamesResult = await globals.processUtils.run(<String>[
    'xcrun',
    'install_name_tool',
    '-id',
    newInstallName,
    for (final MapEntry<String, String> entry in oldToNewInstallNames.entries) ...<String>[
      '-change',
      entry.key,
      entry.value,
    ],
    dylibFile.path,
  ]);
  if (setInstallNamesResult.exitCode != 0) {
    throwToolExit(
      'Failed to change install names in $dylibFile:\n'
      'id -> $newInstallName\n'
      'dependencies -> $newInstallName\n'
      '${setInstallNamesResult.stderr}',
    );
  }
}

Future<Set<String>> getInstallNamesDylib(File dylibFile) async {
  final RunResult installNameResult = await globals.processUtils.run(<String>[
    'xcrun',
    'otool',
    '-D',
    dylibFile.path,
  ]);
  if (installNameResult.exitCode != 0) {
    throwToolExit('Failed to get the install name of $dylibFile:\n${installNameResult.stderr}');
  }

  return <String>{
    for (final List<String> architectureSection in parseOtoolArchitectureSections(
      installNameResult.stdout,
    ).values)
      // For each architecture, a separate install name is reported, which are
      // not necessarily the same.
      architectureSection.single,
  };
}

/// Creates a dSYM bundle for a dylib.
Future<void> dsymutilDylib(File dylibFile, String dsymPath) async {
  final RunResult result = await globals.processUtils.run(<String>[
    'xcrun',
    'dsymutil',
    dylibFile.path,
    '-o',
    dsymPath,
  ]);
  if (result.exitCode != 0) {
    throwToolExit('dsymutil failed with exit code ${result.exitCode}');
  }
}

/// Strips a dylib.
///
/// This is useful for release builds to reduce binary size.
Future<void> stripDylib(File dylibFile) async {
  final RunResult result = await globals.processUtils.run(<String>[
    'xcrun',
    'strip',
    '-x', // Remove local symbols.
    '-S', // Remove debugging symbol table.
    dylibFile.path,
  ]);
  if (result.exitCode != 0) {
    globals.logger.printError(result.stdout);
    globals.logger.printError(result.stderr);
    throwToolExit('strip failed with exit code ${result.exitCode}');
  }
}

Future<void> codesignDylib(
  String? codesignIdentity,
  BuildMode buildMode,
  FileSystemEntity target,
) async {
  if (codesignIdentity == null || codesignIdentity.isEmpty) {
    codesignIdentity = '-';
  }
  final codesignCommand = <String>[
    'xcrun',
    'codesign',
    '--force',
    '--sign',
    codesignIdentity,
    if (buildMode != BuildMode.release) ...<String>[
      // Mimic Xcode's timestamp codesigning behavior on non-release binaries.
      '--timestamp=none',
    ],
    target.path,
  ];
  final RunResult codesignResult = await globals.processUtils.run(codesignCommand);
  if (codesignResult.exitCode != 0) {
    throwToolExit(
      'Failed to code sign binary: exit code: ${codesignResult.exitCode} '
      '${codesignResult.stdout} ${codesignResult.stderr}',
    );
  }
}

/// Flutter expects `xcrun` to be on the path on macOS hosts.
///
/// Use the `clang`, `ar`, and `ld` that would be used if run with `xcrun`.
///
/// If no XCode installation was found, [throwIfNotFound] controls whether this
/// throws or returns `null`.
Future<CCompilerConfig?> cCompilerConfigMacOS({required bool throwIfNotFound}) async {
  final Uri? compiler = await _findXcrunBinary('clang', throwIfNotFound);
  final Uri? archiver = await _findXcrunBinary('ar', throwIfNotFound);
  final Uri? linker = await _findXcrunBinary('ld', throwIfNotFound);

  if (compiler == null || archiver == null || linker == null) {
    assert(!throwIfNotFound);
    return null;
  }

  return CCompilerConfig(compiler: compiler, archiver: archiver, linker: linker);
}

/// Invokes `xcrun --find` to find the full path to [binaryName].
Future<Uri?> _findXcrunBinary(String binaryName, bool throwIfNotFound) async {
  final RunResult xcrunResult = await globals.processUtils.run(<String>[
    'xcrun',
    '--find',
    binaryName,
  ]);
  if (xcrunResult.exitCode != 0) {
    if (throwIfNotFound) {
      throwToolExit('Failed to find $binaryName with xcrun:\n${xcrunResult.stderr}');
    } else {
      return null;
    }
  }
  return Uri.file(xcrunResult.stdout.trim());
}

/// Converts [fileName] into a suitable framework name.
///
/// On MacOS and iOS, dylibs need to be packaged in a framework.
///
/// In order for resolution to work, the file name inside the framework must be
/// equal to the framework name.
///
/// Dylib names on MacOS/iOS usually have a dylib extension. If so, remove it.
///
/// Dylib names on MacOS/iOS are usually prefixed with 'lib'. So, if the file is
/// a dylib, try to remove the prefix.
///
/// The bundle ID string must contain only alphanumeric characters
/// (A–Z, a–z, and 0–9), hyphens (-), and periods (.).
/// https://developer.apple.com/documentation/bundleresources/information_property_list/cfbundleidentifier
///
/// The [alreadyTakenNames] are used to ensure that the framework name does not
/// conflict with previously chosen names.
Uri frameworkUri(String fileName, Set<String> alreadyTakenNames) {
  final List<String> splitFileName = fileName.split('.');
  final bool isDylib;
  if (splitFileName.length >= 2) {
    isDylib = splitFileName.last == 'dylib';
    if (isDylib) {
      fileName = splitFileName.sublist(0, splitFileName.length - 1).join('.');
    }
  } else {
    isDylib = false;
  }
  if (isDylib && fileName.startsWith('lib')) {
    fileName = fileName.replaceFirst('lib', '');
  }
  fileName = fileName.replaceAll(RegExp(r'[^A-Za-z0-9_-]'), '');
  if (alreadyTakenNames.contains(fileName)) {
    final prefixName = fileName;
    for (var i = 1; i < 1000; i++) {
      fileName = '$prefixName$i';
      if (!alreadyTakenNames.contains(fileName)) {
        break;
      }
    }
    if (alreadyTakenNames.contains(fileName)) {
      throwToolExit('Failed to rename $fileName in native assets packaging.');
    }
  }
  alreadyTakenNames.add(fileName);
  return Uri(path: '$fileName.framework/$fileName');
}

Map<Architecture?, List<String>> parseOtoolArchitectureSections(String output) {
  // The output of `otool -D`, for example, looks like below. For each
  // architecture, there is a separate section.
  //
  // /build/native_assets/ios/buz.framework/buz (architecture x86_64):
  // @rpath/libbuz.dylib
  // /build/native_assets/ios/buz.framework/buz (architecture arm64):
  // @rpath/libbuz.dylib
  //
  // Some versions of `otool` don't print the architecture name if the
  // binary only has one architecture:
  //
  // /build/native_assets/ios/buz.framework/buz:
  // @rpath/libbuz.dylib

  const outputArchitectures = <String, Architecture>{
    'arm': Architecture.arm,
    'arm64': Architecture.arm64,
    'x86_64': Architecture.x64,
  };
  final architectureHeaderPattern = RegExp(r'^[^(]+( \(architecture (.+)\))?:$');
  final Iterator<String> lines = output.trim().split('\n').iterator;
  Architecture? currentArchitecture;
  final architectureSections = <Architecture?, List<String>>{};

  while (lines.moveNext()) {
    final String line = lines.current;
    final Match? architectureHeader = architectureHeaderPattern.firstMatch(line);
    if (architectureHeader != null) {
      if (architectureSections.containsKey(null)) {
        throwToolExit('Expected a single architecture section in otool output: $output');
      }
      final String? architectureString = architectureHeader.group(2);
      if (architectureString != null) {
        currentArchitecture = outputArchitectures[architectureString];
        if (currentArchitecture == null) {
          throwToolExit('Unknown architecture in otool output: $architectureString');
        }
      }
      architectureSections[currentArchitecture] = <String>[];
      continue;
    } else {
      architectureSections[currentArchitecture]!.add(line.trim());
    }
  }

  return architectureSections;
}

/// Groups native assets by their target framework path for multi-architecture
/// bundling.
///
/// On macOS and iOS, architecture-specific binaries for the same Asset ID are
/// combined into a single "fat" (universal) binary using `lipo`. This function
/// ensures that all assets with the same ID map to the same framework location.
///
/// If different architectures for the same Asset ID have different framework
/// names, a warning is issued, and the name of the first encountered
/// architecture is used.
Map<KernelAssetPath, List<FlutterCodeAsset>> fatAssetTargetLocations(
  List<FlutterCodeAsset> nativeAssets,
  KernelAsset Function(FlutterCodeAsset asset, Set<String> alreadyTakenNames)
  targetLocationCallback,
) {
  final alreadyTakenNames = <String>{};
  final result = <KernelAssetPath, List<FlutterCodeAsset>>{};
  final idToPath = <String, KernelAssetPath>{};
  for (final asset in nativeAssets) {
    // Use same target path for all assets with the same id.
    final String assetId = asset.codeAsset.id;
    final KernelAssetPath? existingPath = idToPath[assetId];
    final KernelAssetPath currentPath = targetLocationCallback(asset, alreadyTakenNames).path;

    if (existingPath != null && existingPath != currentPath) {
      final String existingName = (existingPath as KernelAssetAbsolutePath).uri.pathSegments.first;
      final String currentName = (currentPath as KernelAssetAbsolutePath).uri.pathSegments.first;
      printXcodeWarning(
        'Code asset "$assetId" has different framework names for '
        'different architectures. Picking "$existingName" and '
        'ignoring "$currentName". This is likely an issue in the '
        'package providing the asset. Please report this to the '
        'package maintainers and ensure the "build.dart" hook '
        'produces consistent filenames.',
      );
    }

    final KernelAssetPath path = existingPath ?? currentPath;
    idToPath[assetId] = path;
    result[path] ??= <FlutterCodeAsset>[];
    result[path]!.add(asset);
  }
  return result;
}
