// Copyright 2013 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'dart:convert';
import 'dart:math';
import 'dart:typed_data';

import 'package:test/bootstrap/browser.dart';
import 'package:test/test.dart';
import 'package:ui/src/engine.dart';
import 'package:ui/ui.dart' as ui;
import 'package:web_engine_tester/golden_tester.dart';

import '../common/fake_asset_manager.dart';
import '../common/test_initialization.dart';
import 'utils.dart';

void main() {
  internalBootstrapBrowserTest(() => testMain);
}

// This fragment shader generates some voronoi noise. It uses a pseudo-random
// number generator implemented in the shader itself, so its output is
// deterministic.
// Generated from engine/src/flutter/lib/ui/fixtures/shaders/general_shaders/voronoi.frag
// For changes, update that file and regenerate using impellerc.
const String kVoronoiShaderSksl = r'''
{
  "format_version": 1,
  "sksl": {
    "entrypoint": "foo_fragment_main",
    "shader": "// This SkSL shader is autogenerated by spirv-cross.\n\nfloat4 flutter_FragCoord;\n\nuniform float uTileSize;\n\nvec4 fragColor;\n\nvec2 FLT_flutter_local_FlutterFragCoord()\n{\n    return flutter_FragCoord.xy;\n}\n\nfloat FLT_flutter_local_rand(vec2 co)\n{\n    return fract(sin(dot(co, vec2(12.98980045318603515625, 78.233001708984375))) * 43758.546875);\n}\n\nvec2 FLT_flutter_local_fuzzGridPoint(vec2 coordinate)\n{\n    vec2 param = coordinate * 400.0;\n    vec2 param_1 = coordinate * 400.0;\n    return coordinate + vec2((FLT_flutter_local_rand(param) - 0.5) * 0.800000011920928955078125, (FLT_flutter_local_rand(param_1) - 0.5) * 0.800000011920928955078125);\n}\n\nvec3 FLT_flutter_local_getColorForGridPoint(vec2 coordinate)\n{\n    vec2 param = coordinate * 100.0;\n    vec2 param_1 = coordinate * 200.0;\n    vec2 param_2 = coordinate * 300.0;\n    return vec3(FLT_flutter_local_rand(param), FLT_flutter_local_rand(param_1), FLT_flutter_local_rand(param_2));\n}\n\nvoid FLT_main()\n{\n    vec2 uv = FLT_flutter_local_FlutterFragCoord() / vec2(uTileSize);\n    vec2 upperLeft = floor(uv);\n    vec2 upperRight = vec2(ceil(uv.x), floor(uv.y));\n    vec2 bottomLeft = vec2(floor(uv.x), ceil(uv.y));\n    vec2 bottomRight = ceil(uv);\n    vec2 closestPoint = upperLeft;\n    vec2 param_3 = upperLeft;\n    float dist = distance(uv, FLT_flutter_local_fuzzGridPoint(param_3));\n    vec2 param_4 = upperRight;\n    float upperRightDistance = distance(uv, FLT_flutter_local_fuzzGridPoint(param_4));\n    if (upperRightDistance < dist)\n    {\n        dist = upperRightDistance;\n        closestPoint = upperRight;\n    }\n    vec2 param_5 = bottomLeft;\n    float bottomLeftDistance = distance(uv, FLT_flutter_local_fuzzGridPoint(param_5));\n    if (bottomLeftDistance < dist)\n    {\n        dist = bottomLeftDistance;\n        closestPoint = bottomLeft;\n    }\n    vec2 param_6 = bottomRight;\n    float bottomRightDistance = distance(uv, FLT_flutter_local_fuzzGridPoint(param_6));\n    if (bottomRightDistance < dist)\n    {\n        dist = bottomRightDistance;\n        closestPoint = bottomRight;\n    }\n    vec2 param_7 = closestPoint;\n    fragColor = vec4(FLT_flutter_local_getColorForGridPoint(param_7), 1.0);\n}\n\nhalf4 main(float2 iFragCoord)\n{\n      flutter_FragCoord = float4(iFragCoord, 0, 0);\n      FLT_main();\n      return fragColor;\n}\n",
    "stage": 1,
    "uniforms": [
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 0,
        "name": "uTileSize",
        "rows": 1,
        "type": 10
      }
    ]
  }
}''';

// Fragment shader that outputs the provided texture.
// Generated from engine/src/flutter/lib/ui/fixtures/shaders/general_shaders/texture.frag
// For changes, update that file and regenerate using impellerc.
const kTextureShaderSksl = r'''
{
  "format_version": 1,
  "sksl": {
    "entrypoint": "texture_fragment_main",
    "shader": "// This SkSL shader is autogenerated by spirv-cross.\n\nfloat4 flutter_FragCoord;\n\nuniform vec2 u_size;\nuniform shader u_texture;\nuniform half2 u_texture_size;\n\nvec4 frag_color;\n\nvec2 FLT_flutter_local_FlutterFragCoord()\n{\n    return flutter_FragCoord.xy;\n}\n\nvoid FLT_main()\n{\n    frag_color = u_texture.eval(u_texture_size * ( FLT_flutter_local_FlutterFragCoord() / u_size));\n}\n\nhalf4 main(float2 iFragCoord)\n{\n      flutter_FragCoord = float4(iFragCoord, 0, 0);\n      FLT_main();\n      return frag_color;\n}\n",
    "stage": 1,
    "uniforms": [
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 0,
        "name": "u_size",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 0,
        "columns": 1,
        "location": 1,
        "name": "u_texture",
        "rows": 1,
        "type": 12
      }
    ]
  }
}''';

// Generated from engine/src/flutter/impeller/fixtures/ink_sparkle.frag
// For changes, update that file and regenerate using impellerc.
const String kInkSparkleSksl = r'''
{
  "format_version": 1,
  "sksl": {
    "entrypoint": "ink_sparkle_fragment_main",
    "shader": "// This SkSL shader is autogenerated by spirv-cross.\n\nfloat4 flutter_FragCoord;\n\nuniform vec4 u_color;\nuniform float u_alpha;\nuniform vec4 u_sparkle_color;\nuniform float u_sparkle_alpha;\nuniform float u_blur;\nuniform vec2 u_center;\nuniform float u_radius_scale;\nuniform float u_max_radius;\nuniform vec2 u_resolution_scale;\nuniform vec2 u_noise_scale;\nuniform float u_noise_phase;\nuniform vec2 u_circle1;\nuniform vec2 u_circle2;\nuniform vec2 u_circle3;\nuniform vec2 u_rotation1;\nuniform vec2 u_rotation2;\nuniform vec2 u_rotation3;\n\nvec4 fragColor;\n\nvec2 FLT_flutter_local_FlutterFragCoord()\n{\n    return flutter_FragCoord.xy;\n}\n\nmat2 FLT_flutter_local_rotate2d(vec2 rad)\n{\n    return mat2(vec2(rad.x, -rad.y), vec2(rad.y, rad.x));\n}\n\nfloat FLT_flutter_local_soft_circle(vec2 uv, vec2 xy, float radius, float blur)\n{\n    float blur_half = blur * 0.5;\n    float d = distance(uv, xy);\n    return 1.0 - smoothstep(1.0 - blur_half, 1.0 + blur_half, d / radius);\n}\n\nfloat FLT_flutter_local_circle_grid(vec2 resolution, inout vec2 p, vec2 xy, vec2 rotation, float cell_diameter)\n{\n    vec2 param = rotation;\n    p = (FLT_flutter_local_rotate2d(param) * (xy - p)) + xy;\n    p = mod(p, vec2(cell_diameter)) / resolution;\n    float cell_uv = (cell_diameter / resolution.y) * 0.5;\n    float r = 0.64999997615814208984375 * cell_uv;\n    vec2 param_1 = p;\n    vec2 param_2 = vec2(cell_uv);\n    float param_3 = r;\n    float param_4 = r * 50.0;\n    return FLT_flutter_local_soft_circle(param_1, param_2, param_3, param_4);\n}\n\nfloat FLT_flutter_local_turbulence(vec2 uv)\n{\n    vec2 uv_scale = uv * vec2(0.800000011920928955078125);\n    vec2 param = vec2(0.800000011920928955078125);\n    vec2 param_1 = uv_scale;\n    vec2 param_2 = u_circle1;\n    vec2 param_3 = u_rotation1;\n    float param_4 = 0.17000000178813934326171875;\n    float _301 = FLT_flutter_local_circle_grid(param, param_1, param_2, param_3, param_4);\n    float g1 = _301;\n    vec2 param_5 = vec2(0.800000011920928955078125);\n    vec2 param_6 = uv_scale;\n    vec2 param_7 = u_circle2;\n    vec2 param_8 = u_rotation2;\n    float param_9 = 0.20000000298023223876953125;\n    float _313 = FLT_flutter_local_circle_grid(param_5, param_6, param_7, param_8, param_9);\n    float g2 = _313;\n    vec2 param_10 = vec2(0.800000011920928955078125);\n    vec2 param_11 = uv_scale;\n    vec2 param_12 = u_circle3;\n    vec2 param_13 = u_rotation3;\n    float param_14 = 0.2750000059604644775390625;\n    float _326 = FLT_flutter_local_circle_grid(param_10, param_11, param_12, param_13, param_14);\n    float g3 = _326;\n    float v = (((g1 * g1) + g2) - g3) * 0.5;\n    return clamp(0.449999988079071044921875 + (0.800000011920928955078125 * v), 0.0, 1.0);\n}\n\nfloat FLT_flutter_local_soft_ring(vec2 uv, vec2 xy, float radius, float thickness, float blur)\n{\n    vec2 param = uv;\n    vec2 param_1 = xy;\n    float param_2 = radius + thickness;\n    float param_3 = blur;\n    float circle_outer = FLT_flutter_local_soft_circle(param, param_1, param_2, param_3);\n    vec2 param_4 = uv;\n    vec2 param_5 = xy;\n    float param_6 = max(radius - thickness, 0.0);\n    float param_7 = blur;\n    float circle_inner = FLT_flutter_local_soft_circle(param_4, param_5, param_6, param_7);\n    return clamp(circle_outer - circle_inner, 0.0, 1.0);\n}\n\nfloat FLT_flutter_local_triangle_noise(inout vec2 n)\n{\n    n = fract(n * vec2(5.398700237274169921875, 5.442100048065185546875));\n    n += vec2(dot(n.yx, n + vec2(21.5351009368896484375, 14.3136997222900390625)));\n    float xy = n.x * n.y;\n    return (fract(xy * 95.43070220947265625) + fract(xy * 75.0496063232421875)) - 1.0;\n}\n\nfloat FLT_flutter_local_threshold(float v, float l, float h)\n{\n    return step(l, v) * (1.0 - step(h, v));\n}\n\nfloat FLT_flutter_local_sparkle(vec2 uv, float t)\n{\n    vec2 param = uv;\n    float _224 = FLT_flutter_local_triangle_noise(param);\n    float n = _224;\n    float param_1 = n;\n    float param_2 = 0.0;\n    float param_3 = 0.0500000007450580596923828125;\n    float s = FLT_flutter_local_threshold(param_1, param_2, param_3);\n    float param_4 = n + sin(3.1415927410125732421875 * (t + 0.3499999940395355224609375));\n    float param_5 = 0.100000001490116119384765625;\n    float param_6 = 0.1500000059604644775390625;\n    s += FLT_flutter_local_threshold(param_4, param_5, param_6);\n    float param_7 = n + sin(3.1415927410125732421875 * (t + 0.699999988079071044921875));\n    float param_8 = 0.20000000298023223876953125;\n    float param_9 = 0.25;\n    s += FLT_flutter_local_threshold(param_7, param_8, param_9);\n    float param_10 = n + sin(3.1415927410125732421875 * (t + 1.0499999523162841796875));\n    float param_11 = 0.300000011920928955078125;\n    float param_12 = 0.3499999940395355224609375;\n    s += FLT_flutter_local_threshold(param_10, param_11, param_12);\n    return clamp(s, 0.0, 1.0) * 0.550000011920928955078125;\n}\n\nvoid FLT_main()\n{\n    vec2 p = FLT_flutter_local_FlutterFragCoord();\n    vec2 uv_1 = p * u_resolution_scale;\n    vec2 density_uv = uv_1 - mod(p, u_noise_scale);\n    float radius = u_max_radius * u_radius_scale;\n    vec2 param_13 = uv_1;\n    float turbulence = FLT_flutter_local_turbulence(param_13);\n    vec2 param_14 = p;\n    vec2 param_15 = u_center;\n    float param_16 = radius;\n    float param_17 = 0.0500000007450580596923828125 * u_max_radius;\n    float param_18 = u_blur;\n    float ring = FLT_flutter_local_soft_ring(param_14, param_15, param_16, param_17, param_18);\n    vec2 param_19 = density_uv;\n    float param_20 = u_noise_phase;\n    float sparkle = ((FLT_flutter_local_sparkle(param_19, param_20) * ring) * turbulence) * u_sparkle_alpha;\n    vec2 param_21 = p;\n    vec2 param_22 = u_center;\n    float param_23 = radius;\n    float param_24 = u_blur;\n    float wave_alpha = (FLT_flutter_local_soft_circle(param_21, param_22, param_23, param_24) * u_alpha) * u_color.w;\n    vec4 wave_color = vec4(u_color.xyz * wave_alpha, wave_alpha);\n    vec4 sparkle_color = vec4(u_sparkle_color.xyz * u_sparkle_color.w, u_sparkle_color.w);\n    fragColor = mix(wave_color, sparkle_color, vec4(sparkle));\n}\n\nhalf4 main(float2 iFragCoord)\n{\n      flutter_FragCoord = float4(iFragCoord, 0, 0);\n      FLT_main();\n      return fragColor;\n}\n",
    "stage": 1,
    "uniforms": [
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 0,
        "name": "u_color",
        "rows": 4,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 1,
        "name": "u_alpha",
        "rows": 1,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 2,
        "name": "u_sparkle_color",
        "rows": 4,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 3,
        "name": "u_sparkle_alpha",
        "rows": 1,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 4,
        "name": "u_blur",
        "rows": 1,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 5,
        "name": "u_center",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 6,
        "name": "u_radius_scale",
        "rows": 1,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 7,
        "name": "u_max_radius",
        "rows": 1,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 8,
        "name": "u_resolution_scale",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 9,
        "name": "u_noise_scale",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 10,
        "name": "u_noise_phase",
        "rows": 1,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 11,
        "name": "u_circle1",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 12,
        "name": "u_circle2",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 13,
        "name": "u_circle3",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 14,
        "name": "u_rotation1",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 15,
        "name": "u_rotation2",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 16,
        "name": "u_rotation3",
        "rows": 2,
        "type": 10
      }
    ]
  }
}
''';

// Simple shader with a bunch of array uniforms
// Generated from engine/src/flutter/lib/ui/fixtures/shaders/general_shaders/many_arrays.frag
// For changes, update that file and regenerate using impellerc.
const String kManyArraysSksl = r'''
{
  "format_version": 1,
  "sksl": {
    "entrypoint": "foo_fragment_main",
    "shader": "// This SkSL shader is autogenerated by spirv-cross.\n\nfloat4 flutter_FragCoord;\n\nuniform vec2 uSize;\nuniform float uFloats[10];\nuniform float uLoneFloat;\nuniform vec2 uVectors[3];\nuniform vec3 uLoneVector;\nuniform mat4 uMatrices[2];\nuniform mat4 uLoneMatrix;\n\nvec4 fragColor;\n\nvec2 FLT_flutter_local_FlutterFragCoord()\n{\n    return flutter_FragCoord.xy;\n}\n\nvoid FLT_main()\n{\n    vec2 pos = FLT_flutter_local_FlutterFragCoord();\n    vec2 uv = pos / uSize;\n    float barWidth = 0.100000001490116119384765625;\n    float barIndex = floor(uv.x / barWidth);\n    for (int i = 0; i < 10; i++)\n    {\n        if (i == int(barIndex))\n        {\n            float barHeight = uFloats[i];\n            if (uv.y > (1.0 - barHeight))\n            {\n                fragColor = vec4(0.3300000131130218505859375, 0.0900000035762786865234375, 0.0900000035762786865234375, 1.0);\n            }\n            else\n            {\n                fragColor = vec4(1.0);\n            }\n        }\n    }\n}\n\nhalf4 main(float2 iFragCoord)\n{\n      flutter_FragCoord = float4(iFragCoord, 0, 0);\n      FLT_main();\n      return fragColor;\n}\n",
    "stage": 1,
    "uniforms": [
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 0,
        "name": "uSize",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 10,
        "bit_width": 32,
        "columns": 1,
        "location": 1,
        "name": "uFloats",
        "rows": 1,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 11,
        "name": "uLoneFloat",
        "rows": 1,
        "type": 10
      },
      {
        "array_elements": 3,
        "bit_width": 32,
        "columns": 1,
        "location": 12,
        "name": "uVectors",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 15,
        "name": "uLoneVector",
        "rows": 3,
        "type": 10
      },
      {
        "array_elements": 2,
        "bit_width": 32,
        "columns": 4,
        "location": 16,
        "name": "uMatrices",
        "rows": 4,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 4,
        "location": 18,
        "name": "uLoneMatrix",
        "rows": 4,
        "type": 10
      }
    ]
  }
}
''';

// Simple shader with a float uniform
// Generated from engine/src/flutter/lib/ui/fixtures/shaders/uniforms/float_uniform.frag
// For changes, update that file and regenerate using impellerc.
const String kFloatSkSl = r'''
{
  "format_version": 2,
  "sksl": {
    "entrypoint": "float_uniform_fragment_main",
    "shader": "// This SkSL shader is autogenerated by spirv-cross.\n\nfloat4 flutter_FragCoord;\n\nuniform float color_r;\n\nvec4 fragColor;\n\nvoid FLT_main()\n{\n    fragColor = vec4(color_r, 0.0, 0.0, 1.0);\n}\n\nhalf4 main(float2 iFragCoord)\n{\n      flutter_FragCoord = float4(iFragCoord, 0, 0);\n      FLT_main();\n      return fragColor;\n}\n",
    "stage": 1,
    "uniforms": [
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 0,
        "name": "color_r",
        "rows": 1,
        "type": 10
      }
    ]
  }
}
''';

// Simple shader with a vec2 uniform
// Generated from engine/src/flutter/lib/ui/fixtures/shaders/uniforms/vec2_uniform.frag
// For changes, update that file and regenerate using impellerc.
const String kVec2SkSl = r'''
{
  "format_version": 2,
  "sksl": {
    "entrypoint": "vec2_uniform_fragment_main",
    "shader": "// This SkSL shader is autogenerated by spirv-cross.\n\nfloat4 flutter_FragCoord;\n\nuniform vec2 color_rg;\n\nvec4 fragColor;\n\nvoid FLT_main()\n{\n    fragColor = vec4(color_rg, 0.0, 1.0);\n}\n\nhalf4 main(float2 iFragCoord)\n{\n      flutter_FragCoord = float4(iFragCoord, 0, 0);\n      FLT_main();\n      return fragColor;\n}\n",
    "stage": 1,
    "uniforms": [
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 0,
        "name": "color_rg",
        "rows": 2,
        "type": 10
      }
    ]
  }
}
''';

// Simple shader with a vec3 uniform
// Generated from engine/src/flutter/lib/ui/fixtures/shaders/uniforms/vec3_uniform.frag
// For changes, update that file and regenerate using impellerc.
const String kVec3SkSl = r'''
{
  "format_version": 2,
  "sksl": {
    "entrypoint": "vec3_uniform_fragment_main",
    "shader": "// This SkSL shader is autogenerated by spirv-cross.\n\nfloat4 flutter_FragCoord;\n\nuniform vec3 color_rgb;\n\nvec4 fragColor;\n\nvoid FLT_main()\n{\n    fragColor = vec4(color_rgb, 1.0);\n}\n\nhalf4 main(float2 iFragCoord)\n{\n      flutter_FragCoord = float4(iFragCoord, 0, 0);\n      FLT_main();\n      return fragColor;\n}\n",
    "stage": 1,
    "uniforms": [
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 0,
        "name": "color_rgb",
        "rows": 3,
        "type": 10
      }
    ]
  }
}
''';

// Simple shader with a vec4 uniform
// Generated from engine/src/flutter/lib/ui/fixtures/shaders/uniforms/vec4_uniform.frag
// For changes, update that file and regenerate using impellerc.
const String kVec4SkSl = r'''
{
  "format_version": 2,
  "sksl": {
    "entrypoint": "vec4_uniform_fragment_main",
    "shader": "// This SkSL shader is autogenerated by spirv-cross.\n\nfloat4 flutter_FragCoord;\n\nuniform vec4 color_rgba;\n\nvec4 fragColor;\n\nvoid FLT_main()\n{\n    fragColor = color_rgba;\n}\n\nhalf4 main(float2 iFragCoord)\n{\n      flutter_FragCoord = float4(iFragCoord, 0, 0);\n      FLT_main();\n      return fragColor;\n}\n",
    "stage": 1,
    "uniforms": [
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 0,
        "name": "color_rgba",
        "rows": 4,
        "type": 10
      }
    ]
  }
}
''';

// Simple shader with a float[] uniform
// Generated from engine/src/flutter/lib/ui/fixtures/shaders/uniforms/float_array_uniform.frag
// For changes, update that file and regenerate using impellerc.
const String kFloatArraySkSl = r'''
{
  "format_version": 2,
  "sksl": {
    "entrypoint": "float_array_uniform_fragment_main",
    "shader": "// This SkSL shader is autogenerated by spirv-cross.\n\nfloat4 flutter_FragCoord;\n\nuniform float color_array[4];\n\nvec4 fragColor;\n\nvoid FLT_main()\n{\n    fragColor = vec4(color_array[0], color_array[1], color_array[2], color_array[3]);\n}\n\nhalf4 main(float2 iFragCoord)\n{\n      flutter_FragCoord = float4(iFragCoord, 0, 0);\n      FLT_main();\n      return fragColor;\n}\n",
    "stage": 1,
    "uniforms": [
      {
        "array_elements": 4,
        "bit_width": 32,
        "columns": 1,
        "location": 0,
        "name": "color_array",
        "rows": 1,
        "type": 10
      }
    ]
  }
}
''';

// Simple shader with a vec2[] uniform
// Generated from engine/src/flutter/lib/ui/fixtures/shaders/uniforms/vec2_array_uniform.frag
// For changes, update that file and regenerate using impellerc.
const String kVec2ArraySkSl = r'''
{
  "format_version": 2,
  "sksl": {
    "entrypoint": "vec2_array_uniform_fragment_main",
    "shader": "// This SkSL shader is autogenerated by spirv-cross.\n\nfloat4 flutter_FragCoord;\n\nuniform vec2 color_array[2];\n\nvec4 fragColor;\n\nvoid FLT_main()\n{\n    fragColor = vec4(color_array[0].x, color_array[0].y, color_array[1].x, color_array[1].y);\n}\n\nhalf4 main(float2 iFragCoord)\n{\n      flutter_FragCoord = float4(iFragCoord, 0, 0);\n      FLT_main();\n      return fragColor;\n}\n",
    "stage": 1,
    "uniforms": [
      {
        "array_elements": 2,
        "bit_width": 32,
        "columns": 1,
        "location": 0,
        "name": "color_array",
        "rows": 2,
        "type": 10
      }
    ]
  }
}
''';

// Simple shader with a vec3[] uniform
// Generated from engine/src/flutter/lib/ui/fixtures/shaders/uniforms/vec3_array_uniform.frag
// For changes, update that file and regenerate using impellerc.
const String kVec3ArraySkSl = r'''
{
  "format_version": 2,
  "sksl": {
    "entrypoint": "vec3_array_uniform_fragment_main",
    "shader": "// This SkSL shader is autogenerated by spirv-cross.\n\nfloat4 flutter_FragCoord;\n\nuniform vec2 u_size;\nuniform vec3 color_array[2];\n\nvec4 fragColor;\n\nvec2 FLT_flutter_local_FlutterFragCoord()\n{\n    return flutter_FragCoord.xy;\n}\n\nvoid FLT_main()\n{\n    vec2 uv = FLT_flutter_local_FlutterFragCoord() / u_size;\n    if (uv.x < 0.5)\n    {\n        fragColor = vec4(color_array[0], 1.0);\n    }\n    else\n    {\n        fragColor = vec4(color_array[1], 1.0);\n    }\n}\n\nhalf4 main(float2 iFragCoord)\n{\n      flutter_FragCoord = float4(iFragCoord, 0, 0);\n      FLT_main();\n      return fragColor;\n}\n",
    "stage": 1,
    "uniforms": [
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 0,
        "name": "u_size",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 2,
        "bit_width": 32,
        "columns": 1,
        "location": 1,
        "name": "color_array",
        "rows": 3,
        "type": 10
      }
    ]
  }
}
''';

// Simple shader with a vec4[] uniform
// Generated from engine/src/flutter/lib/ui/fixtures/shaders/general_shaders/vec4_array_uniform.frag
// For changes, update that file and regenerate using impellerc.
const String kVec4ArraySkSl = r'''
{
  "format_version": 2,
  "sksl": {
    "entrypoint": "vec4_array_uniform_fragment_main",
    "shader": "// This SkSL shader is autogenerated by spirv-cross.\n\nfloat4 flutter_FragCoord;\n\nuniform vec2 u_size;\nuniform vec4 color_array[2];\n\nvec4 fragColor;\n\nvec2 FLT_flutter_local_FlutterFragCoord()\n{\n    return flutter_FragCoord.xy;\n}\n\nvoid FLT_main()\n{\n    vec2 uv = FLT_flutter_local_FlutterFragCoord() / u_size;\n    if (uv.x < 0.5)\n    {\n        fragColor = color_array[0];\n    }\n    else\n    {\n        fragColor = color_array[1];\n    }\n}\n\nhalf4 main(float2 iFragCoord)\n{\n      flutter_FragCoord = float4(iFragCoord, 0, 0);\n      FLT_main();\n      return fragColor;\n}\n",
    "stage": 1,
    "uniforms": [
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 0,
        "name": "u_size",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 2,
        "bit_width": 32,
        "columns": 1,
        "location": 1,
        "name": "color_array",
        "rows": 4,
        "type": 10
      }
    ]
  }
}
''';

const String kAllUniformsSkSl = r'''
{
  "format_version": 2,
  "sksl": {
    "entrypoint": "all_uniforms_fragment_main",
    "shader": "// This SkSL shader is autogenerated by spirv-cross.\n\nfloat4 flutter_FragCoord;\n\nuniform float uFloat;\nuniform vec2 uVec2;\nuniform vec3 uVec3;\nuniform vec4 uVec4;\nuniform float uFloatArray[10];\nuniform vec2 uVec2Array[10];\nuniform vec3 uVec3Array[10];\nuniform vec4 uVec4Array[10];\n\nvec4 fragColor;\n\nfloat N_COLOR_VALUES;\n\nvec2 FLT_flutter_local_FlutterFragCoord()\n{\n    return flutter_FragCoord.xy;\n}\n\nvoid FLT_main()\n{\n    N_COLOR_VALUES = 44.0;\n    float u = FLT_flutter_local_FlutterFragCoord().x / N_COLOR_VALUES;\n    float increment = 1.0 / N_COLOR_VALUES;\n    float offset = increment;\n    if (u < offset)\n    {\n        fragColor = vec4(uFloat, 0.0, 0.0, 1.0);\n        return;\n    }\n    offset += increment;\n    if (u < offset)\n    {\n        fragColor = vec4(uVec2, 0.0, 1.0);\n        return;\n    }\n    offset += increment;\n    if (u < offset)\n    {\n        fragColor = vec4(uVec3, 1.0);\n        return;\n    }\n    offset += increment;\n    if (u < offset)\n    {\n        fragColor = uVec4;\n        return;\n    }\n    offset += increment;\n    for (int i = 0; i < 10; i++)\n    {\n        if (u < offset)\n        {\n            fragColor = vec4(uFloatArray[i], 0.0, 0.0, 1.0);\n            return;\n        }\n        offset += increment;\n    }\n    for (int i_1 = 0; i_1 < 10; i_1++)\n    {\n        if (u < offset)\n        {\n            fragColor = vec4(uVec2Array[i_1], 0.0, 1.0);\n            return;\n        }\n        offset += increment;\n    }\n    for (int i_2 = 0; i_2 < 10; i_2++)\n    {\n        if (u < offset)\n        {\n            fragColor = vec4(uVec3Array[i_2], 1.0);\n            return;\n        }\n        offset += increment;\n    }\n    for (int i_3 = 0; i_3 < 10; i_3++)\n    {\n        if (u < offset)\n        {\n            fragColor = uVec4Array[i_3];\n            return;\n        }\n        offset += increment;\n    }\n}\n\nhalf4 main(float2 iFragCoord)\n{\n      flutter_FragCoord = float4(iFragCoord, 0, 0);\n      FLT_main();\n      return fragColor;\n}\n",
    "stage": 1,
    "uniforms": [
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 0,
        "name": "uFloat",
        "rows": 1,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 1,
        "name": "uVec2",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 2,
        "name": "uVec3",
        "rows": 3,
        "type": 10
      },
      {
        "array_elements": 0,
        "bit_width": 32,
        "columns": 1,
        "location": 3,
        "name": "uVec4",
        "rows": 4,
        "type": 10
      },
      {
        "array_elements": 10,
        "bit_width": 32,
        "columns": 1,
        "location": 4,
        "name": "uFloatArray",
        "rows": 1,
        "type": 10
      },
      {
        "array_elements": 10,
        "bit_width": 32,
        "columns": 1,
        "location": 14,
        "name": "uVec2Array",
        "rows": 2,
        "type": 10
      },
      {
        "array_elements": 10,
        "bit_width": 32,
        "columns": 1,
        "location": 24,
        "name": "uVec3Array",
        "rows": 3,
        "type": 10
      },
      {
        "array_elements": 10,
        "bit_width": 32,
        "columns": 1,
        "location": 34,
        "name": "uVec4Array",
        "rows": 4,
        "type": 10
      }
    ]
  }
}
''';

Future<void> testMain() async {
  setUpUnitTests(withImplicitView: true, setUpTestViewDimensions: false);

  const region = ui.Rect.fromLTWH(0, 0, 300, 300);

  late FakeAssetScope assetScope;
  setUpAll(() {
    assetScope = fakeAssetManager.pushAssetScope();
    assetScope.setAsset('voronoi_shader', ByteData.sublistView(utf8.encode(kVoronoiShaderSksl)));
    assetScope.setAsset('texture_shader', ByteData.sublistView(utf8.encode(kTextureShaderSksl)));
    assetScope.setAsset('many_arrays', ByteData.sublistView(utf8.encode(kManyArraysSksl)));
    assetScope.setAsset('ink_sparkle', ByteData.sublistView(utf8.encode(kInkSparkleSksl)));
    assetScope.setAsset('float_uniform', ByteData.sublistView(utf8.encode(kFloatSkSl)));
    assetScope.setAsset('vec2_uniform', ByteData.sublistView(utf8.encode(kVec2SkSl)));
    assetScope.setAsset('vec3_uniform', ByteData.sublistView(utf8.encode(kVec3SkSl)));
    assetScope.setAsset('vec4_uniform', ByteData.sublistView(utf8.encode(kVec4SkSl)));
    assetScope.setAsset('float_array_uniform', ByteData.sublistView(utf8.encode(kFloatArraySkSl)));
    assetScope.setAsset('vec2_array_uniform', ByteData.sublistView(utf8.encode(kVec2ArraySkSl)));
    assetScope.setAsset('vec3_array_uniform', ByteData.sublistView(utf8.encode(kVec3ArraySkSl)));
    assetScope.setAsset('vec4_array_uniform', ByteData.sublistView(utf8.encode(kVec4ArraySkSl)));
    assetScope.setAsset('all_uniforms', ByteData.sublistView(utf8.encode(kAllUniformsSkSl)));
  });

  tearDownAll(() {
    fakeAssetManager.popAssetScope(assetScope);
  });

  test('fragment shader', () async {
    final ui.FragmentProgram program = await renderer.createFragmentProgram('voronoi_shader');
    final ui.FragmentShader shader = program.fragmentShader();

    Future<void> drawCircle(String goldenFilename) async {
      final recorder = ui.PictureRecorder();
      final canvas = ui.Canvas(recorder, region);
      canvas.drawCircle(const ui.Offset(150, 150), 100, ui.Paint()..shader = shader);

      await drawPictureUsingCurrentRenderer(recorder.endRecording());

      await matchGoldenFile(goldenFilename, region: region);
    }

    shader.setFloat(0, 10.0);
    await drawCircle('fragment_shader_voronoi_tile10px.png');

    // Make sure we can reuse the shader object with a new uniform value.
    shader.setFloat(0, 25.0);
    await drawCircle('fragment_shader_voronoi_tile25px.png');

    // Test reusing a Paint object with the same shader.
    final reusablePaint = ui.Paint()..shader = shader;

    Future<void> drawCircleReusePaint(String goldenFilename) async {
      final recorder = ui.PictureRecorder();
      final canvas = ui.Canvas(recorder, region);
      canvas.drawCircle(const ui.Offset(150, 150), 100, reusablePaint);

      await drawPictureUsingCurrentRenderer(recorder.endRecording());

      await matchGoldenFile(goldenFilename, region: region);
    }

    shader.setFloat(0, 10.0);
    await drawCircleReusePaint('fragment_shader_voronoi_tile10px_reuse_paint.png');

    // Make sure we can reuse the shader object with a new uniform value and the same Paint object.
    shader.setFloat(0, 25.0);
    await drawCircleReusePaint('fragment_shader_voronoi_tile25px_reuse_paint.png');
  }, skip: isWimp); // https://github.com/flutter/flutter/issues/175431

  for (final (filterQuality, goldenFilename) in [
    (ui.FilterQuality.none, 'web_fragment_shader_texture_with_quality_none.png'),
    (ui.FilterQuality.low, 'web_fragment_shader_texture_with_quality_low.png'),
    (ui.FilterQuality.medium, 'web_fragment_shader_texture_with_quality_medium.png'),
    (ui.FilterQuality.high, 'web_fragment_shader_texture_with_quality_high.png'),
  ]) {
    test('fragment shader renders sampler with filter quality ${filterQuality.name}', () async {
      final ui.Image image = _createOvalGradientImage(imageDimension: 16);

      final ui.FragmentProgram program = await renderer.createFragmentProgram('texture_shader');
      final ui.FragmentShader shader = program.fragmentShader();

      // Resolution
      shader.getUniformFloat('u_size', 0).set(region.width);
      shader.getUniformFloat('u_size', 1).set(region.height);

      // Image
      shader.setImageSampler(0, image, filterQuality: filterQuality);

      final recorder = ui.PictureRecorder();
      final canvas = ui.Canvas(recorder, region);
      canvas.drawRect(
        ui.Rect.fromLTRB(0, 0, region.width, region.height),
        ui.Paint()..shader = shader,
      );

      await drawPictureUsingCurrentRenderer(recorder.endRecording());

      await matchGoldenFile(goldenFilename, region: region);
    }, skip: isWimp); // https://github.com/flutter/flutter/issues/175431
  }

  // Regression test for https://github.com/flutter/flutter/issues/180862.
  test('fragment shader with space in name loads correctly', () async {
    assetScope.setAsset('voronoi%20shader', ByteData.sublistView(utf8.encode(kVoronoiShaderSksl)));
    await expectLater(ui.FragmentProgram.fromAsset('voronoi shader'), completes);
  }, skip: isWimp); // https://github.com/flutter/flutter/issues/175431

  group('fragmentShader uniforms', () {
    late Map<Type, ui.FragmentShader> shaderMap;
    setUpAll(() async {
      shaderMap = {
        ui.UniformFloatSlot: (await renderer.createFragmentProgram(
          'float_uniform',
        )).fragmentShader(),
        ui.UniformVec2Slot: (await renderer.createFragmentProgram('vec2_uniform')).fragmentShader(),
        ui.UniformVec3Slot: (await renderer.createFragmentProgram('vec3_uniform')).fragmentShader(),
        ui.UniformVec4Slot: (await renderer.createFragmentProgram('vec4_uniform')).fragmentShader(),
        ui.UniformArray<ui.UniformFloatSlot>: (await renderer.createFragmentProgram(
          'float_array_uniform',
        )).fragmentShader(),
        ui.UniformArray<ui.UniformVec2Slot>: (await renderer.createFragmentProgram(
          'vec2_array_uniform',
        )).fragmentShader(),
        ui.UniformArray<ui.UniformVec3Slot>: (await renderer.createFragmentProgram(
          'vec3_array_uniform',
        )).fragmentShader(),
        ui.UniformArray<ui.UniformVec4Slot>: (await renderer.createFragmentProgram(
          'vec4_array_uniform',
        )).fragmentShader(),
      };
    });

    group('float', () {
      group('float', () {
        test('set using setUniformFloat', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformFloatSlot]!;
          const color = ui.Color.fromARGB(255, 255, 0, 0);
          shader.setFloat(0, color.r);
          await _expectShaderRendersColor(shader, color);
        });

        test('set using getUniformFloat', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformFloatSlot]!;
          const color = ui.Color.fromARGB(255, 50, 0, 0);
          shader.getUniformFloat('color_r').set(color.r);
          await _expectShaderRendersColor(shader, color);
        });

        test('getUniformFloat offset overflow', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformFloatSlot]!;
          expect(
            () => shader.getUniformFloat('color_r', 2),
            throwsA(
              isA<IndexError>().having(
                (e) => e.message,
                'message',
                contains('Index `2` out of bounds for `color_r`.'),
              ),
            ),
          );
        });

        test('getUniformFloat offset underflow', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformFloatSlot]!;
          expect(
            () => shader.getUniformFloat('color_r', -1),
            throwsA(
              isA<IndexError>().having(
                (e) => e.message,
                'message',
                contains('Index `-1` out of bounds for `color_r`.'),
              ),
            ),
          );
        });
      });
      group('vec2', () {
        test('set using setFloat', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformVec2Slot]!;
          const color = ui.Color.fromARGB(255, 255, 255, 0);
          shader.setFloat(0, color.r);
          shader.setFloat(1, color.g);
          await _expectShaderRendersColor(shader, color);
        });

        test('set using getUniformVec2', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformVec2Slot]!;
          const color = ui.Color.fromARGB(255, 50, 50, 0);
          shader.getUniformVec2('color_rg').set(color.r, color.g);
          await _expectShaderRendersColor(shader, color);
        });

        test('wrong datatype', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformVec3Slot]!;
          expect(
            () => shader.getUniformVec2('color_rgb'),
            throwsA(
              isA<ArgumentError>().having(
                (e) => e.message,
                'message',
                contains('`color_rgb` has size 3, not size 2.'),
              ),
            ),
          );
        });
      });
      group('vec3', () {
        test('set using setFloat', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformVec3Slot]!;
          const color = ui.Color.fromARGB(255, 67, 42, 12);
          shader.setFloat(0, color.r);
          shader.setFloat(1, color.g);
          shader.setFloat(2, color.b);
          // Note: The original test also called getUniformVec3 after setFloat.
          // Assuming this was intentional to test idempotency or a specific interaction.
          shader.getUniformVec3('color_rgb').set(color.r, color.g, color.b);
          await _expectShaderRendersColor(shader, color);
        });

        test('set using getUniformVec3', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformVec3Slot]!;
          const color = ui.Color.fromARGB(255, 42, 67, 12);
          shader.getUniformVec3('color_rgb').set(color.r, color.g, color.b);
          await _expectShaderRendersColor(shader, color);
        });

        test('wrong datatype', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformVec2Slot]!;
          expect(
            () => shader.getUniformVec3('color_rg'),
            throwsA(
              isA<ArgumentError>().having(
                (e) => e.message,
                'message',
                contains('`color_rg` has size 2, not size 3.'),
              ),
            ),
          );
        });
      });

      group('vec4', () {
        test('set using setFloat', () async {
          const color = ui.Color.fromARGB(255, 67, 42, 12);
          final ui.FragmentShader shader = shaderMap[ui.UniformVec4Slot]!;
          shader.setFloat(0, color.r);
          shader.setFloat(1, color.g);
          shader.setFloat(2, color.b);
          shader.setFloat(3, color.a);
          await _expectShaderRendersColor(shader, color);
        });

        test('set using getUniformVec4', () async {
          const color = ui.Color.fromARGB(255, 12, 37, 27);
          final ui.FragmentShader shader = shaderMap[ui.UniformVec4Slot]!;
          shader.getUniformVec4('color_rgba').set(color.r, color.g, color.b, color.a);
          await _expectShaderRendersColor(shader, color);
        });

        test('wrong datatype', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformVec3Slot]!;
          expect(
            () => shader.getUniformVec4('color_rgb'),
            throwsA(
              isA<ArgumentError>().having(
                (e) => e.message,
                'message',
                contains('`color_rgb` has size 3, not size 4.'),
              ),
            ),
          );
        });
      });

      group('float array', () {
        test('set using setFloat', () async {
          const color = ui.Color.fromARGB(255, 11, 22, 96);
          final ui.FragmentShader shader = shaderMap[ui.UniformArray<ui.UniformFloatSlot>]!;
          shader.setFloat(0, color.r);
          shader.setFloat(1, color.g);
          shader.setFloat(2, color.b);
          shader.setFloat(3, color.a);
          await _expectShaderRendersColor(shader, color);
        });

        test('set using getUniformFloatArray', () async {
          const color = ui.Color.fromARGB(255, 96, 11, 22);
          final ui.FragmentShader shader = shaderMap[ui.UniformArray<ui.UniformFloatSlot>]!;
          final ui.UniformArray<ui.UniformFloatSlot> colorRgba = shader.getUniformFloatArray(
            'color_array',
          );
          colorRgba[0].set(color.r);
          colorRgba[1].set(color.g);
          colorRgba[2].set(color.b);
          colorRgba[3].set(color.a);
          await _expectShaderRendersColor(shader, color);
        });
      });

      group('vec2 array', () {
        test('set using setFloat', () async {
          const color = ui.Color.fromARGB(255, 67, 42, 12);
          final ui.FragmentShader shader = shaderMap[ui.UniformArray<ui.UniformVec2Slot>]!;
          shader.setFloat(0, color.r);
          shader.setFloat(1, color.g);
          shader.setFloat(2, color.b);
          shader.setFloat(3, color.a);
          await _expectShaderRendersColor(shader, color);
        });

        test('set using getUniformVec2Array', () async {
          const color = ui.Color.fromARGB(255, 1, 73, 26);
          final ui.FragmentShader shader = shaderMap[ui.UniformArray<ui.UniformVec2Slot>]!;
          final ui.UniformArray<ui.UniformVec2Slot> colorRgba = shader.getUniformVec2Array(
            'color_array',
          );
          colorRgba[0].set(color.r, color.g);
          colorRgba[1].set(color.b, color.a);
          await _expectShaderRendersColor(shader, color);
        });

        test('wrong datatype', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformVec3Slot]!;
          expect(
            () => shader.getUniformVec2Array('color_rgb'),
            throwsA(
              isA<ArgumentError>().having(
                (e) => e.message,
                'message',
                contains('Uniform size (3) for "color_rgb" is not a multiple of 2.'),
              ),
            ),
          );
        });
      });

      group('vec3 array', () {
        test('set using setFloat', () async {
          const cpuColors = [
            ui.Color.fromARGB(255, 67, 42, 12),
            ui.Color.fromARGB(255, 11, 22, 96),
          ];
          final ui.FragmentShader shader = shaderMap[ui.UniformArray<ui.UniformVec3Slot>]!;
          shader.setFloat(0, 2);
          shader.setFloat(1, 2);
          shader.setFloat(2, cpuColors[0].r);
          shader.setFloat(3, cpuColors[0].g);
          shader.setFloat(4, cpuColors[0].b);
          shader.setFloat(5, cpuColors[1].r);
          shader.setFloat(6, cpuColors[1].g);
          shader.setFloat(7, cpuColors[1].b);
          await _expectShaderRendersBarcode(shader, cpuColors);
        });

        test('set using getUniformVec3Array', () async {
          const cpuColors = [
            ui.Color.fromARGB(255, 11, 22, 96),
            ui.Color.fromARGB(255, 67, 42, 12),
          ];
          final ui.FragmentShader shader = shaderMap[ui.UniformArray<ui.UniformVec3Slot>]!;
          shader.getUniformVec2('u_size').set(2, 2);
          final ui.UniformArray<ui.UniformVec3Slot> gpuColors = shader.getUniformVec3Array(
            'color_array',
          );
          gpuColors[0].set(cpuColors[0].r, cpuColors[0].g, cpuColors[0].b);
          gpuColors[1].set(cpuColors[1].r, cpuColors[1].g, cpuColors[1].b);
          await _expectShaderRendersBarcode(shader, cpuColors);
        });

        test('wrong datatype', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformVec4Slot]!;
          expect(
            () => shader.getUniformVec3Array('color_rgba'),
            throwsA(
              isA<ArgumentError>().having(
                (e) => e.message,
                'message',
                contains('Uniform size (4) for "color_rgba" is not a multiple of 3.'),
              ),
            ),
          );
        });
      });

      group('vec4 array', () {
        test('set using setFloat', () async {
          const cpuColors = [
            ui.Color.fromARGB(255, 67, 42, 12),
            ui.Color.fromARGB(255, 11, 22, 96),
          ];
          final ui.FragmentShader shader = shaderMap[ui.UniformArray<ui.UniformVec4Slot>]!;
          // 'u_size'
          shader.setFloat(0, 2);
          shader.setFloat(1, 2);
          shader.setFloat(2, cpuColors[0].r);
          shader.setFloat(3, cpuColors[0].g);
          shader.setFloat(4, cpuColors[0].b);
          shader.setFloat(5, cpuColors[0].a);
          shader.setFloat(6, cpuColors[1].r);
          shader.setFloat(7, cpuColors[1].g);
          shader.setFloat(8, cpuColors[1].b);
          shader.setFloat(9, cpuColors[1].a);
          await _expectShaderRendersBarcode(shader, cpuColors);
        });

        test('set using getUniformVec4Array', () async {
          const cpuColors = [
            ui.Color.fromARGB(255, 11, 22, 96),
            ui.Color.fromARGB(255, 67, 42, 12),
          ];
          final ui.FragmentShader shader = shaderMap[ui.UniformArray<ui.UniformVec4Slot>]!;
          shader.getUniformVec2('u_size').set(2, 2);
          final ui.UniformArray<ui.UniformVec4Slot> colors = shader.getUniformVec4Array(
            'color_array',
          );
          colors[0].set(cpuColors[0].r, cpuColors[0].g, cpuColors[0].b, cpuColors[0].a);
          colors[1].set(cpuColors[1].r, cpuColors[1].g, cpuColors[1].b, cpuColors[1].a);
          await _expectShaderRendersBarcode(shader, cpuColors);
        });

        test('wrong datatype', () async {
          final ui.FragmentShader shader = shaderMap[ui.UniformVec3Slot]!;
          expect(
            () => shader.getUniformVec4Array('color_rgb'),
            throwsA(
              isA<ArgumentError>().having(
                (e) => e.message,
                'message',
                contains('Uniform size (3) for "color_rgb" is not a multiple of 4.'),
              ),
            ),
          );
        });
      });

      group('all uniforms', () {
        late ui.FragmentProgram program;
        late List<ui.Color> cpuColors;
        final random = Random(1337);
        setUpAll(() async {
          program = await ui.FragmentProgram.fromAsset('all_uniforms');
        });

        setUp(() async {
          cpuColors = List<ui.Color>.empty(growable: true);
          // uFloat
          cpuColors.add(ui.Color.fromARGB(255, random.nextInt(255), 0, 0));
          // uVec2
          cpuColors.add(ui.Color.fromARGB(255, random.nextInt(255), random.nextInt(255), 0));
          // uVec3
          cpuColors.add(
            ui.Color.fromARGB(255, random.nextInt(255), random.nextInt(255), random.nextInt(255)),
          );
          // uVec4
          cpuColors.add(
            ui.Color.fromARGB(255, random.nextInt(255), random.nextInt(255), random.nextInt(255)),
          );
          for (var i = 0; i < 10; ++i) {
            cpuColors.add(ui.Color.fromARGB(255, random.nextInt(255), 0, 0));
          }

          for (var i = 0; i < 10; ++i) {
            cpuColors.add(ui.Color.fromARGB(255, random.nextInt(255), random.nextInt(255), 0));
          }

          for (var i = 0; i < 10; ++i) {
            cpuColors.add(
              ui.Color.fromARGB(255, random.nextInt(255), random.nextInt(255), random.nextInt(255)),
            );
          }
          for (var i = 0; i < 10; ++i) {
            cpuColors.add(
              ui.Color.fromARGB(255, random.nextInt(255), random.nextInt(255), random.nextInt(255)),
            );
          }
        });

        test('set using setFloat', () async {
          final ui.FragmentShader shader = program.fragmentShader();
          // uFloat
          shader.setFloat(0, cpuColors[0].r);
          //uVec2
          shader.setFloat(1, cpuColors[1].r);
          shader.setFloat(2, cpuColors[1].g);
          //uVec3
          shader.setFloat(3, cpuColors[2].r);
          shader.setFloat(4, cpuColors[2].g);
          shader.setFloat(5, cpuColors[2].b);
          //uVec4
          shader.setFloat(6, cpuColors[3].r);
          shader.setFloat(7, cpuColors[3].g);
          shader.setFloat(8, cpuColors[3].b);
          shader.setFloat(9, cpuColors[3].a);

          var shaderOffset = 10;
          var colorOffset = 4;

          for (var i = 0; i < 10; ++i) {
            shader.setFloat(shaderOffset++, cpuColors[colorOffset++].r);
          }
          for (var i = 0; i < 10; ++i) {
            shader.setFloat(shaderOffset++, cpuColors[colorOffset].r);
            shader.setFloat(shaderOffset++, cpuColors[colorOffset++].g);
          }
          for (var i = 0; i < 10; ++i) {
            shader.setFloat(shaderOffset++, cpuColors[colorOffset].r);
            shader.setFloat(shaderOffset++, cpuColors[colorOffset].g);
            shader.setFloat(shaderOffset++, cpuColors[colorOffset++].b);
          }
          for (var i = 0; i < 10; ++i) {
            shader.setFloat(shaderOffset++, cpuColors[colorOffset].r);
            shader.setFloat(shaderOffset++, cpuColors[colorOffset].g);
            shader.setFloat(shaderOffset++, cpuColors[colorOffset].b);
            shader.setFloat(shaderOffset++, cpuColors[colorOffset++].a);
          }
          await _expectShaderRendersBarcode(shader, cpuColors);
        });

        test('set using getUniform*', () async {
          final ui.FragmentShader shader = program.fragmentShader();
          shader.getUniformFloat('uFloat').set(cpuColors[0].r);
          shader.getUniformVec2('uVec2').set(cpuColors[1].r, cpuColors[1].g);
          shader.getUniformVec3('uVec3').set(cpuColors[2].r, cpuColors[2].g, cpuColors[2].b);
          shader
              .getUniformVec4('uVec4')
              .set(cpuColors[3].r, cpuColors[3].g, cpuColors[3].b, cpuColors[3].a);

          final ui.UniformArray<ui.UniformFloatSlot> floatArray = shader.getUniformFloatArray(
            'uFloatArray',
          );
          final ui.UniformArray<ui.UniformVec2Slot> vec2Array = shader.getUniformVec2Array(
            'uVec2Array',
          );
          final ui.UniformArray<ui.UniformVec3Slot> vec3Array = shader.getUniformVec3Array(
            'uVec3Array',
          );
          final ui.UniformArray<ui.UniformVec4Slot> vec4Array = shader.getUniformVec4Array(
            'uVec4Array',
          );

          var colorOffset = 4;

          for (var i = 0; i < 10; ++i) {
            floatArray[i].set(cpuColors[colorOffset++].r);
          }
          for (var i = 0; i < 10; ++i) {
            vec2Array[i].set(cpuColors[colorOffset].r, cpuColors[colorOffset].g);
            ++colorOffset;
          }
          for (var i = 0; i < 10; ++i) {
            vec3Array[i].set(
              cpuColors[colorOffset].r,
              cpuColors[colorOffset].g,
              cpuColors[colorOffset].b,
            );
            ++colorOffset;
          }
          for (var i = 0; i < 10; ++i) {
            vec4Array[i].set(
              cpuColors[colorOffset].r,
              cpuColors[colorOffset].g,
              cpuColors[colorOffset].b,
              cpuColors[colorOffset].a,
            );
            ++colorOffset;
          }
          await _expectShaderRendersBarcode(shader, cpuColors);
        });
      });
    });
  }, skip: isWimp);
}

// Image of an oval painted with a linear gradient.
ui.Image _createOvalGradientImage({required int imageDimension}) {
  final recorder = ui.PictureRecorder();
  final canvas = ui.Canvas(recorder);
  canvas.drawPaint(ui.Paint()..color = const ui.Color(0xFF000000));
  canvas.drawOval(
    ui.Rect.fromCenter(
      center: ui.Offset(imageDimension * 0.5, imageDimension * 0.5),
      width: imageDimension * 0.6,
      height: imageDimension * 0.9,
    ),
    ui.Paint()
      ..shader = ui.Gradient.linear(
        ui.Offset.zero,
        ui.Offset(imageDimension.toDouble(), imageDimension.toDouble()),
        [const ui.Color(0xFFFF0000), const ui.Color(0xFF00FF00)],
      ),
  );
  final ui.Picture picture = recorder.endRecording();
  try {
    return picture.toImageSync(imageDimension, imageDimension);
  } finally {
    picture.dispose();
  }
}

ui.Image _imageFromShader({required ui.Shader shader, required int imageDimension}) {
  final recorder = ui.PictureRecorder();
  final canvas = ui.Canvas(recorder);
  final paint = ui.Paint()..shader = shader;
  canvas.drawPaint(paint);
  final ui.Picture picture = recorder.endRecording();
  return picture.toImageSync(imageDimension, imageDimension);
}

Future<ByteData?> _imageByteDataFromShader({
  required ui.Shader shader,
  int imageDimension = 100,
}) async {
  final ui.Image image = _imageFromShader(shader: shader, imageDimension: imageDimension);
  return image.toByteData();
}

Future<void> _expectShaderRendersColor(ui.Shader shader, ui.Color color) async {
  final ByteData renderedBytes = (await _imageByteDataFromShader(shader: shader))!;

  expect(renderedBytes.lengthInBytes % 4, 0);
  for (var byteOffset = 0; byteOffset < renderedBytes.lengthInBytes; byteOffset += 4) {
    final pixelColor = ui.Color.fromARGB(
      renderedBytes.getUint8(byteOffset + 3),
      renderedBytes.getUint8(byteOffset),
      renderedBytes.getUint8(byteOffset + 1),
      renderedBytes.getUint8(byteOffset + 2),
    );

    expect(pixelColor, color);
  }
}

Future<void> _expectShaderRendersBarcode(ui.Shader shader, List<ui.Color> barcodeColors) async {
  final ByteData renderedBytes = (await _imageByteDataFromShader(
    shader: shader,
    imageDimension: barcodeColors.length,
  ))!;

  expect(renderedBytes.lengthInBytes % 4, 0);
  final List<ui.Color> renderedColors = List.generate(barcodeColors.length, (int xCoord) {
    return ui.Color.fromARGB(
      renderedBytes.getUint8(xCoord * 4 + 3),
      renderedBytes.getUint8(xCoord * 4),
      renderedBytes.getUint8(xCoord * 4 + 1),
      renderedBytes.getUint8(xCoord * 4 + 2),
    );
  });

  for (var i = 0; i < barcodeColors.length; ++i) {
    final ui.Color renderedColor = renderedColors[i];
    final ui.Color expectedColor = barcodeColors[i];
    final reasonString =
        'Comparison failed on color $i. \nExpected: $expectedColor.\nActual: $renderedColor.';
    expect(renderedColor.r.clamp(-1, 1), closeTo(expectedColor.r, 0.06), reason: reasonString);
    expect(renderedColor.g.clamp(-1, 1), closeTo(expectedColor.g, 0.06), reason: reasonString);
    expect(renderedColor.b.clamp(-1, 1), closeTo(expectedColor.b, 0.06), reason: reasonString);
    expect(renderedColor.a.clamp(-1, 1), closeTo(expectedColor.a, 0.06), reason: reasonString);
  }
}
