// Copyright 2013 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "flutter/display_list/skia/dl_sk_conversions.h"

#include "flutter/display_list/dl_color.h"
#include "flutter/display_list/effects/dl_color_filters.h"
#include "flutter/display_list/effects/dl_color_sources.h"
#include "flutter/display_list/effects/dl_image_filters.h"
#include "flutter/display_list/geometry/dl_geometry_conversions.h"
#include "third_party/skia/include/core/SkColorFilter.h"
#include "third_party/skia/include/effects/SkGradient.h"
#include "third_party/skia/include/effects/SkImageFilters.h"
#include "third_party/skia/include/effects/SkRuntimeEffect.h"

namespace flutter {

// clang-format off
constexpr float kInvertColorMatrix[20] = {
  -1.0,    0,    0, 1.0, 0,
     0, -1.0,    0, 1.0, 0,
     0,    0, -1.0, 1.0, 0,
   1.0,  1.0,  1.0, 1.0, 0
};
// clang-format on

SkPaint ToSk(const DlPaint& paint) {
  SkPaint sk_paint;

  sk_paint.setAntiAlias(paint.isAntiAlias());
  sk_paint.setColor(ToSkColor4f(paint.getColor()));
  sk_paint.setBlendMode(ToSk(paint.getBlendMode()));
  sk_paint.setStyle(ToSk(paint.getDrawStyle()));
  sk_paint.setStrokeWidth(paint.getStrokeWidth());
  sk_paint.setStrokeMiter(paint.getStrokeMiter());
  sk_paint.setStrokeCap(ToSk(paint.getStrokeCap()));
  sk_paint.setStrokeJoin(ToSk(paint.getStrokeJoin()));
  sk_paint.setImageFilter(ToSk(paint.getImageFilterPtr()));
  auto color_filter = ToSk(paint.getColorFilterPtr());
  if (paint.isInvertColors()) {
    auto invert_filter = SkColorFilters::Matrix(kInvertColorMatrix);
    if (color_filter) {
      invert_filter = invert_filter->makeComposed(color_filter);
    }
    color_filter = invert_filter;
  }
  sk_paint.setColorFilter(color_filter);

  auto color_source = paint.getColorSourcePtr();
  if (color_source) {
    // Unconditionally set dither to true for gradient shaders.
    sk_paint.setDither(color_source->isGradient());
    sk_paint.setShader(ToSk(color_source));
  }

  sk_paint.setMaskFilter(ToSk(paint.getMaskFilterPtr()));

  return sk_paint;
}

SkPaint ToStrokedSk(const DlPaint& paint) {
  DlPaint stroked_paint = paint;
  stroked_paint.setDrawStyle(DlDrawStyle::kStroke);
  return ToSk(stroked_paint);
}

SkPaint ToNonShaderSk(const DlPaint& paint) {
  DlPaint non_shader_paint = paint;
  non_shader_paint.setColorSource(nullptr);
  return ToSk(non_shader_paint);
}

sk_sp<SkShader> ToSk(const DlColorSource* source) {
  if (!source) {
    return nullptr;
  }
  SkMatrix scratch_matrix;
  std::vector<SkColor4f> scratch_colors;
  static auto ToSkGradient =
      [](const DlGradientColorSourceBase* gradient,
         std::vector<SkColor4f>& scratch_colors) -> SkGradient {
    scratch_colors.clear();
    scratch_colors.reserve(gradient->stop_count());
    for (int i = 0; i < gradient->stop_count(); ++i) {
      DlColor color = gradient->colors()[i];
      // TODO(180904): avoid clamp and truncation
      scratch_colors.push_back(SkColor4f::FromColor(color.argb()));
      // DlColor esrgb = color.withColorSpace(DlColorSpace::kExtendedSRGB);
      // scratch_colors.emplace_back(esrgb.getRedF(), esrgb.getGreenF(),
      //                             esrgb.getBlueF(), esrgb.getAlphaF());
    }
    return SkGradient(
        SkGradient::Colors(scratch_colors,
                           {gradient->stops(), gradient->stop_count()},
                           ToSk(gradient->tile_mode())),
        SkGradient::Interpolation());
  };
  switch (source->type()) {
    case DlColorSourceType::kImage: {
      const DlImageColorSource* image_source = source->asImage();
      FML_DCHECK(image_source != nullptr);
      auto image = image_source->image();
      if (!image || !image->skia_image()) {
        return nullptr;
      }
      return image->skia_image()->makeShader(
          ToSk(image_source->horizontal_tile_mode()),
          ToSk(image_source->vertical_tile_mode()),
          ToSk(image_source->sampling()),
          ToSk(image_source->matrix_ptr(), scratch_matrix));
    }
    case DlColorSourceType::kLinearGradient: {
      const DlLinearGradientColorSource* linear_source =
          source->asLinearGradient();
      FML_DCHECK(linear_source != nullptr);
      SkPoint pts[] = {ToSkPoint(linear_source->start_point()),
                       ToSkPoint(linear_source->end_point())};
      return SkShaders::LinearGradient(
          pts, ToSkGradient(linear_source, scratch_colors),
          ToSk(linear_source->matrix_ptr(), scratch_matrix));
    }
    case DlColorSourceType::kRadialGradient: {
      const DlRadialGradientColorSource* radial_source =
          source->asRadialGradient();
      FML_DCHECK(radial_source != nullptr);
      return SkShaders::RadialGradient(
          ToSkPoint(radial_source->center()), radial_source->radius(),
          ToSkGradient(radial_source, scratch_colors),
          ToSk(radial_source->matrix_ptr(), scratch_matrix));
    }
    case DlColorSourceType::kConicalGradient: {
      const DlConicalGradientColorSource* conical_source =
          source->asConicalGradient();
      FML_DCHECK(conical_source != nullptr);
      return SkShaders::TwoPointConicalGradient(
          ToSkPoint(conical_source->start_center()),
          conical_source->start_radius(),
          ToSkPoint(conical_source->end_center()), conical_source->end_radius(),
          ToSkGradient(conical_source, scratch_colors),
          ToSk(conical_source->matrix_ptr(), scratch_matrix));
    }
    case DlColorSourceType::kSweepGradient: {
      const DlSweepGradientColorSource* sweep_source =
          source->asSweepGradient();
      FML_DCHECK(sweep_source != nullptr);
      return SkShaders::SweepGradient(
          SkPoint(sweep_source->center().x, sweep_source->center().y),
          sweep_source->start(), sweep_source->end(),
          ToSkGradient(sweep_source, scratch_colors),
          ToSk(sweep_source->matrix_ptr(), scratch_matrix));
    }
    case DlColorSourceType::kRuntimeEffect: {
      const DlRuntimeEffectColorSource* runtime_source =
          source->asRuntimeEffect();
      FML_DCHECK(runtime_source != nullptr);
      auto runtime_effect = runtime_source->runtime_effect();
      if (!runtime_effect || !runtime_effect->skia_runtime_effect()) {
        return nullptr;
      }

      auto samplers = runtime_source->samplers();
      std::vector<sk_sp<SkShader>> sk_samplers(samplers.size());
      for (size_t i = 0; i < samplers.size(); i++) {
        const auto& sampler = samplers[i];
        if (sampler == nullptr) {
          return nullptr;
        }
        sk_samplers[i] = ToSk(sampler);
      }

      auto uniform_data = runtime_source->uniform_data();
      auto ref = new std::shared_ptr<std::vector<uint8_t>>(uniform_data);
      auto sk_uniform_data = SkData::MakeWithProc(
          uniform_data->data(), uniform_data->size(),
          [](const void* ptr, void* context) {
            delete reinterpret_cast<std::shared_ptr<std::vector<uint8_t>>*>(
                context);
          },
          ref);

      return runtime_effect->skia_runtime_effect()->makeShader(
          sk_uniform_data, sk_samplers.data(), sk_samplers.size());
    }
  }
}

sk_sp<SkImageFilter> ToSk(const DlImageFilter* filter) {
  if (!filter) {
    return nullptr;
  }
  switch (filter->type()) {
    case DlImageFilterType::kBlur: {
      const DlBlurImageFilter* blur_filter = filter->asBlur();
      FML_DCHECK(blur_filter != nullptr);
      return SkImageFilters::Blur(blur_filter->sigma_x(),
                                  blur_filter->sigma_y(),
                                  ToSk(blur_filter->tile_mode()), nullptr);
    }
    case DlImageFilterType::kDilate: {
      const DlDilateImageFilter* dilate_filter = filter->asDilate();
      FML_DCHECK(dilate_filter != nullptr);
      return SkImageFilters::Dilate(dilate_filter->radius_x(),
                                    dilate_filter->radius_y(), nullptr);
    }
    case DlImageFilterType::kErode: {
      const DlErodeImageFilter* erode_filter = filter->asErode();
      FML_DCHECK(erode_filter != nullptr);
      return SkImageFilters::Erode(erode_filter->radius_x(),
                                   erode_filter->radius_y(), nullptr);
    }
    case DlImageFilterType::kMatrix: {
      const DlMatrixImageFilter* matrix_filter = filter->asMatrix();
      FML_DCHECK(matrix_filter != nullptr);
      return SkImageFilters::MatrixTransform(
          ToSkMatrix(matrix_filter->matrix()), ToSk(matrix_filter->sampling()),
          nullptr);
    }
    case DlImageFilterType::kCompose: {
      const DlComposeImageFilter* compose_filter = filter->asCompose();
      FML_DCHECK(compose_filter != nullptr);
      return SkImageFilters::Compose(ToSk(compose_filter->outer()),
                                     ToSk(compose_filter->inner()));
    }
    case DlImageFilterType::kColorFilter: {
      const DlColorFilterImageFilter* cf_filter = filter->asColorFilter();
      FML_DCHECK(cf_filter != nullptr);
      return SkImageFilters::ColorFilter(ToSk(cf_filter->color_filter()),
                                         nullptr);
    }
    case DlImageFilterType::kLocalMatrix: {
      const DlLocalMatrixImageFilter* lm_filter = filter->asLocalMatrix();
      FML_DCHECK(lm_filter != nullptr);
      sk_sp<SkImageFilter> skia_filter = ToSk(lm_filter->image_filter());
      // The image_filter property itself might have been null, or the
      // construction of the SkImageFilter might be optimized to null
      // for any number of reasons. In any case, if the filter is null
      // or optimizaed away, let's then optimize away this local matrix
      // case by returning null.
      if (!skia_filter) {
        return nullptr;
      }
      return skia_filter->makeWithLocalMatrix(ToSkMatrix(lm_filter->matrix()));
    }
    case DlImageFilterType::kRuntimeEffect:
      // UNSUPPORTED.
      return nullptr;
  }
}

sk_sp<SkColorFilter> ToSk(const DlColorFilter* filter) {
  if (!filter) {
    return nullptr;
  }
  switch (filter->type()) {
    case DlColorFilterType::kBlend: {
      const DlBlendColorFilter* blend_filter = filter->asBlend();
      FML_DCHECK(blend_filter != nullptr);
      return SkColorFilters::Blend(ToSkColor4f(blend_filter->color()), nullptr,
                                   ToSk(blend_filter->mode()));
    }
    case DlColorFilterType::kMatrix: {
      const DlMatrixColorFilter* matrix_filter = filter->asMatrix();
      FML_DCHECK(matrix_filter != nullptr);
      float matrix[20];
      matrix_filter->get_matrix(matrix);
      return SkColorFilters::Matrix(matrix);
    }
    case DlColorFilterType::kSrgbToLinearGamma: {
      return SkColorFilters::SRGBToLinearGamma();
    }
    case DlColorFilterType::kLinearToSrgbGamma: {
      return SkColorFilters::LinearToSRGBGamma();
    }
  }
}

sk_sp<SkMaskFilter> ToSk(const DlMaskFilter* filter) {
  if (!filter) {
    return nullptr;
  }
  switch (filter->type()) {
    case DlMaskFilterType::kBlur: {
      const DlBlurMaskFilter* blur_filter = filter->asBlur();
      FML_DCHECK(blur_filter != nullptr);
      return SkMaskFilter::MakeBlur(ToSk(blur_filter->style()),
                                    blur_filter->sigma(),
                                    blur_filter->respectCTM());
    }
  }
}

sk_sp<SkVertices> ToSk(const std::shared_ptr<DlVertices>& vertices) {
  std::vector<SkColor> sk_colors;
  const SkColor* sk_colors_ptr = nullptr;
  if (vertices->colors()) {
    sk_colors.reserve(vertices->vertex_count());
    for (int i = 0; i < vertices->vertex_count(); ++i) {
      sk_colors.push_back(vertices->colors()[i].argb());
    }
    sk_colors_ptr = sk_colors.data();
  }
  return SkVertices::MakeCopy(ToSk(vertices->mode()), vertices->vertex_count(),
                              ToSkPoints(vertices->vertex_data()),
                              ToSkPoints(vertices->texture_coordinate_data()),
                              sk_colors_ptr, vertices->index_count(),
                              vertices->indices());
}

}  // namespace flutter
