package com.exner.tools.meditationtimer.ui.destinations

import android.widget.Toast
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.consumeWindowInsets
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.CheckCircle
import androidx.compose.material.icons.filled.PlayArrow
import androidx.compose.material3.BottomAppBar
import androidx.compose.material3.BottomAppBarDefaults
import androidx.compose.material3.Checkbox
import androidx.compose.material3.ExtendedFloatingActionButton
import androidx.compose.material3.FloatingActionButtonDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.ListItem
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.meditationtimer.ui.BodyText
import com.exner.tools.meditationtimer.ui.DefaultSpacer
import com.exner.tools.meditationtimer.ui.HeaderText
import com.exner.tools.meditationtimer.ui.PushDataToWearOsViewModel
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.navigation.DestinationsNavigator

@Destination<RootGraph>
@Composable
fun PushDataToWearOs(
    pushDataToWearOsViewModel: PushDataToWearOsViewModel = hiltViewModel(),
    destinationsNavigator: DestinationsNavigator
) {
    val context = LocalContext.current

    val processes by pushDataToWearOsViewModel.allProcesses.collectAsStateWithLifecycle(
        emptyList()
    )
    val processesToExport by pushDataToWearOsViewModel.processesToExport.collectAsStateWithLifecycle()

    Scaffold(
        modifier = Modifier.imePadding(),
        content = { innerPadding ->
            Column(
                modifier = Modifier
                    .fillMaxSize()
                    .consumeWindowInsets(innerPadding)
                    .padding(innerPadding)
                    .padding(8.dp)
            ) {
                Text(text = "Select processes to push to your Wear OS device(s)")
                DefaultSpacer()
                LazyColumn {
                    items(items = processes) { mtProcess ->
                        var supText = "${mtProcess.processTime}/${mtProcess.intervalTime}"
                        if (mtProcess.hasAutoChain && null != mtProcess.gotoUuid && mtProcess.gotoUuid != "") {
                            supText += ". Next: '${mtProcess.gotoName}'"
                        }
                        ListItem(
                            headlineContent = { HeaderText(text = mtProcess.name) },
                            supportingContent = { BodyText(text = supText) },
                            leadingContent = {
                                Checkbox(
                                    checked = processesToExport.containsValue(mtProcess),
                                    onCheckedChange = {
                                        pushDataToWearOsViewModel.toggleProcess(mtProcess.uuid, it)
                                    }
                                )
                            }
                        )
                    }
                }
            }
        },
        bottomBar = {
            BottomAppBar(
                actions = {
                    IconButton(onClick = {
                        destinationsNavigator.navigateUp()
                    }) {
                        Icon(
                            imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                            contentDescription = "Cancel"
                        )
                    }
                    AnimatedVisibility(
                        visible = processes.size != processesToExport.size
                    ) {
                        IconButton(onClick = {
                            pushDataToWearOsViewModel.selectAllProcesses()
                        }) {
                            Icon(
                                imageVector = Icons.Default.CheckCircle,
                                contentDescription = "Select all"
                            )
                        }
                    }
                },
                floatingActionButton = {
                    AnimatedVisibility(
                        visible = processesToExport.isNotEmpty()
                    ) {
                        ExtendedFloatingActionButton(
                            text = {
                                Text(
                                    text = "Push ${processesToExport.size} Process" + if (processesToExport.size > 1) {
                                        "es"
                                    } else {
                                        ""
                                    }
                                )
                            },
                            icon = {
                                Icon(
                                    imageVector = Icons.Filled.PlayArrow,
                                    contentDescription = "Push Data"
                                )
                            },
                            onClick = {
                                pushDataToWearOsViewModel.pushSelectedProcesses(context, { numberOfPushedProcesses ->
                                    Toast.makeText(context, "Done pushing $numberOfPushedProcesses processes.", Toast.LENGTH_LONG).show()
                                })
                                destinationsNavigator.navigateUp()
                            },
                            containerColor = BottomAppBarDefaults.bottomAppBarFabColor,
                            elevation = FloatingActionButtonDefaults.bottomAppBarFabElevation()
                        )
                    }
                }
            )
        }
    )
}
