package com.exner.tools.meditationtimer.ui

import android.content.Context
import android.util.Log
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.meditationtimer.data.persistence.MeditationTimerDataRepository
import com.exner.tools.meditationtimer.data.persistence.MeditationTimerProcess
import com.exner.tools.meditationtimer.ui.helpers.shareProcessOnWear
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import javax.inject.Inject

@HiltViewModel
class PushDataToWearOsViewModel @Inject constructor(
    private val repository: MeditationTimerDataRepository
) : ViewModel() {

    val allProcesses = repository.observeProcesses

    private val _processesToExport: MutableStateFlow<Map<String, MeditationTimerProcess>> =
        MutableStateFlow(emptyMap())
    val processesToExport: StateFlow<Map<String, MeditationTimerProcess>> = _processesToExport

    init {
        selectAllProcesses()
    }

    fun selectAllProcesses() {
        viewModelScope.launch {
            val processList = repository.getAllProcesses()
            _processesToExport.value = processList.associate { process ->
                process.uuid to process
            }
        }
    }

    fun pushSelectedProcesses(context: Context, onSuccessCallback: (Int) -> Unit) {
        viewModelScope.launch {
            Log.d(TAG, "Going to push ${processesToExport.value.size} selected processes...")
            processesToExport.value.forEach { (uuid, meditationTimerProcess) ->
                shareProcessOnWear(context, meditationTimerProcess)
            }
            onSuccessCallback(processesToExport.value.size)
        }
    }

    fun toggleProcess(uuid: String, state: Boolean) {
        if (state) {
            viewModelScope.launch {
                val process = repository.loadProcessByUuid(uuid)
                if (process != null) {
                    val tempMap = processesToExport.value.toMutableMap()
                    tempMap.put(process.uuid, process)
                    _processesToExport.value = tempMap
                    // and the chain?
                    if (!process.gotoUuid.isNullOrEmpty()) {
                        toggleProcess(process.gotoUuid, true)
                    }
                }
            }
        } else {
            val tempMap = processesToExport.value.toMutableMap()
            tempMap.remove(uuid)
            _processesToExport.value = tempMap
        }
    }

    companion object {
        private const val TAG = "PushDataVM"
    }
}