package com.example.secreto

import java.security.SecureRandom
import javax.crypto.Cipher
import javax.crypto.SecretKey
import javax.crypto.SecretKeyFactory
import javax.crypto.spec.IvParameterSpec
import javax.crypto.spec.PBEKeySpec
import javax.crypto.spec.SecretKeySpec

class Crypto {

    companion object {
        private const val KEY_SIZE_BITS = 128
        private const val ITERATION_COUNT = 10000
        private const val SALT_SIZE_BYTES = 16
        private const val AES_KEY_SIZE_BYTES = KEY_SIZE_BITS / 8
        private const val IV_SIZE_BYTES = 16

        private fun generateSalt(): ByteArray {
            val secureRandom = SecureRandom()
            val salt = ByteArray(SALT_SIZE_BYTES)
            secureRandom.nextBytes(salt)
            return salt
        }

        private fun generateKey(password: String, salt: ByteArray): SecretKey {
            val keySpec = PBEKeySpec(password.toCharArray(), salt, ITERATION_COUNT, AES_KEY_SIZE_BYTES * 8)
            val secretKeyFactory = SecretKeyFactory.getInstance("PBKDF2WithHmacSHA1")
            val secretKey = secretKeyFactory.generateSecret(keySpec)
            return SecretKeySpec(secretKey.encoded, "AES")
        }

        private fun generateIV(): IvParameterSpec {
            val iv = ByteArray(IV_SIZE_BYTES)
            SecureRandom().nextBytes(iv)
            return IvParameterSpec(iv)
        }

        fun byte2hex(byteArray: ByteArray): String {
            val stringBuilder = StringBuilder()
            for (byte in byteArray) {
                val hex = String.format("%02X", byte)
                stringBuilder.append(hex)
            }
            return stringBuilder.toString()
        }

        fun hex2byte(hexString: String): ByteArray {
            val byteArray = ByteArray(hexString.length / 2)
            for (i in byteArray.indices) {
                val index = i * 2
                val hex = hexString.substring(index, index + 2)
                val byte = hex.toInt(16).toByte()
                byteArray[i] = byte
            }
            return byteArray
        }

        fun encrypt(plaintext: String, password: String): String {
            val salt = generateSalt()
            val secretKey = generateKey(password, salt)

            val iv = generateIV()
            val cipher = Cipher.getInstance("AES/CBC/PKCS7Padding")
            cipher.init(Cipher.ENCRYPT_MODE, secretKey, iv)

            val ciphertext = cipher.doFinal(plaintext.toByteArray(Charsets.UTF_8))
            val encryptedData = ByteArray(salt.size + iv.iv.size + ciphertext.size)

            System.arraycopy(salt, 0, encryptedData, 0, salt.size)
            System.arraycopy(iv.iv, 0, encryptedData, salt.size, iv.iv.size)
            System.arraycopy(ciphertext, 0, encryptedData, salt.size + iv.iv.size, ciphertext.size)

            return byte2hex(encryptedData)
        }

        fun decrypt(hexString: String, password: String): String? {
            try {
                val ciphertext = hex2byte(hexString)
                val salt = ciphertext.copyOfRange(0, SALT_SIZE_BYTES)
                val iv = ciphertext.copyOfRange(SALT_SIZE_BYTES, SALT_SIZE_BYTES + IV_SIZE_BYTES)
                val encryptedData = ciphertext.copyOfRange(SALT_SIZE_BYTES + IV_SIZE_BYTES, ciphertext.size)

                val secretKey = generateKey(password, salt)

                val cipher = Cipher.getInstance("AES/CBC/PKCS7Padding")
                cipher.init(Cipher.DECRYPT_MODE, secretKey, IvParameterSpec(iv))

                val decryptedData = cipher.doFinal(encryptedData)
                return decryptedData.toString(Charsets.UTF_8)
            } catch (e: Exception) {
                return null
            }
        }
    }
}