package com.example.notificationalerter

import android.app.Notification
import android.content.Intent
import android.net.Uri
import android.service.notification.NotificationListenerService
import android.service.notification.StatusBarNotification
import android.util.Log
import androidx.core.app.NotificationCompat
import androidx.core.app.NotificationManagerCompat
import com.example.notificationalerter.data.AppSelectionRepository
import com.example.notificationalerter.utils.NotificationUtils
import com.example.notificationalerter.utils.SoundManager
import java.util.Locale
import com.example.notificationalerter.data.NotificationHistoryItem
import com.example.notificationalerter.data.NotificationHistoryRepository
import java.util.Date

class MyNotificationListenerService : NotificationListenerService() {
    private lateinit var soundManager: SoundManager
    private lateinit var notificationManager: NotificationManagerCompat

    override fun onCreate() {
        super.onCreate()
        Log.d(TAG, "NotificationListenerService onCreate")

        soundManager = SoundManager(this)
        notificationManager = NotificationManagerCompat.from(this)
        NotificationUtils.createNotificationChannel(this)
    }

    override fun onDestroy() {
        super.onDestroy()
        searchWord = null
        Log.d(TAG, "NotificationListenerService onDestroy")
    }

    override fun onNotificationPosted(sbn: StatusBarNotification) {
        super.onNotificationPosted(sbn)
        Log.d(TAG, "Notification posted: ${sbn.packageName}")

        val packageName = sbn.packageName
        if (AppSelectionRepository.isSelected(packageName)) {
            val title = sbn.notification.extras.getString(Notification.EXTRA_TITLE)
            val text = sbn.notification.extras.getString(Notification.EXTRA_TEXT)
            Log.d(TAG, "Title: $title")
            Log.d(TAG, "Text: $text")

            if (containsSearchWord(text)) {
                soundManager.playSound(customSoundUri)
                sendNotificationBroadcast(packageName, title, text)

                val appName = try {
                    val appInfo = packageManager.getApplicationInfo(packageName, 0)
                    appInfo.loadLabel(packageManager).toString()
                } catch (e: Exception) {
                    packageName
                }

                val appIcon = try {
                    val appInfo = packageManager.getApplicationInfo(packageName, 0)
                    appInfo.loadIcon(packageManager)
                } catch (e: Exception) {
                    null
                }

                val historyItem = NotificationHistoryItem(
                    packageName = packageName,
                    appName = appName,
                    title = title,
                    text = text,
                    timestamp = Date()
                ).apply {
                    this.appIcon = appIcon
                }

                val repository = NotificationHistoryRepository.getInstance(this)
                repository.addHistoryItem(historyItem)
            }
        }
    }

    private fun containsSearchWord(text: String?): Boolean {
        if (searchWord == null || searchWord!!.isEmpty()) {
            return false
        }

        val words = text?.lowercase(Locale.getDefault())?.split("\\W+".toRegex())?.filter { it.isNotEmpty() }
        val lowercaseSearchWord = searchWord!!.lowercase(Locale.getDefault())

        return words?.any { it == lowercaseSearchWord } ?: false
    }

    private fun sendNotificationBroadcast(packageName: String, title: String?, text: String?) {
        // Check if notifications are enabled for the app
        if (!notificationManager.areNotificationsEnabled()) {
            Log.w(TAG, "Notifications are disabled for the app. Cannot show notification.")
            return // Exit early if permissions are missing
        }

        // Proceed to build and show the notification
        val intent = Intent(this, MainActivity::class.java).apply {
            flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
        }

        val pendingIntent = android.app.PendingIntent.getActivity(
            this, 0, intent, android.app.PendingIntent.FLAG_IMMUTABLE
        )

        val builder = NotificationCompat.Builder(this, NotificationUtils.CHANNEL_ID)
            .setSmallIcon(android.R.drawable.ic_lock_idle_low_battery)
            .setContentTitle(title)
            .setContentText(text)
            .setPriority(NotificationCompat.PRIORITY_DEFAULT)
            .setContentIntent(pendingIntent)
            .setAutoCancel(true)

        try {
            notificationManager.notify(NOTIFICATION_ID, builder.build())
        } catch (e: SecurityException) {
            Log.e(TAG, "SecurityException: Permission denied for notifications", e)
        }
    }

    companion object {
        private const val TAG = "NotificationListener"
        private const val NOTIFICATION_ID = 175

        private var searchWord: String? = null
        var customSoundUri: Uri? = null

        fun setSearchWord(word: String?) {
            searchWord = word
        }
    }
}