package com.example.controldecolonias.ui.screens.mapa

import android.content.Context
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.viewinterop.AndroidView
import com.example.controldecolonias.ColoniaViewModel
import org.osmdroid.config.Configuration
import org.osmdroid.tileprovider.tilesource.TileSourceFactory
import org.osmdroid.util.GeoPoint
import org.osmdroid.views.MapView
import org.osmdroid.views.overlay.Marker

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PantallaMapaNidos(
    viewModel: ColoniaViewModel
) {
    val nidos by viewModel.todosLosNidos.collectAsState(initial = emptyList())

    Scaffold(
        topBar = { TopAppBar(title = { Text("Mapa de Nidos") }) }
    ) { innerPadding ->
        AndroidView(
            modifier = Modifier.padding(innerPadding).fillMaxSize(),
            factory = { context ->
                // Configuración inicial del mapa
                Configuration.getInstance().load(context, context.getSharedPreferences("osmdroid", Context.MODE_PRIVATE))
                MapView(context).apply {
                    setTileSource(TileSourceFactory.MAPNIK) // Fuente de los mapas (OpenStreetMap estándar)
                    setMultiTouchControls(true) // Permite hacer zoom con los dedos
                }
            },
            update = { mapView ->
                // Centramos el mapa
                val startPoint = if (nidos.isNotEmpty()) {
                    GeoPoint(nidos.first().latitud, nidos.first().longitud)
                } else {
                    GeoPoint(20.38, -99.99) // Coordenadas de San Juan del Río como fallback
                }
                mapView.controller.setZoom(14.0)
                mapView.controller.setCenter(startPoint)

                // Borramos los marcadores antiguos para no duplicarlos
                mapView.overlays.clear()

                // Añadimos un marcador por cada nido en la base de datos
                nidos.forEach { nido ->
                    val marker = Marker(mapView)
                    marker.position = GeoPoint(nido.latitud, nido.longitud)
                    marker.setAnchor(Marker.ANCHOR_CENTER, Marker.ANCHOR_BOTTOM)
                    marker.title = nido.nombre
                    marker.snippet = nido.especie
                    mapView.overlays.add(marker)
                }

                // Forzamos al mapa a redibujarse con los nuevos marcadores
                mapView.invalidate()
            }
        )
    }
}