package com.example.controldecolonias.util

import android.nfc.NdefMessage
import android.nfc.NdefRecord
import android.nfc.Tag
import android.nfc.tech.Ndef
import com.example.controldecolonias.Colonia
import java.nio.charset.Charset

fun parsearPayloadNfc(payload: String): Colonia? {
    return try {
        val partes = payload.split(";")
            .filter { it.isNotBlank() }
            .associate {
                val (clave, valor) = it.split(":", limit = 2)
                clave to valor
            }

        val id = partes["ID"] ?: return null
        val nombre = partes["NM"] ?: "Sin Nombre"
        val especie = partes["SP"] ?: "Desconocida"
        val fecha = partes["DT"]?.toLongOrNull() ?: System.currentTimeMillis()
        val estados = partes["ST"]?.split(",") ?: emptyList()

        Colonia(
            id = id,
            nombre = nombre,
            especie = especie,
            fecha = fecha,
            // AÑADIDO: Leemos el clima y la temperatura del payload
            temperatura = partes["T"]?.toIntOrNull(),
            clima = partes["W"],
            tieneHuevos = estados.contains("H"),
            tieneLarvas = estados.contains("L"),
            tienePupas = estados.contains("P"),
            tieneObreras = estados.contains("O"),
            estaEnferma = estados.contains("E")
        )
    } catch (e: Exception) {
        null
    }
}

fun crearPayloadNfc(colonia: Colonia): String {
    val estado = buildString {
        if (colonia.tieneHuevos) append("H,")
        if (colonia.tieneLarvas) append("L,")
        if (colonia.tienePupas) append("P,")
        if (colonia.tieneObreras) append("O,")
        if (colonia.estaEnferma) append("E,")
    }.trimEnd(',')

    // AÑADIDO: Construimos la cadena con los nuevos datos del clima
    return buildString {
        append("ID:${colonia.id};")
        append("NM:${colonia.nombre};")
        append("SP:${colonia.especie};")
        append("DT:${colonia.fecha};")
        append("ST:$estado;")
        colonia.temperatura?.let { append("T:$it;") }
        colonia.clima?.let { if(it.isNotBlank()) append("W:$it;") }
    }
}

fun escribirEnEtiqueta(mensaje: String, tag: Tag?): Pair<Boolean, String> {
    val ndefRecord = NdefRecord.createTextRecord("es", mensaje)
    val ndefMessage = NdefMessage(arrayOf(ndefRecord))
    try {
        val ndef = Ndef.get(tag)
        ndef?.let {
            it.connect()
            if (it.maxSize < ndefMessage.toByteArray().size) {
                return Pair(false, "Error: La etiqueta no tiene suficiente espacio.")
            }
            it.writeNdefMessage(ndefMessage)
            it.close()
            return Pair(true, "¡Etiqueta escrita con éxito!")
        }
    } catch (e: Exception) {
        return Pair(false, "Error al escribir en la etiqueta: ${e.message}")
    }
    return Pair(false, "Error: No se pudo conectar con la etiqueta.")
}

fun borrarEtiquetaNfc(tag: Tag?): Pair<Boolean, String> {
    val emptyRecord = NdefRecord(NdefRecord.TNF_EMPTY, null, null, null)
    val emptyMessage = NdefMessage(arrayOf(emptyRecord))
    try {
        val ndef = Ndef.get(tag)
        ndef?.let {
            it.connect()
            it.writeNdefMessage(emptyMessage)
            it.close()
            return Pair(true, "¡Etiqueta borrada con éxito!")
        }
    } catch (e: Exception) {
        return Pair(false, "Error al borrar la etiqueta: ${e.message}")
    }
    return Pair(false, "Error: No se pudo conectar con la etiqueta.")
}