package com.example.controldecolonias

import android.content.Context
import androidx.room.Database
import androidx.room.Room
import androidx.room.RoomDatabase
import androidx.room.migration.Migration
import androidx.sqlite.db.SupportSQLiteDatabase

// VERSIÓN INCREMENTADA: De 8 a 9
@Database(entities = [Colonia::class, Nota::class, Nido::class, Alimentacion::class, Tarea::class, Foto::class], version = 9, exportSchema = false)
abstract class AppDatabase : RoomDatabase() {

    abstract fun coloniaDao(): ColoniaDao

    companion object {
        @Volatile
        private var INSTANCIA: AppDatabase? = null

        fun obtenerBaseDeDatos(context: Context): AppDatabase {
            return INSTANCIA ?: synchronized(this) {
                val instancia = Room.databaseBuilder(
                    context.applicationContext,
                    AppDatabase::class.java,
                    "colonias_database"
                )
                    // MIGRACIÓN AÑADIDA: La nueva migración se añade al final de la lista
                    .addMigrations(MIGRATION_1_2, MIGRATION_2_3, MIGRATION_3_4, MIGRATION_4_5, MIGRATION_5_6, MIGRATION_6_7, MIGRATION_7_8, MIGRATION_8_9)
                    .build()
                INSTANCIA = instancia
                instancia
            }
        }

        // --- MIGRACIÓN ANTERIOR (DE 7 a 8) ---
        val MIGRATION_7_8 = object : Migration(7, 8) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE colonias_table ADD COLUMN fechaRecordatorio INTEGER")
            }
        }

        // --- CÓDIGO DE MIGRACIÓN NUEVO (DE 8 a 9) ---
        val MIGRATION_8_9 = object : Migration(8, 9) {
            override fun migrate(db: SupportSQLiteDatabase) {
                // Modificar tabla de colonias
                db.execSQL("ALTER TABLE colonias_table RENAME TO colonias_table_old")
                db.execSQL("""
                    CREATE TABLE `colonias_table` (
                        `id` TEXT NOT NULL, `nombre` TEXT NOT NULL, `especie` TEXT NOT NULL, `fecha` INTEGER NOT NULL,
                        `latitud` REAL, `longitud` REAL, `temperatura` INTEGER, `clima` TEXT,
                        `tieneHuevos` INTEGER NOT NULL, `tieneLarvas` INTEGER NOT NULL, `tienePupas` INTEGER NOT NULL, `tieneObreras` INTEGER NOT NULL, `estaEnferma` INTEGER NOT NULL,
                        `tipoEtiqueta` TEXT NOT NULL,
                        `recordatorioProximaFecha` INTEGER, `recordatorioTipo` TEXT, `recordatorioIntervalo` INTEGER, `recordatorioHora` INTEGER,
                        PRIMARY KEY(`id`)
                    )
                """)
                db.execSQL("""
                    INSERT INTO colonias_table (id, nombre, especie, fecha, latitud, longitud, temperatura, clima, tieneHuevos, tieneLarvas, tienePupas, tieneObreras, estaEnferma, tipoEtiqueta, recordatorioProximaFecha)
                    SELECT id, nombre, especie, fecha, latitud, longitud, temperatura, clima, tieneHuevos, tieneLarvas, tienePupas, tieneObreras, estaEnferma, tipoEtiqueta, fechaRecordatorio
                    FROM colonias_table_old
                """)
                db.execSQL("DROP TABLE colonias_table_old")

                // Añadir campos a la tabla de tareas
                db.execSQL("ALTER TABLE tareas_table ADD COLUMN recordatorioProximaFecha INTEGER")
                db.execSQL("ALTER TABLE tareas_table ADD COLUMN recordatorioTipo TEXT")
                db.execSQL("ALTER TABLE tareas_table ADD COLUMN recordatorioIntervalo INTEGER")
                db.execSQL("ALTER TABLE tareas_table ADD COLUMN recordatorioHora INTEGER")
            }
        }

        // ... (resto de migraciones sin cambios)
        val MIGRATION_1_2 = object : Migration(1, 2) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("""
                    CREATE TABLE IF NOT EXISTS `notas_table` (
                        `id` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                        `coloniaId` TEXT NOT NULL,
                        `fecha` INTEGER NOT NULL,
                        `texto` TEXT NOT NULL,
                        FOREIGN KEY(`coloniaId`) REFERENCES `colonias_table`(`id`) ON UPDATE NO ACTION ON DELETE CASCADE
                    )
                """.trimIndent())
            }
        }
        val MIGRATION_2_3 = object : Migration(2, 3) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL(
                    """
                    CREATE TABLE IF NOT EXISTS `nidos_table` (
                        `id` TEXT NOT NULL,
                        `nombre` TEXT NOT NULL,
                        `especie` TEXT NOT NULL,
                        `fecha` INTEGER NOT NULL,
                        `latitud` REAL NOT NULL,
                        `longitud` REAL NOT NULL,
                        `notas` TEXT NOT NULL,
                        PRIMARY KEY(`id`)
                    )
                """.trimIndent()
                )
            }
        }

        val MIGRATION_3_4 = object : Migration(3, 4) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE colonias_table ADD COLUMN tipoEtiqueta TEXT NOT NULL DEFAULT 'Ninguna'")
            }
        }

        val MIGRATION_4_5 = object : Migration(4, 5) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("""
                    CREATE TABLE IF NOT EXISTS `alimentacion_table` (
                        `id` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                        `coloniaId` TEXT NOT NULL,
                        `fecha` INTEGER NOT NULL,
                        `tipoComida` TEXT NOT NULL,
                        `nota` TEXT NOT NULL,
                        FOREIGN KEY(`coloniaId`) REFERENCES `colonias_table`(`id`) ON UPDATE NO ACTION ON DELETE CASCADE
                    )
                """.trimIndent())
            }
        }

        val MIGRATION_5_6 = object : Migration(5, 6) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("""
                    CREATE TABLE IF NOT EXISTS `tareas_table` (
                        `id` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                        `texto` TEXT NOT NULL,
                        `fechaLimite` INTEGER,
                        `completada` INTEGER NOT NULL DEFAULT 0,
                        `fechaCreacion` INTEGER NOT NULL
                    )
                """.trimIndent())
            }
        }

        val MIGRATION_6_7 = object : Migration(6, 7) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("""
                    CREATE TABLE IF NOT EXISTS `fotos_table` (
                        `id` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                        `coloniaId` TEXT NOT NULL,
                        `uri` TEXT NOT NULL,
                        `descripcion` TEXT NOT NULL,
                        `fecha` INTEGER NOT NULL,
                        FOREIGN KEY(`coloniaId`) REFERENCES `colonias_table`(`id`) ON UPDATE NO ACTION ON DELETE CASCADE
                    )
                """.trimIndent())
            }
        }
    }
}