package com.example.controldecolonias

import android.content.Context
import android.content.Intent
import android.net.Uri
import android.nfc.Tag
import android.nfc.tech.Ndef
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.viewModelScope
import com.example.controldecolonias.util.AlarmScheduler
import com.example.controldecolonias.util.SettingsManager
import com.example.controldecolonias.util.borrarEtiquetaNfc
import com.example.controldecolonias.util.crearPayloadNfc
import com.example.controldecolonias.util.escribirEnEtiqueta
import com.example.controldecolonias.util.generarIdCortoUnico
import com.example.controldecolonias.util.parsearPayloadNfc
import com.google.gson.Gson
import kotlinx.coroutines.flow.*
import kotlinx.coroutines.launch
import java.io.BufferedReader
import java.io.InputStreamReader
import java.nio.charset.Charset

enum class ScanMode { NFC, QR }

sealed class NfcMode {
    object Inactivo : NfcMode()
    object Escaneando : NfcMode()
    object EscaneandoParaAnadir : NfcMode()
    data class Escribiendo(val payload: String) : NfcMode()
    object Borrando : NfcMode()
}
sealed class ResultadoScan {
    object Inactivo : ResultadoScan()
    data class Exitoso(val idColonia: String) : ResultadoScan()
    data class Desconocido(val coloniaEnTag: Colonia) : ResultadoScan()
    data class ConflictoIdExistente(val coloniaEnTag: Colonia) : ResultadoScan()
    data class ConflictoNuevaColonia(val coloniaConNuevoId: Colonia) : ResultadoScan()
    data class Error(val mensaje: String) : ResultadoScan()

    data class IdYaExisteAlAnadir(val id: String, val esNfc: Boolean) : ResultadoScan()
    data class IdValidoParaAnadir(val id: String) : ResultadoScan()
    data class ColoniaValidaParaAnadir(val colonia: Colonia, val idYaExiste: Boolean) : ResultadoScan()
}

class ColoniaViewModel(private val dao: ColoniaDao, private val settingsManager: SettingsManager) : ViewModel() {

    data class RecordatorioConfig(
        val tipo: String,
        val intervalo: Int?,
        val hora: Int,
        val proximaFecha: Long
    )

    private val _textoBusqueda = MutableStateFlow("")
    val textoBusqueda = _textoBusqueda.asStateFlow()

    fun onTextoBusquedaChange(nuevoTexto: String) {
        _textoBusqueda.value = nuevoTexto
    }

    @OptIn(kotlinx.coroutines.FlowPreview::class)
    val coloniasFiltradasYAgrupadas: StateFlow<Map<String, List<Colonia>>> =
        dao.obtenerTodasLasColonias()
            .combine(_textoBusqueda) { colonias, texto ->
                if (texto.isBlank()) {
                    colonias
                } else {
                    colonias.filter {
                        it.nombre.contains(texto, ignoreCase = true) || it.especie.contains(texto, ignoreCase = true)
                    }
                }
            }
            .map { it.groupBy { colonia -> colonia.especie } }
            .stateIn(viewModelScope, SharingStarted.WhileSubscribed(5000), emptyMap())

    @OptIn(kotlinx.coroutines.FlowPreview::class)
    val nidosFiltradosYAgrupados: StateFlow<Map<String, List<Nido>>> =
        dao.obtenerTodosLosNidos()
            .combine(_textoBusqueda) { nidos, texto ->
                if (texto.isBlank()) {
                    nidos
                } else {
                    nidos.filter {
                        it.nombre.contains(texto, ignoreCase = true) || it.especie.contains(texto, ignoreCase = true)
                    }
                }
            }
            .map { it.groupBy { nido -> nido.especie } }
            .stateIn(viewModelScope, SharingStarted.WhileSubscribed(5000), emptyMap())

    private val _scanMode = MutableStateFlow(ScanMode.NFC)
    val scanMode = _scanMode.asStateFlow()
    fun setScanMode(mode: ScanMode) { _scanMode.value = mode }

    private val _nfcMode = MutableStateFlow<NfcMode>(NfcMode.Inactivo)
    val nfcMode = _nfcMode.asStateFlow()
    private val _resultadoScan = MutableStateFlow<ResultadoScan>(ResultadoScan.Inactivo)
    val resultadoScan = _resultadoScan.asStateFlow()

    val apiKey: StateFlow<String> = settingsManager.apiKeyFlow.stateIn(
        scope = viewModelScope,
        started = SharingStarted.WhileSubscribed(5000),
        initialValue = ""
    )
    fun guardarApiKey(apiKey: String) {
        viewModelScope.launch { settingsManager.guardarApiKey(apiKey) }
    }

    val ninjaModeActivado: StateFlow<Boolean> = settingsManager.ninjaModeFlow.stateIn(
        scope = viewModelScope,
        started = SharingStarted.WhileSubscribed(5000),
        initialValue = false
    )

    fun guardarNinjaMode(activado: Boolean) {
        viewModelScope.launch { settingsManager.guardarNinjaMode(activado) }
    }


    val todasLasColonias: Flow<List<Colonia>> = dao.obtenerTodasLasColonias()

    fun obtenerColoniaFullDetail(id: String): Flow<ColoniaFullDetail?> = dao.obtenerColoniaFullDetail(id)

    fun insertar(colonia: Colonia) = viewModelScope.launch { dao.insertar(colonia) }
    fun actualizar(colonia: Colonia) = viewModelScope.launch { dao.actualizar(colonia) }
    fun borrar(colonia: Colonia) = viewModelScope.launch { dao.borrar(colonia) }
    suspend fun idExiste(id: String): Boolean = dao.idExiste(id)
    fun anadirNota(coloniaId: String, texto: String) = viewModelScope.launch { dao.insertarNota(Nota(coloniaId = coloniaId, fecha = System.currentTimeMillis(), texto = texto)) }
    fun actualizarNota(nota: Nota) = viewModelScope.launch { dao.actualizarNota(nota) }
    fun borrarNota(nota: Nota) = viewModelScope.launch { dao.borrarNota(nota) }

    val todosLosNidos: Flow<List<Nido>> = dao.obtenerTodosLosNidos()
    fun obtenerNido(id: String): Flow<Nido?> = dao.obtenerNidoPorId(id)
    fun insertarNido(nido: Nido) = viewModelScope.launch { dao.insertarNido(nido) }
    fun borrarNido(nido: Nido) = viewModelScope.launch { dao.borrarNido(nido) }

    fun obtenerAlimentacion(coloniaId: String): Flow<List<Alimentacion>> = dao.obtenerAlimentacionPorColoniaId(coloniaId)
    fun anadirAlimentacion(coloniaId: String, tipoComida: String, nota: String) = viewModelScope.launch {
        dao.insertarAlimentacion(Alimentacion(coloniaId = coloniaId, fecha = System.currentTimeMillis(), tipoComida = tipoComida, nota = nota))
    }
    fun actualizarAlimentacion(registro: Alimentacion) = viewModelScope.launch { dao.actualizarAlimentacion(registro) }
    fun borrarAlimentacion(registro: Alimentacion) = viewModelScope.launch { dao.borrarAlimentacion(registro) }

    val todasLasTareas: StateFlow<List<Tarea>> = dao.obtenerTodasLasTareas()
        .stateIn(viewModelScope, SharingStarted.WhileSubscribed(5000), emptyList())

    fun insertarTarea(tarea: Tarea) = viewModelScope.launch { dao.insertarTarea(tarea) }
    fun actualizarTarea(tarea: Tarea) = viewModelScope.launch { dao.actualizarTarea(tarea) }

    fun borrarTarea(tarea: Tarea, context: Context) = viewModelScope.launch {
        dao.borrarTarea(tarea)
        AlarmScheduler.cancel(context, tarea = tarea)
    }

    fun anadirFoto(coloniaId: String, uri: Uri, context: Context) {
        viewModelScope.launch {
            val contentResolver = context.contentResolver
            contentResolver.takePersistableUriPermission(uri, Intent.FLAG_GRANT_READ_URI_PERMISSION)
            val foto = Foto(coloniaId = coloniaId, uri = uri.toString(), descripcion = "")
            dao.insertarFoto(foto)
        }
    }

    fun borrarFoto(foto: Foto) = viewModelScope.launch { dao.borrarFoto(foto) }

    fun activarModoEscaneo() { _nfcMode.value = NfcMode.Escaneando }
    fun activarModoEscaneoParaAnadir() { _nfcMode.value = NfcMode.EscaneandoParaAnadir }
    fun activarModoEscritura(payload: String) { _nfcMode.value = NfcMode.Escribiendo(payload) }
    fun activarModoBorrado() { _nfcMode.value = NfcMode.Borrando }
    fun cancelarModoNfc() { _nfcMode.value = NfcMode.Inactivo }

    fun onTagDetectado(tag: Tag) {
        val modoActual = _nfcMode.value
        if (modoActual == NfcMode.Inactivo) return

        val payload = try {
            val ndef = Ndef.get(tag)
            ndef?.connect()
            val ndefMessage = ndef?.cachedNdefMessage
            val rawPayload = ndefMessage?.records?.firstOrNull()?.payload ?: byteArrayOf()
            String(rawPayload, Charset.forName("UTF-8")).drop(3)
        } catch (e: Exception) {
            _resultadoScan.value = ResultadoScan.Error("No se pudo leer la etiqueta.")
            null
        } finally {
            Ndef.get(tag)?.close()
        }

        if (payload == null) return

        when (modoActual) {
            is NfcMode.Escaneando -> procesarQrLeido(payload)
            is NfcMode.EscaneandoParaAnadir -> procesarScanParaAnadir(payload, esNfc = true)
            is NfcMode.Escribiendo -> escribirEnEtiqueta(modoActual.payload, tag)
            is NfcMode.Borrando -> borrarEtiquetaNfc(tag)
            else -> {}
        }
        _nfcMode.value = NfcMode.Inactivo
    }

    fun resetearScan() { _resultadoScan.value = ResultadoScan.Inactivo }

    fun procesarQrLeido(qrData: String) {
        viewModelScope.launch {
            if (qrData.contains(";") && qrData.contains(":")) {
                val coloniaEnTag = parsearPayloadNfc(qrData)
                if (coloniaEnTag == null) {
                    _resultadoScan.value = ResultadoScan.Error("Formato de código no válido.")
                    return@launch
                }
                if (dao.idExiste(coloniaEnTag.id)) {
                    _resultadoScan.value = ResultadoScan.Exitoso(coloniaEnTag.id)
                } else {
                    _resultadoScan.value = ResultadoScan.Desconocido(coloniaEnTag)
                }
            } else {
                if (dao.idExiste(qrData)) {
                    _resultadoScan.value = ResultadoScan.Exitoso(qrData)
                } else {
                    _resultadoScan.value = ResultadoScan.Error("La colonia con ID '$qrData' no está en la base de datos.")
                }
            }
        }
    }

    fun procesarScanParaAnadir(scannedData: String, esNfc: Boolean) {
        viewModelScope.launch {
            if (scannedData.contains(";") && scannedData.contains(":")) {
                val coloniaEnTag = parsearPayloadNfc(scannedData)
                if (coloniaEnTag == null) {
                    _resultadoScan.value = ResultadoScan.Error("Formato de código no válido.")
                    return@launch
                }
                val existe = dao.idExiste(coloniaEnTag.id)
                _resultadoScan.value = ResultadoScan.ColoniaValidaParaAnadir(coloniaEnTag, idYaExiste = existe)
            } else {
                if (dao.idExiste(scannedData)) {
                    _resultadoScan.value = ResultadoScan.IdYaExisteAlAnadir(scannedData, esNfc)
                } else {
                    _resultadoScan.value = ResultadoScan.IdValidoParaAnadir(scannedData)
                }
            }
        }
    }

    fun importarComoNueva(colonia: Colonia) = viewModelScope.launch {
        val nuevoId = generarIdCortoUnico(this@ColoniaViewModel)
        val nuevaColonia = colonia.copy(id = nuevoId, tipoEtiqueta = "Ninguna")
        dao.insertar(nuevaColonia)
    }

    val totalColonias: StateFlow<Int> = dao.contarColonias().stateIn(viewModelScope, SharingStarted.WhileSubscribed(5000), 0)
    val totalNidos: StateFlow<Int> = dao.contarNidos().stateIn(viewModelScope, SharingStarted.WhileSubscribed(5000), 0)
    val totalEnfermas: StateFlow<Int> = dao.contarColoniasEnfermas().stateIn(viewModelScope, SharingStarted.WhileSubscribed(5000), 0)

    fun exportarColoniaEspecifica(coloniaId: String, onDataReady: (String) -> Unit) {
        viewModelScope.launch {
            val colonia = dao.obtenerColoniaPorIdSync(coloniaId)
            if (colonia != null) {
                val backupData = BackupData(
                    colonias = listOf(colonia),
                    notas = dao.exportarNotasDeColonia(coloniaId),
                    alimentacion = dao.exportarAlimentacionDeColonia(coloniaId)
                )
                val jsonString = Gson().toJson(backupData)
                onDataReady(jsonString)
            }
        }
    }

    fun exportarDatos(onDataReady: (String) -> Unit) {
        viewModelScope.launch {
            val backupData = BackupData(
                colonias = dao.exportarColonias(),
                notas = dao.exportarNotas(),
                nidos = dao.exportarNidos(),
                alimentacion = dao.exportarAlimentacion(),
                tareas = dao.exportarTareas(),
                fotos = dao.exportarFotos()
            )
            val jsonString = Gson().toJson(backupData)
            onDataReady(jsonString)
        }
    }

    fun importarDatos(uri: Uri, context: Context, onResult: (Boolean, String) -> Unit) {
        viewModelScope.launch {
            try {
                val inputStream = context.contentResolver.openInputStream(uri)
                val reader = BufferedReader(InputStreamReader(inputStream))
                val jsonString = reader.readText()
                val backupData = Gson().fromJson(jsonString, BackupData::class.java)
                dao.importarBackup(backupData)
                onResult(true, "¡Datos importados con éxito!")
            } catch (e: Exception) {
                e.printStackTrace()
                onResult(false, "Error al importar: ${e.message}")
            }
        }
    }

    fun exportarNidos(onDataReady: (String) -> Unit) {
        viewModelScope.launch {
            val backupData = BackupData(
                nidos = dao.exportarNidos()
            )
            val jsonString = Gson().toJson(backupData)
            onDataReady(jsonString)
        }
    }

    fun importarDatosSelectivo(uri: Uri, context: Context, onResult: (Boolean, String) -> Unit) {
        viewModelScope.launch {
            try {
                val inputStream = context.contentResolver.openInputStream(uri)
                val reader = BufferedReader(InputStreamReader(inputStream))
                val jsonString = reader.readText()
                val backupData = Gson().fromJson(jsonString, BackupData::class.java)

                var coloniasImportadas = 0
                var nidosImportados = 0

                backupData.colonias.forEach { dao.insertar(it); coloniasImportadas++ }
                backupData.nidos.forEach { dao.insertarNido(it); nidosImportados++ }
                backupData.notas.forEach { dao.insertarNota(it) }
                backupData.alimentacion.forEach { dao.insertarAlimentacion(it) }
                backupData.fotos.forEach { dao.insertarFoto(it) }
                backupData.tareas.forEach { dao.insertarTarea(it) }

                onResult(true, "Importación selectiva completada:\n$coloniasImportadas colonias\n$nidosImportados nidos")
            } catch (e: Exception) {
                e.printStackTrace()
                onResult(false, "Error al importar: ${e.message}")
            }
        }
    }

    // --- FUNCIONES DE RECORDATORIOS ---
    fun guardarRecordatorio(colonia: Colonia, config: RecordatorioConfig, context: Context) = viewModelScope.launch {
        val coloniaActualizada = colonia.copy(
            recordatorioProximaFecha = config.proximaFecha,
            recordatorioTipo = config.tipo,
            recordatorioIntervalo = config.intervalo,
            recordatorioHora = config.hora
        )
        actualizar(coloniaActualizada)
        AlarmScheduler.schedule(context, coloniaActualizada)
    }

    fun cancelarRecordatorio(colonia: Colonia, context: Context) = viewModelScope.launch {
        AlarmScheduler.cancel(context, colonia)
        val coloniaActualizada = colonia.copy(
            recordatorioProximaFecha = null,
            recordatorioTipo = null,
            recordatorioIntervalo = null,
            recordatorioHora = null
        )
        actualizar(coloniaActualizada)
    }

    fun guardarRecordatorioTarea(tarea: Tarea, context: Context) = viewModelScope.launch {
        if (tarea.id == 0) {
            val nuevoId = dao.insertarTarea(tarea)
            val tareaConId = tarea.copy(id = nuevoId.toInt())

            if (tareaConId.recordatorioProximaFecha != null) {
                AlarmScheduler.schedule(context, tarea = tareaConId)
            }
        } else {
            actualizarTarea(tarea)
            if (tarea.recordatorioProximaFecha != null) {
                AlarmScheduler.schedule(context, tarea = tarea)
            } else {
                AlarmScheduler.cancel(context, tarea = tarea)
            }
        }
    }
}

class ColoniaViewModelFactory(
    private val dao: ColoniaDao,
    private val settingsManager: SettingsManager
) : ViewModelProvider.Factory {
    override fun <T : ViewModel> create(modelClass: Class<T>): T {
        if (modelClass.isAssignableFrom(ColoniaViewModel::class.java)) {
            @Suppress("UNCHECKED_CAST")
            return ColoniaViewModel(dao, settingsManager) as T
        }
        throw IllegalArgumentException("Unknown ViewModel class")
    }
}