package com.example.controldecolonias.ui.screens.tareas

import android.app.DatePickerDialog
import android.app.TimePickerDialog
import android.widget.Toast
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.Alarm
import androidx.compose.material.icons.filled.AlarmOn
import androidx.compose.material.icons.filled.Close
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.unit.dp
import com.example.controldecolonias.ColoniaViewModel
import com.example.controldecolonias.Tarea
import com.example.controldecolonias.ui.components.TareaItem
import com.example.controldecolonias.ui.screens.colonias.DialogoConfigurarRecordatorio
import com.example.controldecolonias.util.formatearTimestamp
import java.util.Calendar

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PantallaPrincipalTareas(viewModel: ColoniaViewModel) {
    val tareas by viewModel.todasLasTareas.collectAsState()
    var mostrarDialogo by remember { mutableStateOf(false) }
    var tareaAEditar by remember { mutableStateOf<Tarea?>(null) }
    val context = LocalContext.current // Obtenemos el context aquí para usarlo en el borrado

    Scaffold(
        topBar = { TopAppBar(title = { Text("Gestor de Tareas") }) },
        floatingActionButton = {
            FloatingActionButton(onClick = {
                tareaAEditar = null
                mostrarDialogo = true
            }) {
                Icon(Icons.Default.Add, contentDescription = "Añadir Tarea")
            }
        }
    ) { innerPadding ->
        if (tareas.isEmpty()) {
            Box(modifier = Modifier.fillMaxSize().padding(innerPadding), contentAlignment = Alignment.Center) {
                Text("No hay tareas pendientes.\nPulsa '+' para añadir la primera.")
            }
        } else {
            LazyColumn(
                modifier = Modifier
                    .padding(innerPadding)
                    .padding(horizontal = 8.dp)
            ) {
                items(tareas.sortedBy { it.completada }) { tarea ->
                    TareaItem(
                        tarea = tarea,
                        onCheckedChange = { completada ->
                            viewModel.actualizarTarea(tarea.copy(completada = completada))
                        },
                        onClick = {
                            tareaAEditar = tarea
                            mostrarDialogo = true
                        }
                    )
                }
            }
        }
    }

    if (mostrarDialogo) {
        DialogoAnadirEditarTarea(
            tarea = tareaAEditar,
            viewModel = viewModel,
            onDismiss = { mostrarDialogo = false },
            onDelete = { tarea ->
                // Pasamos el context a la función de borrado
                viewModel.borrarTarea(tarea, context)
                mostrarDialogo = false
            }
        )
    }
}

@Composable
fun DialogoAnadirEditarTarea(
    tarea: Tarea?,
    viewModel: ColoniaViewModel,
    onDismiss: () -> Unit,
    onDelete: (Tarea) -> Unit
) {
    var texto by remember { mutableStateOf(tarea?.texto ?: "") }
    var fechaLimite by remember { mutableStateOf(tarea?.fechaLimite) }
    val context = LocalContext.current
    var mostrarDialogoRecordatorio by remember { mutableStateOf(false) }

    var tempRecordatorioProximaFecha by remember { mutableStateOf(tarea?.recordatorioProximaFecha) }
    var tempRecordatorioTipo by remember { mutableStateOf(tarea?.recordatorioTipo) }
    var tempRecordatorioIntervalo by remember { mutableStateOf(tarea?.recordatorioIntervalo) }
    var tempRecordatorioHora by remember { mutableStateOf(tarea?.recordatorioHora) }

    val calendar = Calendar.getInstance()
    fechaLimite?.let { calendar.timeInMillis = it }
    val datePickerDialog = DatePickerDialog(
        context,
        { _, year, month, dayOfMonth ->
            val newCal = Calendar.getInstance()
            newCal.set(year, month, dayOfMonth)
            fechaLimite = newCal.timeInMillis
        },
        calendar.get(Calendar.YEAR),
        calendar.get(Calendar.MONTH),
        calendar.get(Calendar.DAY_OF_MONTH)
    )

    val timePickerDialog = TimePickerDialog(
        context,
        { _, hourOfDay, minute ->
            val newCal = Calendar.getInstance()
            fechaLimite?.let { newCal.timeInMillis = it }
            newCal.set(Calendar.HOUR_OF_DAY, hourOfDay)
            newCal.set(Calendar.MINUTE, minute)
            fechaLimite = newCal.timeInMillis
        },
        calendar.get(Calendar.HOUR_OF_DAY),
        calendar.get(Calendar.MINUTE),
        true
    )

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text(if (tarea == null) "Nueva Tarea" else "Editar Tarea") },
        text = {
            Column {
                OutlinedTextField(
                    value = texto,
                    onValueChange = { texto = it },
                    label = { Text("Descripción de la tarea") },
                    modifier = Modifier.fillMaxWidth()
                )
                Spacer(Modifier.height(16.dp))
                Text("Fecha Límite (opcional):", style = MaterialTheme.typography.labelMedium)
                Row(verticalAlignment = Alignment.CenterVertically) {
                    OutlinedButton(onClick = { datePickerDialog.show() }, modifier = Modifier.weight(1f)) {
                        Text(fechaLimite?.let { formatearTimestamp(it, "dd/MM/yyyy") } ?: "Elegir fecha")
                    }
                    Spacer(Modifier.width(8.dp))
                    OutlinedButton(onClick = { timePickerDialog.show() }, modifier = Modifier.weight(1f)) {
                        Text(fechaLimite?.let { formatearTimestamp(it, "HH:mm") } ?: "Elegir hora")
                    }
                    if (fechaLimite != null) {
                        IconButton(onClick = { fechaLimite = null }) {
                            Icon(Icons.Default.Close, contentDescription = "Limpiar fecha")
                        }
                    }
                }
                Spacer(Modifier.height(16.dp))

                if (tempRecordatorioTipo == null) {
                    Button(onClick = { mostrarDialogoRecordatorio = true }, modifier = Modifier.fillMaxWidth()) {
                        Icon(imageVector = Icons.Default.Alarm, contentDescription = "Añadir Recordatorio", modifier = Modifier.size(ButtonDefaults.IconSize))
                        Spacer(Modifier.width(ButtonDefaults.IconSpacing))
                        Text("Añadir Recordatorio")
                    }
                } else {
                    OutlinedButton(onClick = { mostrarDialogoRecordatorio = true }, modifier = Modifier.fillMaxWidth()) {
                        Icon(imageVector = Icons.Default.AlarmOn, contentDescription = "Editar Recordatorio", modifier = Modifier.size(ButtonDefaults.IconSize))
                        Spacer(Modifier.width(ButtonDefaults.IconSpacing))
                        Text("Editar Recordatorio")
                    }
                }
            }
        },
        confirmButton = {
            Button(onClick = {
                if (texto.isNotBlank()) {
                    val tareaFinal = tarea?.copy(
                        texto = texto,
                        fechaLimite = fechaLimite,
                        recordatorioProximaFecha = tempRecordatorioProximaFecha,
                        recordatorioTipo = tempRecordatorioTipo,
                        recordatorioIntervalo = tempRecordatorioIntervalo,
                        recordatorioHora = tempRecordatorioHora
                    ) ?: Tarea(
                        texto = texto,
                        fechaLimite = fechaLimite,
                        recordatorioProximaFecha = tempRecordatorioProximaFecha,
                        recordatorioTipo = tempRecordatorioTipo,
                        recordatorioIntervalo = tempRecordatorioIntervalo,
                        recordatorioHora = tempRecordatorioHora
                    )

                    viewModel.guardarRecordatorioTarea(tareaFinal, context)
                    onDismiss()

                } else {
                    Toast.makeText(context, "La descripción no puede estar vacía", Toast.LENGTH_SHORT).show()
                }
            }) { Text("Guardar") }
        },
        dismissButton = {
            Row {
                if (tarea != null) {
                    TextButton(onClick = { onDelete(tarea) }) {
                        Text("Eliminar", color = MaterialTheme.colorScheme.error)
                    }
                }
                Spacer(modifier = Modifier.weight(1f))
                TextButton(onClick = onDismiss) { Text("Cancelar") }
            }
        }
    )

    if (mostrarDialogoRecordatorio) {
        DialogoConfigurarRecordatorio(
            tipoInicial = tempRecordatorioTipo,
            intervaloInicial = tempRecordatorioIntervalo,
            horaInicial = tempRecordatorioHora,
            onDismiss = { mostrarDialogoRecordatorio = false },
            onConfirm = { tipo, intervalo, hora, proximaFecha ->
                tempRecordatorioTipo = tipo
                tempRecordatorioIntervalo = intervalo
                tempRecordatorioHora = hora
                tempRecordatorioProximaFecha = proximaFecha
                mostrarDialogoRecordatorio = false
            },
            onCancel = {
                tempRecordatorioTipo = null
                tempRecordatorioIntervalo = null
                tempRecordatorioHora = null
                tempRecordatorioProximaFecha = null
                mostrarDialogoRecordatorio = false
            }
        )
    }
}