package com.example.controldecolonias.ui.screens.nidos

import android.Manifest
import android.content.Intent
import android.content.pm.PackageManager
import android.net.Uri
import android.widget.Toast
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.Map
import androidx.compose.material.icons.filled.Search
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import androidx.core.content.ContextCompat
import androidx.navigation.NavController
import com.example.controldecolonias.ColoniaViewModel
import com.example.controldecolonias.Nido
import com.example.controldecolonias.ui.components.DetalleItem
import com.example.controldecolonias.ui.components.NidoItem
import com.example.controldecolonias.ui.navigation.Rutas
import com.example.controldecolonias.util.construirNidoDesdeFormulario
import com.example.controldecolonias.util.formatearTimestamp
import com.example.controldecolonias.util.generarIdCortoUnico
import com.example.controldecolonias.util.obtenerUbicacion
import kotlinx.coroutines.launch

@Composable
fun PantallaPrincipalNidos(navController: NavController, viewModel: ColoniaViewModel) {
    PantallaListaDeNidos(
        viewModel = viewModel,
        onNavigateToAnadir = { navController.navigate(Rutas.ANADIR_EDITAR_NIDO) },
        onNidoClick = { id -> navController.navigate("detalleNido/$id") },
        onNavigateToMapa = { navController.navigate(Rutas.MAPA_NIDOS) }
    )
}

@OptIn(ExperimentalMaterial3Api::class, ExperimentalFoundationApi::class)
@Composable
fun PantallaListaDeNidos(
    viewModel: ColoniaViewModel, onNavigateToAnadir: () -> Unit, onNidoClick: (String) -> Unit, onNavigateToMapa: () -> Unit
) {
    val nidosAgrupados by viewModel.nidosFiltradosYAgrupados.collectAsState()
    val textoBusqueda by viewModel.textoBusqueda.collectAsState()

    Scaffold(
        topBar = {
            TopAppBar(
                title = { Text("Nidos en Campo") },
                actions = {
                    IconButton(onClick = onNavigateToMapa) {
                        Icon(Icons.Default.Map, contentDescription = "Ver Mapa")
                    }
                }
            )
        },
        floatingActionButton = {
            FloatingActionButton(onClick = onNavigateToAnadir) {
                Icon(Icons.Default.Add, contentDescription = "Añadir Nido")
            }
        }
    ) { innerPadding ->
        Column(modifier = Modifier.padding(innerPadding)) {
            OutlinedTextField(
                value = textoBusqueda,
                onValueChange = viewModel::onTextoBusquedaChange,
                modifier = Modifier.fillMaxWidth().padding(horizontal = 16.dp, vertical = 8.dp),
                label = { Text("Buscar por Nombre o Especie...") },
                singleLine = true,
                leadingIcon = { Icon(Icons.Default.Search, contentDescription = null) }
            )

            if (nidosAgrupados.isEmpty()) {
                Box(modifier = Modifier.fillMaxSize().padding(8.dp), contentAlignment = Alignment.Center) {
                    Text(if (textoBusqueda.isNotBlank()) "No se encontraron nidos." else "No hay nidos registrados.\nPulsa '+' para añadir el primero.")
                }
            } else {
                LazyColumn(modifier = Modifier.padding(horizontal = 8.dp)) {
                    nidosAgrupados.forEach { (especie, nidosDeLaEspecie) ->
                        stickyHeader {
                            Surface(modifier = Modifier.fillMaxWidth(), color = MaterialTheme.colorScheme.surfaceVariant) {
                                Text(
                                    text = "${if (especie.isBlank()) "Sin Especie" else especie} (${nidosDeLaEspecie.size})",
                                    style = MaterialTheme.typography.titleMedium,
                                    modifier = Modifier.padding(horizontal = 8.dp, vertical = 4.dp)
                                )
                            }
                        }
                        items(nidosDeLaEspecie) { nido ->
                            NidoItem(nido = nido, onClick = { onNidoClick(nido.id) })
                        }
                    }
                }
            }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PantallaAnadirEditarNido(
    nidoId: String?, viewModel: ColoniaViewModel, onNavigateBack: () -> Unit, onGuardarNido: (Nido) -> Unit
) {
    val nidoExistente by if (nidoId != null) viewModel.obtenerNido(nidoId).collectAsState(initial = null) else remember { mutableStateOf(null) }

    var nombre by remember { mutableStateOf("") }
    var especie by remember { mutableStateOf("") }
    var notas by remember { mutableStateOf("") }
    var latitud by remember { mutableStateOf<Double?>(null) }
    var longitud by remember { mutableStateOf<Double?>(null) }
    var textoBotonGps by remember { mutableStateOf("Obtener Ubicación Actual") }
    val context = LocalContext.current
    val coroutineScope = rememberCoroutineScope()

    LaunchedEffect(nidoExistente) {
        if (nidoId != null) {
            nidoExistente?.let {
                nombre = it.nombre; especie = it.especie; notas = it.notas
                latitud = it.latitud; longitud = it.longitud
                if (it.latitud != 0.0) textoBotonGps = "Ubicación Obtenida"
            }
        }
    }

    val locationPermissionLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.RequestMultiplePermissions(),
        onResult = { permissions ->
            if (permissions.getOrDefault(Manifest.permission.ACCESS_FINE_LOCATION, false)) {
                textoBotonGps = "Obteniendo..."; obtenerUbicacion(context) { location ->
                    latitud = location?.latitude; longitud = location?.longitude; textoBotonGps = "Ubicación Obtenida"
                }
            } else { textoBotonGps = "Permiso denegado" }
        }
    )

    Scaffold(topBar = { TopAppBar(title = { Text(if (nidoId == null) "Añadir Nido" else "Editar Nido") }) }) { innerPadding ->
        Column(modifier = Modifier.padding(innerPadding).padding(16.dp).fillMaxSize().verticalScroll(rememberScrollState())) {
            OutlinedTextField(value = nombre, onValueChange = { nombre = it }, label = { Text("Nombre/Referencia del nido") }, singleLine = true, modifier = Modifier.fillMaxWidth())
            Spacer(modifier = Modifier.height(8.dp))
            OutlinedTextField(value = especie, onValueChange = { especie = it }, label = { Text("Especie (opcional)") }, singleLine = true, modifier = Modifier.fillMaxWidth())
            Spacer(modifier = Modifier.height(8.dp))
            OutlinedTextField(value = notas, onValueChange = { notas = it }, label = { Text("Notas") }, modifier = Modifier.fillMaxWidth().height(150.dp))
            Spacer(modifier = Modifier.height(16.dp))
            Button(onClick = {
                if (ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED) {
                    textoBotonGps = "Obteniendo..."; obtenerUbicacion(context) { location ->
                        latitud = location?.latitude; longitud = location?.longitude; textoBotonGps = "Ubicación Obtenida"
                    }
                } else { locationPermissionLauncher.launch(arrayOf(Manifest.permission.ACCESS_FINE_LOCATION)) }
            }, enabled = latitud == null) { Text(textoBotonGps) }
            if(latitud != null && longitud != null) { Text("Lat: $latitud, Lon: $longitud") }

            Spacer(modifier = Modifier.weight(1f))
            Row(modifier = Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.End) {
                TextButton(onClick = onNavigateBack) { Text("Cancelar") }
                Button(onClick = {
                    coroutineScope.launch {
                        val idFinal = nidoId ?: generarIdCortoUnico(viewModel)
                        val nido = construirNidoDesdeFormulario(
                            id = idFinal, nombre = nombre, especie = especie,
                            notas = notas, latitud = latitud, longitud = longitud,
                            fechaOriginal = nidoExistente?.fecha
                        )
                        if (nido != null) {
                            onGuardarNido(nido)
                        } else {
                            Toast.makeText(context, "El nombre y la ubicación GPS son obligatorios", Toast.LENGTH_SHORT).show()
                        }
                    }
                }) { Text("Guardar") }
            }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PantallaDetalleNido(
    nidoId: String, viewModel: ColoniaViewModel, onNavigateBack: () -> Unit, onNavigateToEditar: (String) -> Unit
) {
    val nido by viewModel.obtenerNido(nidoId).collectAsState(initial = null)
    val context = LocalContext.current
    var mostrarDialogoBorrado by remember { mutableStateOf(false) }

    Scaffold(topBar = { TopAppBar(title = { Text(nido?.nombre ?: "Detalle de Nido") }) }) { innerPadding ->
        if (nido == null) {
            Box(modifier = Modifier.fillMaxSize().padding(innerPadding), contentAlignment = Alignment.Center) { CircularProgressIndicator() }
        } else {
            Column(modifier = Modifier.padding(innerPadding).padding(16.dp).fillMaxSize().verticalScroll(rememberScrollState())) {
                DetalleItem(etiqueta = "Nombre/Referencia", valor = nido!!.nombre)
                if(nido!!.especie.isNotBlank()) DetalleItem(etiqueta = "Especie", valor = nido!!.especie)
                DetalleItem(etiqueta = "Fecha de registro", valor = formatearTimestamp(nido!!.fecha, "dd/MM/yyyy HH:mm"))
                DetalleItem(etiqueta = "Coordenadas", valor = "${nido!!.latitud}, ${nido!!.longitud}")
                Spacer(modifier = Modifier.height(8.dp))
                Text("Notas:", style = MaterialTheme.typography.labelSmall, color = MaterialTheme.colorScheme.primary)
                Text(nido!!.notas, style = MaterialTheme.typography.bodyLarge)
                Spacer(modifier = Modifier.height(24.dp))
                Button(onClick = {
                    val gmmIntentUri = Uri.parse("geo:${nido!!.latitud},${nido!!.longitud}?q=${nido!!.latitud},${nido!!.longitud}(${Uri.encode(nido!!.nombre)})")
                    val mapIntent = Intent(Intent.ACTION_VIEW, gmmIntentUri)
                    mapIntent.setPackage("com.google.android.apps.maps")
                    try {
                        context.startActivity(mapIntent)
                    } catch (e: Exception) {
                        Toast.makeText(context, "No se pudo abrir Google Maps", Toast.LENGTH_SHORT).show()
                    }
                }, modifier = Modifier.fillMaxWidth()) {
                    Icon(Icons.Default.Map, contentDescription = "Abrir en Mapa")
                    Spacer(Modifier.size(ButtonDefaults.IconSpacing))
                    Text("Llevarme al Nido")
                }
                Spacer(modifier = Modifier.weight(1f))
                Row(modifier = Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.SpaceAround) {
                    Button(onClick = onNavigateBack, colors = ButtonDefaults.buttonColors(containerColor = MaterialTheme.colorScheme.secondary)) { Text("Volver") }
                    Button(onClick = { onNavigateToEditar(nido!!.id) }) { Text("Editar") }
                    Button(onClick = { mostrarDialogoBorrado = true }, colors = ButtonDefaults.buttonColors(containerColor = MaterialTheme.colorScheme.error)) { Text("Eliminar") }
                }
            }
        }
    }
    if (mostrarDialogoBorrado) {
        AlertDialog(
            onDismissRequest = { mostrarDialogoBorrado = false },
            title = { Text("Confirmar Borrado") },
            text = { Text("¿Estás seguro de que quieres eliminar el nido '${nido?.nombre}'? Esta acción no se puede deshacer.") },
            confirmButton = { Button(onClick = { nido?.let { viewModel.borrarNido(it) }; onNavigateBack() }, colors = ButtonDefaults.buttonColors(containerColor = MaterialTheme.colorScheme.error)) { Text("Eliminar") } },
            dismissButton = { TextButton(onClick = { mostrarDialogoBorrado = false }) { Text("Cancelar") } }
        )
    }
}