package com.example.controldecolonias.ui.screens.herramientas

import android.net.Uri
import android.widget.Toast
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.selection.selectable
import androidx.compose.foundation.selection.selectableGroup
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.semantics.Role
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.unit.dp
import com.example.controldecolonias.ColoniaViewModel
import com.example.controldecolonias.ScanMode
import com.example.controldecolonias.util.generarPdfDeQrs
import java.io.OutputStreamWriter
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale

// Estado para manejar la visibilidad y los datos del diálogo de QR masivo
private sealed class QrDialogState {
    object Hidden : QrDialogState()
    data class Visible(val cantidad: Int = 0) : QrDialogState()
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PantallaHerramientas(
    viewModel: ColoniaViewModel,
    onNavigateBack: () -> Unit
) {
    val context = LocalContext.current
    val scanMode by viewModel.scanMode.collectAsState()
    // --- CORRECCIÓN AQUÍ ---
    // Faltaba obtener el estado del Modo Ninja desde el ViewModel
    val ninjaModeActivado by viewModel.ninjaModeActivado.collectAsState()

    var mostrarDialogoImportar by remember { mutableStateOf(false) }
    val apiKeyGuardada by viewModel.apiKey.collectAsState()
    var apiKeyTexto by remember(apiKeyGuardada) { mutableStateOf(apiKeyGuardada) }
    var qrDialogState by remember { mutableStateOf<QrDialogState>(QrDialogState.Hidden) }

    val exportLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.CreateDocument("application/json"),
        onResult = { uri: Uri? ->
            uri?.let {
                viewModel.exportarDatos { jsonString ->
                    try {
                        context.contentResolver.openOutputStream(it)?.use { outputStream ->
                            OutputStreamWriter(outputStream).use { writer ->
                                writer.write(jsonString)
                            }
                        }
                        Toast.makeText(context, "Exportación exitosa", Toast.LENGTH_SHORT).show()
                    } catch (e: Exception) {
                        Toast.makeText(context, "Error al exportar: ${e.message}", Toast.LENGTH_LONG).show()
                    }
                }
            }
        }
    )

    val exportNidosLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.CreateDocument("application/json"),
        onResult = { uri: Uri? ->
            uri?.let {
                viewModel.exportarNidos { jsonString ->
                    try {
                        context.contentResolver.openOutputStream(it)?.use { outputStream ->
                            OutputStreamWriter(outputStream).use { writer ->
                                writer.write(jsonString)
                            }
                        }
                        Toast.makeText(context, "Exportación exitosa", Toast.LENGTH_SHORT).show()
                    } catch (e: Exception) {
                        Toast.makeText(context, "Error al exportar: ${e.message}", Toast.LENGTH_LONG).show()
                    }
                }
            }
        }
    )

    val importLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.OpenDocument(),
        onResult = { uri: Uri? ->
            uri?.let {
                viewModel.importarDatosSelectivo(it, context) { _, mensaje ->
                    Toast.makeText(context, mensaje, Toast.LENGTH_LONG).show()
                }
            }
        }
    )

    val importCompletoLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.OpenDocument(),
        onResult = { uri: Uri? ->
            uri?.let {
                viewModel.importarDatos(it, context) { _, mensaje ->
                    Toast.makeText(context, mensaje, Toast.LENGTH_LONG).show()
                }
            }
        }
    )

    val guardarPdfQrLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.CreateDocument("application/pdf"),
        onResult = { uri: Uri? ->
            uri?.let { safeUri ->
                val currentState = qrDialogState
                if (currentState is QrDialogState.Visible) {
                    generarPdfDeQrs(context, currentState.cantidad, safeUri)
                }
            }
            qrDialogState = QrDialogState.Hidden
        }
    )

    Scaffold(topBar = { TopAppBar(title = { Text("Herramientas y Ajustes") }) }) { innerPadding ->
        Column(
            modifier = Modifier
                .padding(innerPadding)
                .padding(16.dp)
                .fillMaxWidth()
                .verticalScroll(rememberScrollState()),
            verticalArrangement = Arrangement.spacedBy(16.dp)
        ) {
            Text("Privacidad", style = MaterialTheme.typography.titleMedium)
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(vertical = 8.dp),
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = Arrangement.SpaceBetween
            ) {
                Column(modifier = Modifier.weight(1f)) {
                    Text("Modo Ninja", style = MaterialTheme.typography.bodyLarge)
                    Text("No guardar ni mostrar ubicación de colonias.", style = MaterialTheme.typography.bodySmall)
                }
                Switch(
                    checked = ninjaModeActivado,
                    onCheckedChange = { activado -> viewModel.guardarNinjaMode(activado) }
                )
            }

            Divider()

            Text("Ajustes de API", style = MaterialTheme.typography.titleMedium)
            OutlinedTextField(
                value = apiKeyTexto,
                onValueChange = { apiKeyTexto = it },
                label = { Text("API Key de OpenWeatherMap") },
                modifier = Modifier.fillMaxWidth(),
                singleLine = true
            )
            Button(
                onClick = {
                    viewModel.guardarApiKey(apiKeyTexto)
                    Toast.makeText(context, "API Key guardada", Toast.LENGTH_SHORT).show()
                },
                modifier = Modifier.align(Alignment.End)
            ) { Text("Guardar Key") }

            Text("Método de Escaneo Predeterminado", style = MaterialTheme.typography.titleMedium)
            Column(Modifier.selectableGroup()) {
                Row(
                    Modifier
                        .fillMaxWidth()
                        .height(56.dp)
                        .selectable(
                            selected = (scanMode == ScanMode.NFC),
                            onClick = { viewModel.setScanMode(ScanMode.NFC) },
                            role = Role.RadioButton
                        )
                        .padding(horizontal = 16.dp),
                    verticalAlignment = Alignment.CenterVertically
                ) {
                    RadioButton(selected = (scanMode == ScanMode.NFC), onClick = null)
                    Text(text = "NFC (Sensor)", style = MaterialTheme.typography.bodyLarge, modifier = Modifier.padding(start = 16.dp))
                }
                Row(
                    Modifier
                        .fillMaxWidth()
                        .height(56.dp)
                        .selectable(
                            selected = (scanMode == ScanMode.QR),
                            onClick = { viewModel.setScanMode(ScanMode.QR) },
                            role = Role.RadioButton
                        )
                        .padding(horizontal = 16.dp),
                    verticalAlignment = Alignment.CenterVertically
                ) {
                    RadioButton(selected = (scanMode == ScanMode.QR), onClick = null)
                    Text(text = "Cámara (Código QR)", style = MaterialTheme.typography.bodyLarge, modifier = Modifier.padding(start = 16.dp))
                }
            }

            Text("Copias de Seguridad", style = MaterialTheme.typography.titleMedium)
            Button(onClick = {
                val sdf = SimpleDateFormat("yyyyMMdd_HHmmss", Locale.getDefault())
                val timestamp = sdf.format(Date())
                exportLauncher.launch("ControlDeColonias_Backup_$timestamp.json")
            }, modifier = Modifier.fillMaxWidth()) {
                Text("Exportar Copia de seguridad completa")
            }

            OutlinedButton(onClick = {
                val sdf = SimpleDateFormat("yyyyMMdd_HHmmss")
                val timestamp = sdf.format(Date())
                exportNidosLauncher.launch("NidosDeCampo_$timestamp.json")
            }, modifier = Modifier.fillMaxWidth()) {
                Text("Exportar nidos de campo")
            }

            Button(onClick = { mostrarDialogoImportar = true }, modifier = Modifier.fillMaxWidth()) {
                Text("Importar copia de seguridad completa")
            }

            OutlinedButton(onClick = {
                importLauncher.launch(arrayOf("application/json"))
            }, modifier = Modifier.fillMaxWidth()) {
                Text("Importar/Fusionar archivo")
            }

            Text("Herramientas de Códigos QR", style = MaterialTheme.typography.titleMedium)
            Button(onClick = {
                qrDialogState = QrDialogState.Visible()
            }, modifier = Modifier.fillMaxWidth()) {
                Text("Generar QR para Pre-impresión")
            }

            Text("Herramientas NFC", style = MaterialTheme.typography.titleMedium)
            Button(onClick = {
                viewModel.activarModoBorrado()
                Toast.makeText(context, "Acerca una etiqueta para borrar", Toast.LENGTH_SHORT).show()
            }, modifier = Modifier.fillMaxWidth()) {
                Text("Borrar Etiqueta NFC")
            }

            Spacer(modifier = Modifier.height(16.dp))

            Button(onClick = onNavigateBack, modifier = Modifier.fillMaxWidth()) { Text("Volver") }
        }
    }

    if (mostrarDialogoImportar) {
        AlertDialog(
            onDismissRequest = { mostrarDialogoImportar = false },
            title = { Text("Confirmar Importación") },
            text = { Text("¡ADVERTENCIA! Esto borrará todos los datos actuales y los reemplazará con los de la copia de seguridad. ¿Estás seguro?") },
            confirmButton = {
                Button(onClick = {
                    importCompletoLauncher.launch(arrayOf("application/json"))
                    mostrarDialogoImportar = false
                }, colors = ButtonDefaults.buttonColors(containerColor = MaterialTheme.colorScheme.error)) {
                    Text("Continuar")
                }
            },
            dismissButton = {
                TextButton(onClick = { mostrarDialogoImportar = false }) { Text("Cancelar") }
            }
        )
    }

    if (qrDialogState is QrDialogState.Visible) {
        DialogoGeneradorQrMasivo(
            onDismiss = { qrDialogState = QrDialogState.Hidden },
            onGenerate = { cantidad ->
                qrDialogState = QrDialogState.Visible(cantidad)
                val sdf = SimpleDateFormat("yyyyMMdd", Locale.getDefault())
                val timestamp = sdf.format(Date())
                guardarPdfQrLauncher.launch("QRs_Preimpresion_$timestamp.pdf")
            }
        )
    }
}

@Composable
fun DialogoGeneradorQrMasivo(
    onDismiss: () -> Unit,
    onGenerate: (Int) -> Unit
) {
    var cantidad by remember { mutableStateOf("") }
    val context = LocalContext.current

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text("Generar QR Masivos") },
        text = {
            OutlinedTextField(
                value = cantidad,
                onValueChange = { if (it.length <= 3) cantidad = it.filter { char -> char.isDigit() } },
                label = { Text("Cantidad de códigos a generar") },
                keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                singleLine = true
            )
        },
        confirmButton = {
            Button(
                onClick = {
                    val num = cantidad.toIntOrNull()
                    if (num != null && num > 0) {
                        onGenerate(num)
                    } else {
                        Toast.makeText(context, "Introduce un número válido", Toast.LENGTH_SHORT).show()
                    }
                }
            ) {
                Text("Guardar PDF")
            }
        },
        dismissButton = {
            TextButton(onClick = onDismiss) { Text("Cancelar") }
        }
    )
}